/*******************************************************************************
 * Copyright (c) 2009, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.persistence.tools.db.driver;

import java.util.ArrayList;
import org.eclipse.persistence.tools.gen.db.Database;
import org.eclipse.persistence.tools.utility.StringTools;

/**
 * Microsoft SQL Server
 *
 * @version 2.5
 */
@SuppressWarnings("nls")
class SQLServer extends AbstractWorkbenchDriverAdapter {

	SQLServer(Database database) {
		super(database);
	}

	@Override
	CatalogStrategy buildCatalogStrategy() {
		return new SimpleCatalogStrategy();
	}

	@Override
	FoldingStrategy buildFoldingStrategy() {
		return NonFoldingStrategy.instance();
	}

	/**
	 * By default, SQL Server identifiers are case-sensitive, even without
	 * delimiters. This can depend on the collation setting....
	 */
	//TODO query database for collation setting
	@Override
	boolean regularNamesMatch(String name1, String name2) {
		return name1.equals(name2);
	}

	/**
	 * SQL Server will use the user-requested database; if that database is not
	 * found, it will default to <code>master</code>.
	 */
	@Override
	void addDefaultCatalogNamesTo(ArrayList<String> names) {
		names.add(this.database.getName());
		names.add(MASTER_CATALOG_IDENTIFIER);
	}
	private static final String MASTER_CATALOG_IDENTIFIER = "master";

	/**
	 * The default schema on SQL Server for any database (catalog) is
	 * <code>dbo</code>.
	 */
	@Override
	void addDefaultSchemaNamesTo(ArrayList<String> names) {
		names.add(DEFAULT_SCHEMA_NAME);
	}
	private static final String DEFAULT_SCHEMA_NAME = "dbo";

	@Override
	char[] getExtendedRegularNameStartCharacters() {
		return EXTENDED_REGULAR_NAME_START_CHARACTERS;
	}
	private static final char[] EXTENDED_REGULAR_NAME_START_CHARACTERS = new char[] { '_', '@', '#' };

	@Override
	char[] getExtendedRegularNamePartCharacters() {
		return EXTENDED_REGULAR_NAME_PART_CHARACTERS;
	}
	private static final char[] EXTENDED_REGULAR_NAME_PART_CHARACTERS = new char[] { '$' };

	//TODO query database for delimiter setting
	@Override
	String delimitName(String name) {
		return '[' + name + ']';
	}

	/**
	 * By default, SQL Server delimits identifiers with brackets
	 * (<code>[]</code>); but it
	 * can also be configured to use double-quotes.
	 */
	//TODO query database for delimiter setting
	@Override
	boolean identifierIsDelimited(String identifier) {
		return StringTools.isBracketed(identifier)
					|| super.identifierIsDelimited(identifier);
	}


	// ********** factory **********

	static class Factory implements WorkbenchDriverAdapterFactory {
		private static final String[] VENDORS = {
			"SQL Server"
		};
		@Override
		public String[] getSupportedVendors() {
			return VENDORS;
		}
		@Override
		public WorkbenchDriverAdapter buildAdapter(Database database) {
			return new SQLServer(database);
		}
	}
}