/*******************************************************************************
 * Copyright (c) 2008, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen.db;

/**
 * Database table
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.
 *
 * @version 2.5
 */
public interface Table extends DatabaseObject {

	/**
	 * Returns the table's schema.
	 */
	Schema getSchema();


	// ********** columns **********

	/**
	 * Returns the table's columns.
	 */
	Iterable<Column> getColumns();

	/**
	 * Returns the number of columns the table contains.
	 */
	int getColumnsSize();

	/**
	 * Returns the column with the specified name. The name must be an exact match
	 * of the column's name.
	 * @see #getColumnForIdentifier(String)
	 */
	Column getColumnNamed(String name);

	/**
	 * Returns the table's column identifers, sorted by name.
	 * @see #getColumnForIdentifier(String)
	 */
	Iterable<String> getSortedColumnIdentifiers();

	/**
	 * Returns the column for the specified identifier. The identifier should
	 * be an SQL identifier (i.e. quoted when case-sensitive or containing
	 * special characters, unquoted otherwise).
	 * @see #getColumnNamed(String)
	 * @see #getSortedColumnIdentifiers()
	 */
	Column getColumnForIdentifier(String identifier);


	// ********** primary key columns **********

	/**
	 * Returns the table's primary key columns.
	 */
	Iterable<Column> getPrimaryKeyColumns();

	/**
	 * Returns the number of primary key columns the table contains.
	 */
	int getPrimaryKeyColumnsSize();

	/**
	 * Returns the table's single primary key column. Throw an
	 * {@link IllegalStateException} if the table has more than one primary key column.
	 */
	Column getPrimaryKeyColumn();


	// ********** foreign keys **********

	/**
	 * Returns the table's foreign keys.
	 */
	Iterable<ForeignKey> getForeignKeys();

	/**
	 * Returns the number of foreign keys the table contains.
	 */
	int getForeignKeysSize();


	// ********** join table support **********

	/**
	 * Returns whether the table is possibly a <em>join</em> table
	 * (i.e. it contains only 2 foreign keys). Whether the table <em>actually</em> is
	 * a <em>join</em> table is determined by the semantics of the database design.
	 */
	boolean isPossibleJoinTable();

	/**
	 * Returns the foreign key to the
	 * <em>owning</em> table.
	 * @see #isPossibleJoinTable()
	 */
	ForeignKey getJoinTableOwningForeignKey();

	/**
	 * Returns the foreign key to the
	 * <em>non-owning</em> table.
	 * @see #isPossibleJoinTable()
	 */
	ForeignKey getJoinTableNonOwningForeignKey();

	/**
	 * Returns whether its name matches
	 * the JPA default (i.e. <code>"OWNINGTABLE_NONOWNINGTABLE"</code>).
	 * @see #isPossibleJoinTable()
	 */
	boolean joinTableNameIsDefault();
}