/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.EnumType;
import javax.persistence.FetchType;
import javax.persistence.TemporalType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityIDGeneratedValue;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.w3c.dom.Element;

/**
 * The external form of a basic mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class BasicMapping extends ConvertibleMapping
                         implements ExternalBasicMapping {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String BASIC = "basic";

	/**
	 * The element name used to store and retrieve the expiry child text node.
	 */
	static final String ENUMERATED = "enumerated";

	/**
	 * The attribute name used to store and retrieve the fetch property.
	 */
	static final String FETCH = "fetch";

	/**
	 * The element name used to store and retrieve the lob child text node.
	 */
	static final String LOB = "lob";

	/**
	 * The attribute name used to store and retrieve the mutable property.
	 */
	static final String MUTABLE = "mutable";

	/**
	 * The attribute name used to store and retrieve the optional property.
	 */
	static final String OPTIONAL = "optional";

	/**
	 * Returns insert property.
	 */
	static final String RETURN_INSERT = "return-insert";

	/**
	 * Returns only on insert property.
	 */
	static final String RETURN_ONLY = "return-only";

	/**
	 * Returns update property.
	 */
	static final String RETURN_UPDATE = "return-update";

	/**
	 * The element name used to store and retrieve the temporal child text node.
	 */
	static final String TEMPORAL = "temporal";

	/**
	 * Creates a new <code>BasicMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the external form of the mapping in the list of children
	 */
	BasicMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityIDGeneratedValue addGeneratedValue() {
		EntityIDGeneratedValue generatedValue = buildGeneratedValue();
		generatedValue.addSelf();
		return generatedValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator addSequenceGenerator() {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator();
		sequenceGenerator.addSelf();
		return sequenceGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator addTableGenerator() {
		TableGenerator tableGenerator = buildTableGenerator();
		tableGenerator.addSelf();
		return tableGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ACCESS);
		names.add(MUTABLE);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private EntityColumn buildColumn() {
		return new EntityColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		names.add(NoSqlField.FIELD);
		names.add(EntityIDGeneratedValue.GENERATED_VALUE);
		names.add(LOB);
		names.add(TEMPORAL);
		names.add(ENUMERATED);
		names.add(CONVERT);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		names.add(RETURN_INSERT);
		names.add(RETURN_UPDATE);
		return names;
	}

	private EntityIDGeneratedValue buildGeneratedValue() {
		return new EntityIDGeneratedValue(this);
	}

	private ExternalNoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	private SequenceGenerator buildSequenceGenerator() {
		return new SequenceGenerator(this);
	}

	private TableGenerator buildTableGenerator() {
		return new TableGenerator(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn() {
		return buildColumn();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return BASIC;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getEnumeratedType() {
		return getChildEnumNode(ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityIDGeneratedValue getGeneratedValue() {

		if (hasChild(EntityIDGeneratedValue.GENERATED_VALUE)) {
			return buildGeneratedValue();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator getSequenceGenerator() {
		Element element = getChild(SequenceGenerator.SEQUENCE_GENERATOR);

		if (element == null) {
			return null;
		}

		return buildSequenceGenerator();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator getTableGenerator() {
		Element element = getChild(TableGenerator.TABLE_GENERATOR);

		if (element == null) {
			return null;
		}

		return buildTableGenerator();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getTemporalType() {
		return getChildEnumNode(TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final boolean isBasicMapping() {
		return true;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isLob() {
		return getChildBooleanNode(LOB);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isMutable() {
		return getBooleanAttribute(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOptional() {
		return getBooleanAttribute(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeGeneratedValue() {
		EntityIDGeneratedValue generatedValue = buildGeneratedValue();
		generatedValue.removeSelf();
	}

	@Override
	public void removeSequenceGenerator() {
		SequenceGenerator generator = buildSequenceGenerator();
		generator.removeSelf();
	}

	@Override
	public void removeTableGenerator() {
		TableGenerator generator = buildTableGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean returnOnInsert() {
		return hasChild(RETURN_INSERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean returnOnlyOnInsert() {

		Element returnInsertElement = getChild(RETURN_INSERT);

		if (returnInsertElement != null) {
			return getBooleanAttribute(returnInsertElement, RETURN_ONLY);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean returnOnUpdate() {
		return hasChild(RETURN_UPDATE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumn(String columnName) {

		removeChild(EntityColumn.COLUMN);

		if (columnName != null) {
			EntityColumn entityColumn = buildColumn();
			entityColumn.addSelf();
			entityColumn.setName(columnName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEnumeratedType(EnumType type) {
		updateChildTextNode(ENUMERATED, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLob(Boolean lob) {
		updateChildTextNode(LOB, lob);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMutable(Boolean mutable) {
		setAttribute(MUTABLE, mutable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		ExternalNoSqlField noSqlField = buildNoSqlField();
		noSqlField.setName(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptional(Boolean optional) {
		setAttribute(OPTIONAL, optional);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReturnOnInsert(Boolean returnOnInsert) {
		if (returnOnInsert == Boolean.TRUE) {
			addChild(RETURN_INSERT);
		}
		else {
			removeChild(RETURN_INSERT);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReturnOnlyOnInsert(Boolean returnOnlyOnInsert) {
		if (returnOnlyOnInsert != null) {
			Element returnInsertElement = addChild(RETURN_INSERT);
			setAttribute(returnInsertElement, RETURN_ONLY, returnOnlyOnInsert);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReturnOnUpdate(Boolean returnOnUpdate) {
		if (returnOnUpdate == Boolean.TRUE) {
			addChild(RETURN_UPDATE);
		}
		else {
			removeChild(RETURN_UPDATE);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSequenceGenerator(String generatorName) {
		updateChildTextNode(SequenceGenerator.SEQUENCE_GENERATOR, generatorName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTableGenerator(String generatorName) {
		updateChildTextNode(TableGenerator.TABLE_GENERATOR, generatorName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTemporalType(TemporalType type) {
		updateChildTextNode(TEMPORAL, type);
	}
}