/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalClassConverter;
import org.w3c.dom.Element;

/**
 * The abstract definition of an external class converter.
 *
 * @version 2.5
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
abstract class ClassConverter extends AbstractExternalForm
                              implements ExternalClassConverter {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * Creates a new <code>ClassConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ClassConverter(BasicMapMapping parent) {
		super(parent);
		this.index = -1;
	}

	/**
	 * Creates a new <code>ClassConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ClassConverter(ConvertibleMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>ClassConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ClassConverter(ElementCollectionMapping parent) {
		super(parent);
		this.index = -1;
	}

	/**
	 * Creates a new <code>ClassConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ClassConverter(EmbeddableEntity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>ClassConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ClassConverter(ORMConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected final Element addSelf(String elementName, List<String> elementNamesOrder) {

		if (index == -1) {
			return super.addSelf(elementName, elementNamesOrder);
		}

		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		return names;
	}
	
	/**
	 * Returns the element name used to retrieve or store the converter class name.
	 *
	 * @return The element name of the converter class name
	 */
	abstract String getConverterClassElementName();

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getConverterClassName() {
		return getAttribute(getConverterClassElementName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {

		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setConverterClassName(String classname) {
		setAttribute(getConverterClassElementName(), classname);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {
		setAttribute(NAME, name);
	}
}