/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.eclipse.persistence.annotations.ExistenceType;
import org.eclipse.persistence.tools.mapping.orm.ExternalCache;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddedIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.eclipse.persistence.tools.mapping.orm.ExternalFetchGroup;
import org.eclipse.persistence.tools.mapping.orm.ExternalIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappedSuperClassEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalMultitenancyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalOptimisticLocking;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKey;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTransformationMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalVariableOneToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalVersionMapping;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.iterable.EmptyListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of an mapped superclass entity, which is a child of an ORM configuration.
 *
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
class MappedSuperClassEntity extends EmbeddableEntity
                             implements ExternalMappedSuperClassEntity {

	/**
	 * The element name used to store and retrieve the cache-intercepter child node.
	 */
	static final String CACHE_INTERCEPTOR = "cache-interceptor";

	/**
	 * The attribute name used to store and retrieve the cacheable property
	 */
	static final String CACHEABLE = "cacheable";

	/**
	 * The attribute name used to store and retrieve the class property of the id-class child node.
	 */
	static final String CLASS = "class";

	/**
	 * The element name used to store and retrieve the exclude-default-listeners child node.
	 */
	static final String EXCLUDE_DEFAULT_LISTENERS = "exclude-default-listeners";

	/**
	 * The element name used to store and retrieve the exclude-superclass-listeners child node.
	 */
	static final String EXCLUDE_SUPERCLASS_LISTENERS = "exclude-superclass-listeners";

	/**
	 * The attribute name used to store and retrieve the existence-checking property.
	 */
	static final String EXISTENCE_CHECKING = "existence-checking";

	/**
	 * The element name used to store and retrieve the id-class child node.
	 */
	static final String ID_CLASS = "id-class";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String MAPPED_SUPERCLASS = "mapped-superclass";

	/**
	 * The attribute name used to store and retrieve the method-name property of the callback child nodes.
	 */
	static final String METHOD_NAME = "method-name";

	/**
	 * The element name used to store and retrieve the multitenant child node.
	 */
	static final String MULTITENANT = "multitenant";

	/**
	 * The attribute name used to store and retrieve the optimistic-locking property.
	 */
	static final String OPTIMISTIC_LOCKING = "optimistic-locking";

	/**
	 * The element name used to store and retrieve the parent-class child node.
	 */
	static final String PARENT_CLASS = "parent-class";

	/**
	 * The element name used to store and retrieve the post-load child node.
	 */
	static final String POST_LOAD = "post-load";

	/**
	 * The element name used to store and retrieve the post-persist child node.
	 */
	static final String POST_PERSIST = "post-persist";

	/**
	 * The element name used to store and retrieve the post-remove child node.
	 */
	static final String POST_REMOVE = "post-remove";

	/**
	 * The element name used to store and retrieve the post-update child node.
	 */
	static final String POST_UPDATE = "post-update";

	/**
	 * The element name used to store and retrieve the pre-persist child node.
	 */
	static final String PRE_PERSIST = "pre-persist";

	/**
	 * The element name used to store and retrieve the pre-remove child node.
	 */
	static final String PRE_REMOVE = "pre-remove";

	/**
	 * The element name used to store and retrieve the pre-update child node.
	 */
	static final String PRE_UPDATE = "pre-update";

	/**
	 * The attribute name used to store and retrieve the primary-key property.
	 */
	static final String PRIMARY_KEY = "primary-key";

	/**
	 * The attribute name used to store and retrieve the read-only property.
	 */
	static final String READ_ONLY =	"read-only";

	/**
	 * Creates a new <code>MappedSuperClassEntity</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with
	 * the same type owned by the parent
	 */
	MappedSuperClassEntity(ORMConfiguration parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalCache addCache() {
		Cache cache = buildCache();
		cache.addSelf();
		return cache;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalEmbeddedIDMapping addEmbeddedIdMapping(String name) {
		EmbeddedIDMapping mapping = buildEmbeddedIdMapping(mappingsSize());
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addEntityListener(int index, String name) {
		EntityListener entityListener = buildEntityListener(index);
		entityListener.addSelf();
		entityListener.setClassName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalFetchGroup addFetchGroup(int index, String name) {
		FetchGroup fetchGroup = buildFetchGroup(index);
		fetchGroup.addSelf();
		fetchGroup.setName(name);
		return fetchGroup;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalMultitenancyPolicy addMultitenancy() {
		MultitenancyPolicy multitenant = buildMultitenant();
		multitenant.addSelf();
		return multitenant;
	}

	/** {@inheritDoc}
	 */
	@Override
	public Boolean isMultitenant() {
		return getChild("multitenant") != null ? Boolean.TRUE : Boolean.FALSE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalIDMapping addIdMapping(String name) {
		IdMapping mapping = buildIdMapping(mappingsSize());
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalOptimisticLocking addOptimisticLocking() {
		OptimisticLocking optimisticLocking = buildOptimisticLocking();
		optimisticLocking.addSelf();
		return optimisticLocking;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalPrimaryKey addPrimaryKey() {
		PrimaryKey primaryKey = buildPrimaryKey();
		primaryKey.addSelf();
		return primaryKey;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator addSequenceGenerator() {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator();
		sequenceGenerator.addSelf();
		return sequenceGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator addTableGenerator() {
		TableGenerator tableGenerator = buildTableGenerator();
		tableGenerator.addSelf();
		return tableGenerator;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalTransformationMapping addTransformationMapping(String name) {
		TransformationMapping mapping = buildTransformationMapping(mappingsSize());
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalVariableOneToOneMapping addVariableOneToOneMapping(String name) {
		VariableOneToOneMapping mapping = buildVariableOneToOneMapping(mappingsSize());
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalVersionMapping addVersionMapping(String name) {
		VersionMapping mapping = buildVersionMapping(mappingsSize());
		mapping.addSelf();
		mapping.setName(name);
		return mapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(CLASS);
		names.add(PARENT_CLASS);
		names.add(ACCESS);
		names.add(CACHEABLE);
		names.add(METADATA_COMPLETE);
		names.add(READ_ONLY);
		names.add(EXISTENCE_CHECKING);
		names.add(EXCLUDE_DEFAULT_MAPPINGS);
		return names;
	}

	private Cache buildCache() {
		return new Cache(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(CUSTOMIZER);
		names.add(CHANGE_TRACKING);
		names.add(ID_CLASS);
		names.add(PrimaryKey.PRIMARY_KEY);
		names.add(OptimisticLocking.OPTIMISTIC_LOCKING);
		names.add(Cache.CACHE);
		names.add(CACHE_INTERCEPTOR);
		names.add(FetchGroup.FETCH_GROUP);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CopyPolicy.COPY_POLICY);
		names.add(InstantiationCopyPolicy.INSTANTIATION_COPY_POLICY);
		names.add(CloneCopyPolicy.CLONE_COPY_POLICY);
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add(EXCLUDE_DEFAULT_LISTENERS);
		names.add(EXCLUDE_SUPERCLASS_LISTENERS);
		names.add(EntityListener.ENTITY_LISTENERS);
		names.add(PRE_PERSIST);
		names.add(POST_PERSIST);
		names.add(PRE_REMOVE);
		names.add(POST_REMOVE);
		names.add(PRE_UPDATE);
		names.add(POST_UPDATE);
		names.add(POST_LOAD);
		names.add(Property.PROPERTY);
		names.add(Mapping.ATTRIBUTES);
		return names;
	}

	private EmbeddedIDMapping buildEmbeddedIdMapping(int index) {
		return new EmbeddedIDMapping(this, index);
	}

	private EntityListener buildEntityListener(int index) {
		return new EntityListener(this, index);
	}

	private FetchGroup buildFetchGroup(int index) {
		return new FetchGroup(this, index);
	}

	private IdMapping buildIdMapping(int index) {
		return new IdMapping(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	Mapping buildMapping(String elementName, int index) {

		// Embedded ID
		if (ObjectTools.equals(elementName, EmbeddedIDMapping.EMBEDDED_ID)) {
			return buildEmbeddedIdMapping(index);
		}

		// ID
		if (ObjectTools.equals(elementName, IdMapping.ID)) {
			return buildIdMapping(index);
		}

		// Transformation
		if (ObjectTools.equals(elementName, TransformationMapping.TRANSFORMATION)) {
			return buildTransformationMapping(index);
		}

		// Variable 1:1
		if (ObjectTools.equals(elementName, VariableOneToOneMapping.VARIABLE_ONE_TO_ONE)) {
			return buildVariableOneToOneMapping(index);
		}

		// Version
		if (ObjectTools.equals(elementName, VersionMapping.VERSION)) {
			return buildVersionMapping(index);
		}

		return super.buildMapping(elementName, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	Collection<String> buildMappingNames() {
		Collection<String> names = super.buildMappingNames();
		names.add(EmbeddedIDMapping.EMBEDDED_ID);
		names.add(IdMapping.ID);
		names.add(TransformationMapping.TRANSFORMATION);
		names.add(VariableOneToOneMapping.VARIABLE_ONE_TO_ONE);
		names.add(VersionMapping.VERSION);
		return names;
	}

	private MultitenancyPolicy buildMultitenant() {
		return new MultitenancyPolicy(this);
	}

	private OptimisticLocking buildOptimisticLocking() {
		return new OptimisticLocking(this);
	}

	private PrimaryKey buildPrimaryKey() {
		return new PrimaryKey(this);
	}

	private SequenceGenerator buildSequenceGenerator() {
		return new SequenceGenerator(this);
	}

	private TableGenerator buildTableGenerator() {
		return new TableGenerator(this);
	}

	private TransformationMapping buildTransformationMapping(int index) {
		return new TransformationMapping(this, index);
	}

	private VariableOneToOneMapping buildVariableOneToOneMapping(int index) {
		return new VariableOneToOneMapping(this, index);
	}

	private VersionMapping buildVersionMapping(int index) {
		return new VersionMapping(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ListIterable<ExternalEntityListener> entityListeners() {

		Element element = getChild(EntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return EmptyListIterable.instance();
		}

		int count = entityListenersSize();
		List<ExternalEntityListener> entityListeners = new ArrayList<ExternalEntityListener>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntityListener entityListener = buildEntityListener(index);
			entityListeners.add(0, entityListener);
		}

		return new ListListIterable<ExternalEntityListener>(entityListeners);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int entityListenersSize() {

		Element element = getChild(EntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return 0;
		}

		return getChildrenSize(element, EntityListener.ENTITY_LISTENER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ListIterable<ExternalFetchGroup> fetchGroups() {

		int count = fetchGroupsSize();
		List<ExternalFetchGroup> fetchGroups = new ArrayList<ExternalFetchGroup>(count);

		for (int index = count; --index >= 0;) {
			ExternalFetchGroup fetchGroup = buildFetchGroup(index);
			fetchGroups.add(0, fetchGroup);
		}

		return new ListListIterable<ExternalFetchGroup>(fetchGroups);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int fetchGroupsSize() {
		return getChildrenSize(FetchGroup.FETCH_GROUP);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalCache getCache() {

		if (hasChild(Cache.CACHE)) {
			return buildCache();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getCacheInterceptorClassName() {
		return getChildAttribute(CACHE_INTERCEPTOR, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return MAPPED_SUPERCLASS;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalEntityListener getEntityListener(int index) {

		Element element = getChild(EntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return null;
		}

		element = getChild(element, EntityListener.ENTITY_LISTENER, index);

		if (element == null) {
			return null;
		}

		return buildEntityListener(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExistenceType getExistenceCheckingType() {
		return getEnumAttribute(EXISTENCE_CHECKING, ExistenceType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalFetchGroup getFetchGroup(int index) {

		Element element = getChild(FetchGroup.FETCH_GROUP, index);

		if (element == null) {
			return null;
		}

		return buildFetchGroup(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getIdClassName() {
		return getChildAttribute(ID_CLASS, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getMappingsDescription() {
		return getChildTextNode(Mapping.ATTRIBUTES, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalMultitenancyPolicy getMultitenant() {
		return buildMultitenant();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalOptimisticLocking getOptimisticLocking() {

		if (hasChild(OptimisticLocking.OPTIMISTIC_LOCKING)) {
			return buildOptimisticLocking();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getParentClassName() {
		return getAttribute(PARENT_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostLoadMethod() {
		return getChildAttribute(POST_LOAD, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostLoadMethodDescription() {
		return getChildTextNode(POST_LOAD, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostPersistMethod() {
		return getChildAttribute(POST_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostPersistMethodDescription() {
		return getChildTextNode(POST_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostRemoveMethod() {
		return getChildAttribute(POST_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostRemoveMethodDescription() {
		return getChildTextNode(POST_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostUpdateMethod() {
		return getChildAttribute(POST_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPostUpdateMethodDescription() {
		return getChildTextNode(POST_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPrePersistMethod() {
		return getChildAttribute(PRE_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPrePersistMethodDescription() {
		return getChildTextNode(PRE_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreRemoveMethod() {
		return getChildAttribute(PRE_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreRemoveMethodDescription() {
		return getChildTextNode(PRE_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreUpdateMethod() {
		return getChildAttribute(PRE_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getPreUpdateMethodDescription() {
		return getChildTextNode(PRE_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalPrimaryKey getPrimaryKey() {

		if (hasChild(PrimaryKey.PRIMARY_KEY)) {
			return buildPrimaryKey();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public SequenceGenerator getSequenceGenerator() {

		Element element = getChild(SequenceGenerator.SEQUENCE_GENERATOR);

		if (element == null) {
			return null;
		}

		return buildSequenceGenerator();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TableGenerator getTableGenerator() {

		Element element = getChild(TableGenerator.TABLE_GENERATOR);

		if (element == null) {
			return null;
		}

		return buildTableGenerator();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isCacheable() {
		return getBooleanAttribute(CACHEABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean isReadOnly() {
		return getBooleanAttribute(READ_ONLY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeCache() {
		Cache cache = buildCache();
		cache.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeEntityListener(int index) {
		EntityListener entityListener = buildEntityListener(index);
		entityListener.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeFetchGroup(int index) {
		FetchGroup fetchGroup = buildFetchGroup(index);
		fetchGroup.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMultiTenancy() {
		MultitenancyPolicy multitenant = buildMultitenant();
		multitenant.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeOptimisticLocking() {
		OptimisticLocking optimisticLocking = buildOptimisticLocking();
		optimisticLocking.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removePrimaryKey() {
		PrimaryKey primaryKey = buildPrimaryKey();
		primaryKey.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCacheable(Boolean isCacheable) {
		setAttribute(CACHEABLE, isCacheable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCacheInterceptorClassName(String className) {
		updateChildAttribute(CACHE_INTERCEPTOR, CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setExcludesClassListeners(Boolean excludeClassListeners) {
		if (excludeClassListeners == Boolean.TRUE) {
			addChild(EXCLUDE_DEFAULT_LISTENERS);
		}
		else {
			removeChild(EXCLUDE_DEFAULT_LISTENERS);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setExcludesSuperClassListeners(Boolean excludesSuperClassListeners) {
		if (excludesSuperClassListeners == Boolean.TRUE) {
			addChild(EXCLUDE_SUPERCLASS_LISTENERS);
		}
		else {
			removeChild(EXCLUDE_SUPERCLASS_LISTENERS);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setExistenceCheckingType(ExistenceType type) {
		setAttribute(EXISTENCE_CHECKING, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setIdClassName(String className) {
		updateChildAttribute(ID_CLASS, CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setMappingsDescription(String description) {

		Element element = getChild(Mapping.ATTRIBUTES);

		if ((element == null) && (description == null)) {
			return;
		}

		if (element == null) {
			element = addChild(Mapping.ATTRIBUTES);
		}

		updateTextNode(element, DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setParentClassName(String className) {
		setAttribute(PARENT_CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostLoadMethod(String value) {
		updateChildAttribute(POST_LOAD, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostLoadMethodDescription(String name) {
		updateChildChildTextNode(POST_LOAD, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostPersistMethod(String value) {
		updateChildAttribute(POST_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostPersistMethodDescription(String name) {
		updateChildChildTextNode(POST_PERSIST, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostRemoveMethod(String value) {
		updateChildAttribute(POST_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostRemoveMethodDescription(String name) {
		updateChildChildTextNode(POST_REMOVE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostUpdateMethod(String value) {
		updateChildAttribute(POST_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPostUpdateMethodDescription(String name) {
		updateChildChildTextNode(POST_UPDATE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPrePersistMethod(String value) {
		updateChildAttribute(PRE_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPrePersistMethodDescription(String name) {
		updateChildChildTextNode(PRE_PERSIST, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreRemoveMethod(String value) {
		updateChildAttribute(PRE_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreRemoveMethodDescription(String name) {
		updateChildChildTextNode(PRE_REMOVE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreUpdateMethod(String value) {
		updateChildAttribute(PRE_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setPreUpdateMethodDescription(String name) {
		updateChildChildTextNode(PRE_UPDATE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setReadOnly(Boolean isReadOnly) {
		setAttribute(READ_ONLY, isReadOnly);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean shouldExcludesClassListeners() {
		return hasChild(EXCLUDE_DEFAULT_LISTENERS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Boolean shouldExcludesSuperClassListeners() {
		return hasChild(EXCLUDE_SUPERCLASS_LISTENERS);
	}
}