/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalConversionValue;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of a object type converter.
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class ObjectTypeConverter extends AbstractExternalForm
                                implements ExternalObjectTypeConverter {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the data-type property.
	 */
	static final String DATA_TYPE = "data-type";

	/**
	 * The element name used to store and retrieve the default-object-value child
	 * text node.
	 */
	static final String DEFAULT_OBJECT_VALUE = "default-object-value";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the object-type property.
	 */
	static final String OBJECT_TYPE = "object-type";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String OBJECT_TYPE_CONVERTER = "object-type-converter";

	/**
	 * Creates a new <code>ObjectTypeConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ObjectTypeConverter(AbstractExternalForm parent) {
		super(parent);
		this.index = -1;
	}

	/**
	 * Creates a new <code>ObjectTypeConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ObjectTypeConverter(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConversionValue addConversionValue(int index) {
		ConversionValue conversionValue = buildConversionValue(index);
		conversionValue.addSelf();
		return conversionValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName) {

		if (index == -1) {
			return super.addSelf(elementName);
		}

		return addChild(getParent(), elementName, index);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(DATA_TYPE);
		names.add(OBJECT_TYPE);
		return names;
	}

	private ConversionValue buildConversionValue(int index) {
		return new ConversionValue(this, index);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ConversionValue.CONVERSION_VALUE);
		names.add(DEFAULT_OBJECT_VALUE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalConversionValue> conversionValues() {

		int count = conversionValuesSize();
		List<ExternalConversionValue> conversionValues = new ArrayList<ExternalConversionValue>(count);

		for (int index = count; --index >= 0;) {
			ExternalConversionValue column = buildConversionValue(index);
			conversionValues.add(0, column);
		}

		return new ListListIterable<ExternalConversionValue>(conversionValues);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int conversionValuesSize() {
		return getChildrenSize(ConversionValue.CONVERSION_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConversionValue getConversionValue(int index) {

		Element element = getChild(ConversionValue.CONVERSION_VALUE, index);

		if (element == null) {
			return null;
		}

		return buildConversionValue(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDatatype() {
		return getAttribute(DATA_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultValue() {
		return getChildTextNode(DEFAULT_OBJECT_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return OBJECT_TYPE_CONVERTER;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjecttype() {
		return getAttribute(OBJECT_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllConversionValue() {
		removeChildren(ConversionValue.CONVERSION_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeConversionValue(int index) {
		ConversionValue conversionValue = buildConversionValue(index);
		conversionValue.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDatatype(String datatypeName) {
		setAttribute(DATA_TYPE, datatypeName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultValue(String value) {
		updateChildTextNode(DEFAULT_OBJECT_VALUE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setObjecttype(String objecttypeName) {
		setAttribute(OBJECT_TYPE, objecttypeName);
	}
}