/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalQueryHint;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The abstract definition of a query, which can either be a named query (JPQL query), a named
 * native query (SQL query), or a named stored procedure query.
 *
 * @see Entity
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
abstract class Query extends AbstractExternalForm
                     implements ExternalNamedQuery {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The element name of the child text node for the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The element name used to store and retrieve the query child text node.
	 */
	static final String QUERY = "query";

	/**
	 * Creates a new <code>Query</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	Query(Entity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>Query</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	Query(ORMConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addQueryHint(int index, String name, String value) {
		QueryHint queryHint = buildQueryHint(index);
		queryHint.addSelf();
		queryHint.setName(name);
		queryHint.setValue(value);
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(QUERY);
		names.add(QueryHint.HINT);
		return names;
	}

	private QueryHint buildQueryHint(int index) {
		return new QueryHint(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getQuery() {
		return getChildTextNode(QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final QueryHint getQueryHint(int index) {

		Element element = getChild(QueryHint.HINT, index);

		if (element != null) {
			return buildQueryHint(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ListIterable<ExternalQueryHint> queryHints() {

		int count = queryHintsSize();
		List<ExternalQueryHint> queryHints = new ArrayList<ExternalQueryHint>(count);

		for (int index = count; --index >= 0;) {
			ExternalQueryHint queryHint = buildQueryHint(index);
			queryHints.add(0, queryHint);
		}

		return new ListListIterable<ExternalQueryHint>(queryHints);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int queryHintsSize() {
		return getChildrenSize(QueryHint.HINT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeQueryHint(int index) {
		QueryHint queryHint = buildQueryHint(index);
		queryHint.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setQuery(String query) {
		updateChildTextNode(QUERY, query);
	}
}