/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalColumnResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalSQLResultSetMapping;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of an SQL Result Set.
 *
 * @see Entity
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
final class SQLResultSetMapping extends AbstractExternalForm
                                implements ExternalSQLResultSetMapping {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The element name of the child text node for the description.
	 */
	static final String DESCRIPTION = "description";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String SQL_RESULT_SET_MAPPING = "sql-result-set-mapping";

	/**
	 * Creates a new <code>SQLResultSetMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	SQLResultSetMapping(Entity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>SQLResultSetMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	SQLResultSetMapping(ORMConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addColumnResult(int index, String name) {
		ColumnResult entityResult = buildColumnResult(index);
		entityResult.addSelf();
		entityResult.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addEntityResult(int index, String className) {
		EntityResult entityResult = buildEntityResult(index);
		entityResult.addSelf();
		entityResult.setEntityClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	private ColumnResult buildColumnResult(int index) {
		return new ColumnResult(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(EntityResult.ENTITY_RESULT);
		names.add(ColumnResult.COLUMN_RESULT);
		return names;
	}

	private EntityResult buildEntityResult(int index) {
		return new EntityResult(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalColumnResult> columnResults() {

		int count = columnResultsSize();
		List<ExternalColumnResult> columnResults = new ArrayList<ExternalColumnResult>(count);

		for (int index = count; --index >= 0;) {
			ExternalColumnResult columnResult = buildColumnResult(index);
			columnResults.add(0, columnResult);
		}

		return new ListListIterable<ExternalColumnResult>(columnResults);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnResultsSize() {
		return getChildrenSize(ColumnResult.COLUMN_RESULT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalEntityResult> entityResults() {

		int count = entityResultsSize();
		List<ExternalEntityResult> entityResults = new ArrayList<ExternalEntityResult>(count);

		for (int index = count; --index >= 0;) {
			ExternalEntityResult entityResult = buildEntityResult(index);
			entityResults.add(0, entityResult);
		}

		return new ListListIterable<ExternalEntityResult>(entityResults);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int entityResultsSize() {
		return getChildrenSize(EntityResult.ENTITY_RESULT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalColumnResult getColumnResult(int index) {

		Element element = getChild(ColumnResult.COLUMN_RESULT, index);

		if (element == null) {
			return null;
		}

		return buildColumnResult(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return SQL_RESULT_SET_MAPPING;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityResult getEntityResult(int index) {

		Element element = getChild(EntityResult.ENTITY_RESULT, index);

		if (element == null) {
			return null;
		}

		return buildEntityResult(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumnResult(int index) {
		EntityResult entityResult = buildEntityResult(index);
		entityResult.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEntityResult(int index) {
		EntityResult entityResult = buildEntityResult(index);
		entityResult.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}
}