/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.collection;

import java.io.Serializable;
import java.util.Collection;
import java.util.EmptyStackException;
import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * Linked list implementation of the {@link Stack} interface.
 * @see LinkedList
 */
public class LinkedStack<E>
	implements Stack<E>, Cloneable, Serializable
{
	private LinkedList<E> elements;

	private static final long serialVersionUID = 1L;


	// ********** constructors **********

	/**
	 * Construct an empty stack.
	 */
	public LinkedStack() {
		super();
		this.elements = new LinkedList<E>();
	}

	/**
	 * Construct a stack containing the elements of the specified
	 * collection. The stack will pop its elements in reverse of the
	 * order they are returned by the collection's iterator (i.e. the
	 * last element returned by the collection's iterator will be the
	 * first element returned by {@link #pop()}).
	 */
	public LinkedStack(Collection<? extends E> collection) {
		super();
		this.elements = new LinkedList<E>(collection);
	}


	// ********** Stack implementation **********

	@Override
	public void push(E element) {
		this.elements.addLast(element);
	}

	@Override
	public E pop() {
		try {
			return this.elements.removeLast();
		} catch (NoSuchElementException ex) {
			throw new EmptyStackException();
		}
	}

	@Override
	public E peek() {
		try {
			return this.elements.getLast();
		} catch (NoSuchElementException ex) {
			throw new EmptyStackException();
		}
	}

	@Override
	public boolean isEmpty() {
		return this.elements.isEmpty();
	}


	// ********** standard methods **********

	@Override
	public LinkedStack<E> clone() {
		try {
			@SuppressWarnings("unchecked")
			LinkedStack<E> clone = (LinkedStack<E>) super.clone();
			@SuppressWarnings("unchecked")
			LinkedList<E> list = (LinkedList<E>) this.elements.clone();
			clone.elements = list;
			return clone;
		} catch (CloneNotSupportedException ex) {
			throw new InternalError();
		}
	}

	/**
	 * Print the elements in the order in which they are "pushed" on to
	 * the stack (as opposed to the order in which they will be "popped"
	 * off of the stack).
	 */
	@Override
	public String toString() {
		return this.elements.toString();
	}
}