/*******************************************************************************
 * Copyright (c) 2005, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.jdbc;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;
import java.util.Properties;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * This implementation of the {@link Driver JDBC interface} delegates all calls
 * to a "real" JDBC driver. This wrapper simplifies the dynamic loading of a
 * JDBC driver that is not on the "Java" classpath (the classpath set at
 * the time the JVM starts up). This is because, if you use a URL class loader
 * to load a driver directly, the {@link java.sql.DriverManager} will not allow your code
 * access to the driver. For security reasons, the driver must be loaded
 * by the same class loader chain that loaded the code that calls
 * {@link java.sql.DriverManager#getConnection(String)}.
 */
public class DriverWrapper
	implements Driver
{
	/** the "real" JDBC driver */
	private final Driver driver;


	/**
	 * Wrap the driver for the specified driver class and class loader
	 * that can load the driver.
	 */
	@SuppressWarnings("unchecked")
	public DriverWrapper(String driverClassName, ClassLoader classLoader)
		throws ClassNotFoundException, InstantiationException, IllegalAccessException
	{
		this((Class<Driver>) Class.forName(driverClassName, true, classLoader));
	}

	/**
	 * Wrap the driver for the specified driver class.
	 */
	public DriverWrapper(Class<Driver> driverClass)
		throws InstantiationException, IllegalAccessException
	{
		this(driverClass.newInstance());
	}

	/**
	 * Wrap the specified driver.
	 */
	public DriverWrapper(Driver driver) {
		super();
		if (driver == null) {
			throw new NullPointerException();
		}
		this.driver = driver;
	}


	// ********** Driver implementation **********

	@Override
	public Connection connect(String url, Properties info) throws SQLException {
		return this.driver.connect(url, info);
	}

	@Override
	public boolean acceptsURL(String url) throws SQLException {
		return this.driver.acceptsURL(url);
	}

	@Override
	public DriverPropertyInfo[] getPropertyInfo(String url, Properties info) throws SQLException {
		return this.driver.getPropertyInfo(url, info);
	}

	@Override
	public int getMajorVersion() {
		return this.driver.getMajorVersion();
	}

	@Override
	public int getMinorVersion() {
		return this.driver.getMinorVersion();
	}

	@Override
	public boolean jdbcCompliant() {
		return this.driver.jdbcCompliant();
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.driver);
	}
}