/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.listener;

import java.lang.reflect.Method;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.model.event.CollectionAddEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionClearEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListAddEvent;
import org.eclipse.persistence.tools.utility.model.event.ListChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.ListClearEvent;
import org.eclipse.persistence.tools.utility.model.event.ListEvent;
import org.eclipse.persistence.tools.utility.model.event.ListMoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListReplaceEvent;

/**
 * This class is used by {@link ReflectiveChangeListener} when the requested listener
 * needs to implement multiple methods (i.e. {@link CollectionChangeListener} or
 * {@link ListChangeListener}).
 */
class MultiMethodReflectiveChangeListener
	extends ReflectiveChangeListener
	implements CollectionChangeListener, ListChangeListener
{
	/** the methods we will invoke on the target object */
	private final Method addMethod;
	private final Method removeMethod;
	private final Method replaceMethod;	// this can be null
	private final Method moveMethod;	// this can be null
	private final Method clearMethod;
	private final Method changeMethod;


	/**
	 * The "replace" and "move" methods are optional.
	 */
	MultiMethodReflectiveChangeListener(Object target, Method addMethod, Method removeMethod, Method replaceMethod, Method moveMethod, Method clearMethod, Method changeMethod) {
		super(target);
		this.addMethod = addMethod;
		this.removeMethod = removeMethod;
		this.replaceMethod = replaceMethod;
		this.moveMethod = moveMethod;
		this.clearMethod = clearMethod;
		this.changeMethod = changeMethod;
	}

	/**
	 * No "replace" or "move" methods.
	 */
	MultiMethodReflectiveChangeListener(Object target, Method addMethod, Method removeMethod, Method clearMethod, Method changeMethod) {
		this(target, addMethod, removeMethod, null, null, clearMethod, changeMethod);
	}


	// ********** CollectionChangeListener implementation **********

	private void invoke(Method method, CollectionEvent event) {
		if (method.getParameterTypes().length == 0) {
			ObjectTools.execute(this.target, method, ObjectTools.EMPTY_OBJECT_ARRAY);
		} else {
			ObjectTools.execute(this.target, method, new CollectionEvent[] {event});
		}
	}

	@Override
	public void itemsAdded(CollectionAddEvent event) {
		this.invoke(this.addMethod, event);
	}

	@Override
	public void itemsRemoved(CollectionRemoveEvent event) {
		this.invoke(this.removeMethod, event);
	}

	@Override
	public void collectionCleared(CollectionClearEvent event) {
		this.invoke(this.clearMethod, event);
	}

	@Override
	public void collectionChanged(CollectionChangeEvent event) {
		this.invoke(this.changeMethod, event);
	}


	// ********** ListChangeListener implementation **********

	private void invoke(Method method, ListEvent event) {
		if (method.getParameterTypes().length == 0) {
			ObjectTools.execute(this.target, method, ObjectTools.EMPTY_OBJECT_ARRAY);
		} else {
			ObjectTools.execute(this.target, method, new ListEvent[] {event});
		}
	}

	@Override
	public void itemsAdded(ListAddEvent event) {
		this.invoke(this.addMethod, event);
	}

	@Override
	public void itemsRemoved(ListRemoveEvent event) {
		this.invoke(this.removeMethod, event);
	}

	@Override
	public void itemsReplaced(ListReplaceEvent event) {
		this.invoke(this.replaceMethod, event);
	}

	@Override
	public void itemsMoved(ListMoveEvent event) {
		this.invoke(this.moveMethod, event);
	}

	@Override
	public void listCleared(ListClearEvent event) {
		this.invoke(this.clearMethod, event);
	}

	@Override
	public void listChanged(ListChangeEvent event) {
		this.invoke(this.changeMethod, event);
	}
}