/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import java.util.Arrays;
import org.eclipse.persistence.tools.utility.model.Model;
import org.eclipse.persistence.tools.utility.model.event.ListAddEvent;
import org.eclipse.persistence.tools.utility.model.event.ListChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.ListClearEvent;
import org.eclipse.persistence.tools.utility.model.event.ListMoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListReplaceEvent;
import org.eclipse.persistence.tools.utility.model.listener.ListChangeListener;

/**
 * Extend {@link ItemAspectListValueModelAdapter} to listen to one or more list
 * aspects of each item in the wrapped list model.
 */
@SuppressWarnings("nls")
public class ItemListListValueModelAdapter<E>
	extends ItemAspectListValueModelAdapter<E>
{

	/** The names of the subject's lists that we listen to. */
	protected final String[] listNames;

	/** Listener that listens to all the items in the list. */
	protected final ListChangeListener itemListListener;


	// ********** constructors **********

	/**
	 * Construct an adapter for the specified item List aspects.
	 */
	public ItemListListValueModelAdapter(ListValueModel<E> listHolder, String... listNames) {
		super(listHolder);
		if (listNames == null) {
			throw new NullPointerException();
		}
		this.listNames = listNames;
		this.itemListListener = this.buildItemListListener();
	}

	/**
	 * Construct an adapter for the specified item List aspects.
	 */
	public ItemListListValueModelAdapter(CollectionValueModel<E> collectionHolder, String... listNames) {
		this(new CollectionListValueModelAdapter<E>(collectionHolder), listNames);
	}


	// ********** initialization **********

	/**
	 * All we really care about is the fact that the List aspect has
	 * changed.  Do the same thing no matter which event occurs.
	 */
	protected ListChangeListener buildItemListListener() {
		return new ListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent event) {
				ItemListListValueModelAdapter.this.itemAspectChanged(event);
			}
			@Override
			public void itemsRemoved(ListRemoveEvent event) {
				ItemListListValueModelAdapter.this.itemAspectChanged(event);
			}
			@Override
			public void itemsReplaced(ListReplaceEvent event) {
				ItemListListValueModelAdapter.this.itemAspectChanged(event);
			}
			@Override
			public void itemsMoved(ListMoveEvent event) {
				ItemListListValueModelAdapter.this.itemAspectChanged(event);
			}
			@Override
			public void listCleared(ListClearEvent event) {
				ItemListListValueModelAdapter.this.itemAspectChanged(event);
			}
			@Override
			public void listChanged(ListChangeEvent event) {
				ItemListListValueModelAdapter.this.itemAspectChanged(event);
			}
			@Override
			public String toString() {
				return "item list listener: " + Arrays.asList(ItemListListValueModelAdapter.this.listNames);
			}
		};
	}


	// ********** behavior **********

	@Override
	protected void engageItem_(Model item) {
		for (String listName : this.listNames) {
			item.addListChangeListener(listName, this.itemListListener);
		}
	}

	@Override
	protected void disengageItem_(Model item) {
		for (String listName : this.listNames) {
			item.removeListChangeListener(listName, this.itemListListener);
		}
	}
}