/*******************************************************************************
 * Copyright (c) 2002, 2007 Innoopract Informationssysteme GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Innoopract Informationssysteme GmbH - initial API and implementation
 ******************************************************************************/

package org.eclipse.swt.graphics;

import org.eclipse.swt.SWT;
import org.eclipse.swt.internal.graphics.ResourceFactory;

/**
 * Instances of this class describe fonts.
 * <p>
 * Application code does <em>not</em> need to explicitly release the
 * resources managed by each instance when those instances are no longer
 * required, and thus no <code>dispose()</code> method is provided.
 *
 * @see Font
 * @since 1.0
 */
public final class FontData {

  private final String name;
  private final int height;
  private final int style;
  
  /**  
   * Constructs a new font data given a font name,
   * the height of the desired font in points, 
   * and a font style.
   *
   * @param name the name of the font (must not be null)
   * @param height the font height in points
   * @param style a bit or combination of NORMAL, BOLD, ITALIC
   *
   * @exception IllegalArgumentException <ul>
   *    <li>ERROR_NULL_ARGUMENT - when the font name is null</li>
   *    <li>ERROR_INVALID_ARGUMENT - if the height is negative</li>
   * </ul>
   */
  public FontData( final String name, final int height, final int style ) {
    if( name == null ) {
      SWT.error( SWT.ERROR_NULL_ARGUMENT );
    }
    this.name = name;
    this.height = height;
    this.style = style;
  }

  /**
   * Constructs a new FontData given a string representation
   * in the form generated by the <code>FontData.toString</code>
   * method.
   * <!--
   * <p>
   * Note that the representation varies between platforms,
   * and a FontData can only be created from a string that was 
   * generated on the same platform.
   * </p>
   * -->
   * 
   * @param string the string representation of a <code>FontData</code> (must not be null)
   *
   * @exception IllegalArgumentException <ul>
   *    <li>ERROR_NULL_ARGUMENT - if the argument is null</li>
   *    <li>ERROR_INVALID_ARGUMENT - if the argument does not represent a valid description</li>
   * </ul>
   *
   * @see #toString
   */
  public FontData( final String string ) {
    if( string == null ) {
      SWT.error( SWT.ERROR_NULL_ARGUMENT );
    }
    int start = 0;
    int end = string.indexOf( '|' );
    if( end == -1 ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    String version1 = string.substring( start, end );
    try {
      if( Integer.parseInt( version1 ) != 1 ) {
        SWT.error( SWT.ERROR_INVALID_ARGUMENT );
      }
    } catch( final NumberFormatException e ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    start = end + 1;
    end = string.indexOf( '|', start );
    if( end == -1 ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    String name = string.substring( start, end );
    start = end + 1;
    end = string.indexOf( '|', start );
    if( end == -1 ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    int height = 0;
    try {
      height = Integer.parseInt( string.substring( start, end ) );
    } catch( NumberFormatException e ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    start = end + 1;
    end = string.indexOf( '|', start );
    if( end == -1 ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    int style = 0;
    try {
      style = Integer.parseInt( string.substring( start, end ) );
    } catch( NumberFormatException e ) {
      SWT.error( SWT.ERROR_INVALID_ARGUMENT );
    }
    start = end + 1;
    end = string.indexOf( '|', start );
    this.name = name;
    this.height = height;
    this.style = style;
  }

  /**
   * Returns a string representation of the receiver which is suitable for
   * constructing an equivalent instance using the <code>FontData(String)</code>
   * constructor.
   * 
   * @return a string representation of the FontData
   */
  public String toString() {
    StringBuffer buffer = new StringBuffer();
    buffer.append( "1|" ); //$NON-NLS-1$
    buffer.append( getName() );
    buffer.append( "|" ); //$NON-NLS-1$
    buffer.append( getHeight() );
    buffer.append( "|" ); //$NON-NLS-1$
    buffer.append( getStyle() );
    buffer.append( "|" ); //$NON-NLS-1$
    return buffer.toString();
  }
  
  /**
   * Returns the height of the receiver in points.
   *
   * @return the height of this FontData
   */
  public int getHeight() {
    return height;
  }

  /**
   * Returns the name of the receiver.
   * 
   * @return the name of this <code>FontData</code>
   */
  public String getName() {
    return name;
  }

  /**
   * Returns the style of the receiver which is a bitwise OR of 
   * one or more of the <code>SWT</code> constants NORMAL, BOLD
   * and ITALIC.
   *
   * @return the style of this <code>FontData</code>
   */
  public int getStyle() {
    return style;
  }
  
  /**
   * Compares the argument to the receiver, and returns true
   * if they represent the <em>same</em> object using a class
   * specific comparison.
   *
   * @param obj the object to compare with this object
   * @return <code>true</code> if the object is the same as this object and <code>false</code> otherwise
   *
   * @see #hashCode
   */
  public boolean equals( final Object obj ) {
    boolean result = false;
    if( obj != null && obj instanceof FontData ) {
      FontData toCompare = ( FontData )obj;
      result =    name.hashCode() == toCompare.name.hashCode()
               && height == toCompare.height
               && style == toCompare.style;
    }
    return result;
  }

  /**
   * Returns an integer hash code for the receiver. Any two 
   * objects that return <code>true</code> when passed to 
   * <code>equals</code> must return the same value for this
   * method.
   *
   * @return the receiver's hash
   *
   * @see #equals
   */
  public int hashCode() {
    return ResourceFactory.fontHashCode( name, height, style );
  }
}
