/*******************************************************************************
 * Copyright (c) 2007, 2012 Innoopract Informationssysteme GmbH and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Innoopract Informationssysteme GmbH - initial API and implementation
 *    EclipseSource - ongoing development
 ******************************************************************************/
package org.eclipse.rap.rwt.internal.theme;

import java.io.IOException;

import org.eclipse.rap.rwt.internal.application.RWTFactory;
import org.eclipse.rap.rwt.internal.service.ContextProvider;
import org.eclipse.rap.rwt.internal.theme.css.ConditionalValue;
import org.eclipse.rap.rwt.internal.theme.css.CssFileReader;
import org.eclipse.rap.rwt.internal.theme.css.StyleSheet;
import org.eclipse.rap.rwt.resources.ResourceLoader;
import org.eclipse.rap.rwt.service.ISessionStore;
import org.eclipse.swt.widgets.Widget;


/**
 * Used to switch between themes at runtime.
 */
public final class ThemeUtil {

  private static final String DEFAULT_THEME_CSS = "resource/theme/default.css";

  public static final String CURR_THEME_ATTR = "org.eclipse.rap.theme.current";

  /**
   * Returns the ids of all themes that are currently registered.
   *
   * @return an array of the theme ids, never <code>null</code>
   */
  public static String[] getAvailableThemeIds() {
    return RWTFactory.getThemeManager().getRegisteredThemeIds();
  }

  /**
   * Returns the id of the currently active theme.
   *
   * @return the id of the current theme, never <code>null</code>
   */
  public static String getCurrentThemeId() {
    ISessionStore sessionStore = ContextProvider.getSessionStore();
    return ( String )sessionStore.getAttribute( CURR_THEME_ATTR );
  }

  /**
   * Sets the current theme to the theme identified by the given id.
   * @param sessionStore TODO
   * @param themeId the id of the theme to activate
   *
   * @throws IllegalArgumentException if no theme with the given id is
   *             registered
   */
  public static void setCurrentThemeId( ISessionStore sessionStore, String themeId ) {
    sessionStore.setAttribute( CURR_THEME_ATTR, themeId );
  }

  public static Theme getCurrentTheme() {
    return RWTFactory.getThemeManager().getTheme( getCurrentThemeId() );
  }

  private static Theme getFallbackTheme() {
    ThemeManager themeManager = RWTFactory.getThemeManager();
    return themeManager.getTheme( ThemeManager.FALLBACK_THEME_ID );
  }

  static StyleSheet readDefaultThemeStyleSheet() {
    StyleSheet result;
    try {
      ResourceLoader resLoader = ThemeManager.STANDARD_RESOURCE_LOADER;
      result = CssFileReader.readStyleSheet( DEFAULT_THEME_CSS, resLoader );
    } catch( IOException ioe ) {
      String msg = "Failed to load default theme: " + DEFAULT_THEME_CSS;
      throw new ThemeManagerException( msg, ioe );
    }
    return result;
  }

  //////////////////////////////////////
  // Methods for accessing themed values

  public static QxType getCssValue( String cssElement, String cssProperty, SimpleSelector selector )
  {
    return getCssValue( cssElement, cssProperty, selector, null );
  }

  public static QxType getCssValue( String cssElement,
                                    String cssProperty,
                                    ValueSelector selector,
                                    Widget widget )
  {
    Theme theme = getCurrentTheme();
    ThemeCssValuesMap valuesMap = theme.getValuesMap();
    ConditionalValue[] values = valuesMap.getValues( cssElement, cssProperty );
    QxType result = selector.select( values, widget );
    if( result == null ) {
      // resort to fallback theme
      theme = getFallbackTheme();
      valuesMap = theme.getValuesMap();
      values = valuesMap.getValues( cssElement, cssProperty );
      result = selector.select( values, widget );
    }
    return result;
  }

  private ThemeUtil() {
    // prevent instantiation
  }
}
