/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.sdk.sourcebuilder.method;

import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.Signature;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.nls.sdk.model.INlsEntry;
import org.eclipse.scout.sdk.util.ScoutUtility;
import org.eclipse.scout.sdk.util.signature.IImportValidator;
import org.eclipse.scout.sdk.util.signature.SignatureCache;
import org.eclipse.scout.sdk.util.signature.SignatureUtility;
import org.eclipse.scout.sdk.util.type.MethodParameter;

/**
 * <h3>{@link MethodBodySourceBuilderFactory}</h3>
 * 
 * @author Andreas Hoegger
 * @since 3.10.0 07.03.2013
 */
public final class MethodBodySourceBuilderFactory {

  private MethodBodySourceBuilderFactory() {
  }

  public static IMethodBodySourceBuilder createAutoGeneratedMethodBody() {
    return new IMethodBodySourceBuilder() {
      @Override
      public void createSource(IMethodSourceBuilder methodBuilder, StringBuilder source, String lineDelimiter, IJavaProject ownerProject, IImportValidator validator) throws CoreException {
        source.append(ScoutUtility.getCommentAutoGeneratedMethodStub());
        String returnTypeSignature = methodBuilder.getReturnTypeSignature();
        if (!StringUtility.isNullOrEmpty(returnTypeSignature) && !Signature.SIG_VOID.equals(returnTypeSignature)) {
          source.append(lineDelimiter).append("return " + ScoutUtility.getDefaultValueOf(returnTypeSignature) + ";");
        }
      }
    };
  }

  public static IMethodBodySourceBuilder createSuperCallMethodBody(final boolean addAutoGeneratedMethodStubComment) {
    return new IMethodBodySourceBuilder() {
      @Override
      public void createSource(IMethodSourceBuilder methodBuilder, StringBuilder source, String lineDelimiter, IJavaProject ownerProject, IImportValidator validator) throws CoreException {
        if (addAutoGeneratedMethodStubComment) {
          source.append(ScoutUtility.getCommentAutoGeneratedMethodStub());
          source.append(lineDelimiter);
        }
        String returnTypeSignature = methodBuilder.getReturnTypeSignature();
        boolean isConstructor = StringUtility.isNullOrEmpty(returnTypeSignature);
        if (!isConstructor && !Signature.SIG_VOID.equals(returnTypeSignature)) {
          source.append("return ");
        }
        source.append("super");
        if (!isConstructor) {
          source.append(".");
          source.append(methodBuilder.getElementName());
        }
        source.append("(");
        List<MethodParameter> parameters = methodBuilder.getParameters();
        MethodParameter[] params = parameters.toArray(new MethodParameter[parameters.size()]);
        for (int i = 0; i < params.length; i++) {
          source.append(params[i].getName());
          if (i < params.length - 1) {
            source.append(", ");
          }
        }
        source.append(");");
      }
    };
  }

  public static IMethodBodySourceBuilder createNlsEntryReferenceBody(final INlsEntry entry) {
    return new IMethodBodySourceBuilder() {
      @Override
      public void createSource(IMethodSourceBuilder methodBuilder, StringBuilder source, String lineDelimiter, IJavaProject ownerProject, IImportValidator validator) throws CoreException {
        source.append("  return ");
        if (entry != null) {
          String nlsTypeSig = SignatureCache.createTypeSignature(entry.getProject().getNlsAccessorType().getFullyQualifiedName());
          source.append(SignatureUtility.getTypeReference(nlsTypeSig, validator));
          source.append(".get(\"").append(entry.getKey()).append("\");");
        }
        else {
          source.append("null;");
        }
      }
    };
  }

  public static IMethodBodySourceBuilder createReturnClassReferenceBody(final String typeSignature) {
    return new IMethodBodySourceBuilder() {

      @Override
      public void createSource(IMethodSourceBuilder methodBuilder, StringBuilder source, String lineDelimiter, IJavaProject ownerProject, IImportValidator validator) throws CoreException {
        source.append("return ").append(validator.getTypeName(typeSignature)).append(".class;");
      }
    };
  }

  public static IMethodBodySourceBuilder createSimpleMethodBody(final String methodBody) {
    return new IMethodBodySourceBuilder() {
      @Override
      public void createSource(IMethodSourceBuilder methodBuilder, StringBuilder source, String lineDelimiter, IJavaProject ownerProject, IImportValidator validator) throws CoreException {
        source.append(methodBody);
      }
    };
  }
}
