/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
/**
 *
 */
package org.eclipse.scout.sdk.operation.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.scout.commons.CompositeObject;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.sdk.extensions.runtime.classes.IRuntimeClasses;
import org.eclipse.scout.sdk.extensions.runtime.classes.RuntimeClasses;
import org.eclipse.scout.sdk.operation.IOperation;
import org.eclipse.scout.sdk.operation.jdt.packageFragment.ExportPolicy;
import org.eclipse.scout.sdk.operation.jdt.type.PrimaryTypeNewOperation;
import org.eclipse.scout.sdk.sourcebuilder.comment.CommentSourceBuilderFactory;
import org.eclipse.scout.sdk.sourcebuilder.field.IFieldSourceBuilder;
import org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder;
import org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder;
import org.eclipse.scout.sdk.util.ScoutUtility;
import org.eclipse.scout.sdk.util.signature.SignatureCache;
import org.eclipse.scout.sdk.util.typecache.IWorkingCopyManager;

/**
 * <h3>{@link ServiceNewOperation}</h3> To create a new service.<br>
 * A service consists out of:
 * <ul>
 * <li><b>a service implementation</b> located in client or server bundles</li>
 * <li><b>service interface</b> used mainly for remote services can be an a shared, server or client bundle</li>
 * <li><b>proxy registrations</b> must be 0...n client bundles</li>
 * <li><b>service registrations</b> can be in client or server bundles.</li>
 * </ul>
 * In case the service implementation bundle is not set no service implementation will be created nor any service
 * registrations added.<br>
 * In case the service interface bundle is null no service proxy registrations will be created.
 * 
 * @author Andreas Hoegger
 * @since 1.0.8 03.02.2010
 */
public class ServiceNewOperation implements IOperation {

  private IJavaProject m_interfaceProject;
  private String m_interfacePackageName;
  private TypeSourceBuilder m_interfaceBuilder;
  private IJavaProject m_implementationProject;
  private String m_implementationPackageName;
  private TypeSourceBuilder m_implementationBuilder;
  private final List<IJavaProject> m_proxyRegistrationProjects;
  private final List<ServiceRegistrationDescription> m_serviceRegistrationDescriptions;

  private IType m_createdServiceInterface;
  private IType m_createdServiceImplementation;
  private boolean m_formatSource;

  public ServiceNewOperation(String serviceInterfaceName, String serviceName) {
    m_formatSource = true;
    m_implementationBuilder = new TypeSourceBuilder(serviceName);
    m_interfaceBuilder = new TypeSourceBuilder(serviceInterfaceName);
    // defaults
    m_implementationBuilder.setFlags(Flags.AccPublic);
    m_implementationBuilder.setCommentSourceBuilder(CommentSourceBuilderFactory.createPreferencesTypeCommentBuilder());

    m_interfaceBuilder.setFlags(Flags.AccPublic | Flags.AccInterface);
    m_interfaceBuilder.setCommentSourceBuilder(CommentSourceBuilderFactory.createPreferencesTypeCommentBuilder());

    m_proxyRegistrationProjects = new ArrayList<IJavaProject>();
    m_serviceRegistrationDescriptions = new ArrayList<ServiceRegistrationDescription>();
  }

  @Override
  public String getOperationName() {
    return "create new service...";
  }

  @Override
  public void validate() throws IllegalArgumentException {
    if (getInterfaceProject() != null || getInterfacePackageName() != null) {
      if (getInterfaceProject() == null || getInterfacePackageName() == null) {
        throw new IllegalArgumentException("interface project and packagename must be both set or null. To avoid the interface creation set both members to null.");
      }
      getInterfaceSourceBuilder().validate();
    }
    if (getImplementationProject() != null || getImplementationPackageName() != null) {
      if (getImplementationProject() == null || getImplementationPackageName() == null) {
        throw new IllegalArgumentException("implementation project and packagename must be both set or null. To avoid the interface creation set both members to null.");
      }
      getImplementationSourceBuilder().validate();
    }
  }

  @Override
  public void run(IProgressMonitor monitor, IWorkingCopyManager workingCopyManager) throws CoreException, IllegalArgumentException {
    if (getInterfaceProject() != null) {
      // create interface
      List<String> interfaceSignatures = getInterfaceSourceBuilder().getInterfaceSignatures();
      String service2Signature = SignatureCache.createTypeSignature(IRuntimeClasses.IService);
      if (interfaceSignatures.isEmpty()) {
        interfaceSignatures.add(service2Signature);
      }
      PrimaryTypeNewOperation interfaceOp = new PrimaryTypeNewOperation(getInterfaceSourceBuilder(), getInterfacePackageName(), getInterfaceProject());
      interfaceOp.setIcuCommentSourceBuilder(CommentSourceBuilderFactory.createPreferencesCompilationUnitCommentBuilder());
      interfaceOp.setPackageExportPolicy(ExportPolicy.AddPackage);

      interfaceOp.setFormatSource(isFormatSource());
      interfaceOp.validate();
      interfaceOp.run(monitor, workingCopyManager);
      m_createdServiceInterface = interfaceOp.getCreatedType();
      workingCopyManager.register(m_createdServiceInterface.getCompilationUnit(), monitor);
      // register
      for (IJavaProject cb : getProxyRegistrationProjects()) {
        ScoutUtility.registerServiceClass(IRuntimeClasses.EXTENSION_POINT_CLIENT_SERVICE_PROXIES, IRuntimeClasses.EXTENSION_ELEMENT_CLIENT_SERVICE_PROXY,
            getCreatedServiceInterface().getFullyQualifiedName(), new ServiceRegistrationDescription(cb, (String) null, IRuntimeClasses.ClientProxyServiceFactory));
      }
    }
    if (getImplementationProject() != null) {
      if (getCreatedServiceInterface() != null) {
        getImplementationSourceBuilder().addInterfaceSignature(SignatureCache.createTypeSignature(getCreatedServiceInterface().getFullyQualifiedName()));
      }
      if (StringUtility.isNullOrEmpty(getImplementationSourceBuilder().getSuperTypeSignature())) {
        getImplementationSourceBuilder().setSuperTypeSignature(RuntimeClasses.getSuperTypeSignature(IRuntimeClasses.IService, getImplementationProject()));
      }
      PrimaryTypeNewOperation implementationOp = new PrimaryTypeNewOperation(getImplementationSourceBuilder(), getImplementationPackageName(), getImplementationProject());
      implementationOp.setIcuCommentSourceBuilder(CommentSourceBuilderFactory.createPreferencesCompilationUnitCommentBuilder());
      implementationOp.setFormatSource(isFormatSource());
      implementationOp.setPackageExportPolicy(ExportPolicy.AddPackage);
      implementationOp.validate();
      implementationOp.run(monitor, workingCopyManager);
      m_createdServiceImplementation = implementationOp.getCreatedType();
      workingCopyManager.register(m_createdServiceImplementation.getCompilationUnit(), monitor);

      // register services
      List<ServiceRegistrationDescription> serviceRegistrationDescs = getServiceRegistrations();
      for (ServiceRegistrationDescription desc : serviceRegistrationDescs) {
        ScoutUtility.registerServiceClass(IRuntimeClasses.EXTENSION_POINT_SERVICES, IRuntimeClasses.EXTENSION_ELEMENT_SERVICE, getCreatedServiceImplementation().getFullyQualifiedName(), desc);
      }
    }
  }

  public void setInterfaceProject(IJavaProject interfaceProject) {
    m_interfaceProject = interfaceProject;
  }

  public IJavaProject getInterfaceProject() {
    return m_interfaceProject;
  }

  public void setInterfacePackageName(String interfacePackageName) {
    m_interfacePackageName = interfacePackageName;
  }

  public String getInterfacePackageName() {
    return m_interfacePackageName;
  }

  public TypeSourceBuilder getInterfaceSourceBuilder() {
    return m_interfaceBuilder;
  }

  public void setFormatSource(boolean formatSource) {
    m_formatSource = formatSource;
  }

  public boolean isFormatSource() {
    return m_formatSource;
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.AbstractJavaElementSourceBuilder#getElementName()
   */
  public String getInterfaceName() {
    return m_interfaceBuilder.getElementName();
  }

  /**
   * @param interfaceSignature
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addInterfaceSignature(java.lang.String)
   */
  public void addInterfaceInterfaceSignature(String interfaceSignature) {
    m_interfaceBuilder.addInterfaceSignature(interfaceSignature);
  }

  /**
   * @param interfaceSignature
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#removeInterfaceSignature(java.lang.String)
   */
  public boolean removeInterfaceInterfaceSignature(String interfaceSignature) {
    return m_interfaceBuilder.removeInterfaceSignature(interfaceSignature);
  }

  /**
   * @param interfaceSignatures
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#setInterfaceSignatures(java.lang.String[])
   */
  public void setInterfaceInterfaceSignatures(String[] interfaceSignatures) {
    m_interfaceBuilder.setInterfaceSignatures(interfaceSignatures);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getInterfaceSignatures()
   */
  public List<String> getInterfaceInterfaceSignatures() {
    return m_interfaceBuilder.getInterfaceSignatures();
  }

  /**
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addFieldSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.field.IFieldSourceBuilder)
   */
  public void addInterfaceFieldSourceBuilder(IFieldSourceBuilder builder) {
    m_interfaceBuilder.addFieldSourceBuilder(builder);
  }

  /**
   * @param builder
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#removeFieldSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.field.IFieldSourceBuilder)
   */
  public boolean removeInterfaceFieldSourceBuilder(IFieldSourceBuilder builder) {
    return m_interfaceBuilder.removeFieldSourceBuilder(builder);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getFieldSourceBuilders()
   */
  public List<IFieldSourceBuilder> getInterfaceFieldSourceBuilders() {
    return m_interfaceBuilder.getFieldSourceBuilders();
  }

  /**
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addMethodSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder)
   */
  public void addInterfaceMethodSourceBuilder(IMethodSourceBuilder builder) {
    m_interfaceBuilder.addMethodSourceBuilder(builder);
  }

  /**
   * @param sortKey
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addSortedMethodSourceBuilder(org.eclipse.scout.commons.CompositeObject,
   *      org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder)
   */
  public void addInterfaceSortedMethodSourceBuilder(CompositeObject sortKey, IMethodSourceBuilder builder) {
    m_interfaceBuilder.addSortedMethodSourceBuilder(sortKey, builder);
  }

  /**
   * @param builder
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#removeMethodSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder)
   */
  public boolean removeInterfaceMethodSourceBuilder(IMethodSourceBuilder builder) {
    return m_interfaceBuilder.removeMethodSourceBuilder(builder);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getMethodSourceBuilders()
   */
  public List<IMethodSourceBuilder> getInterfaceMethodSourceBuilders() {
    return m_interfaceBuilder.getMethodSourceBuilders();
  }

  public void setImplementationProject(IJavaProject implementationProject) {
    m_implementationProject = implementationProject;
  }

  public IJavaProject getImplementationProject() {
    return m_implementationProject;
  }

  public void setImplementationPackageName(String implementationPackageName) {
    m_implementationPackageName = implementationPackageName;
  }

  public String getImplementationPackageName() {
    return m_implementationPackageName;
  }

  public TypeSourceBuilder getImplementationSourceBuilder() {
    return m_implementationBuilder;
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.AbstractJavaElementSourceBuilder#getElementName()
   */
  public String getImplementationName() {
    return m_implementationBuilder.getElementName();
  }

  /**
   * @param superTypeSignature
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#setSuperTypeSignature(java.lang.String)
   */
  public void setImplementationSuperTypeSignature(String superTypeSignature) {
    m_implementationBuilder.setSuperTypeSignature(superTypeSignature);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getSuperTypeSignature()
   */
  public String getImplementationSuperTypeSignature() {
    return m_implementationBuilder.getSuperTypeSignature();
  }

  /**
   * @param interfaceSignature
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addInterfaceSignature(java.lang.String)
   */
  public void addImplementationInterfaceSignature(String interfaceSignature) {
    m_implementationBuilder.addInterfaceSignature(interfaceSignature);
  }

  /**
   * @param interfaceSignature
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#removeInterfaceSignature(java.lang.String)
   */
  public boolean removeImplementationInterfaceSignature(String interfaceSignature) {
    return m_implementationBuilder.removeInterfaceSignature(interfaceSignature);
  }

  /**
   * @param interfaceSignatures
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#setInterfaceSignatures(java.lang.String[])
   */
  public void setImplementationInterfaceSignatures(String[] interfaceSignatures) {
    m_implementationBuilder.setInterfaceSignatures(interfaceSignatures);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getInterfaceSignatures()
   */
  public List<String> getImplementationInterfaceSignatures() {
    return m_implementationBuilder.getInterfaceSignatures();
  }

  /**
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addFieldSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.field.IFieldSourceBuilder)
   */
  public void addImplementationFieldSourceBuilder(IFieldSourceBuilder builder) {
    m_implementationBuilder.addFieldSourceBuilder(builder);
  }

  /**
   * @param sortKey
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addSortedFieldSourceBuilder(org.eclipse.scout.commons.CompositeObject,
   *      org.eclipse.scout.sdk.sourcebuilder.field.IFieldSourceBuilder)
   */
  public void addImplementationSortedFieldSourceBuilder(CompositeObject sortKey, IFieldSourceBuilder builder) {
    m_implementationBuilder.addSortedFieldSourceBuilder(sortKey, builder);
  }

  /**
   * @param builder
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#removeFieldSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.field.IFieldSourceBuilder)
   */
  public boolean removeImplementationFieldSourceBuilder(IFieldSourceBuilder builder) {
    return m_implementationBuilder.removeFieldSourceBuilder(builder);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getFieldSourceBuilders()
   */
  public List<IFieldSourceBuilder> getImplementationFieldSourceBuilders() {
    return m_implementationBuilder.getFieldSourceBuilders();
  }

  /**
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addMethodSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder)
   */
  public void addImplementationMethodSourceBuilder(IMethodSourceBuilder builder) {
    m_implementationBuilder.addMethodSourceBuilder(builder);
  }

  /**
   * @param sortKey
   * @param builder
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#addSortedMethodSourceBuilder(org.eclipse.scout.commons.CompositeObject,
   *      org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder)
   */
  public void addImplementationSortedMethodSourceBuilder(CompositeObject sortKey, IMethodSourceBuilder builder) {
    m_implementationBuilder.addSortedMethodSourceBuilder(sortKey, builder);
  }

  /**
   * @param builder
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#removeMethodSourceBuilder(org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder)
   */
  public boolean removeImplementationMethodSourceBuilder(IMethodSourceBuilder builder) {
    return m_implementationBuilder.removeMethodSourceBuilder(builder);
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.type.TypeSourceBuilder#getMethodSourceBuilders()
   */
  public List<IMethodSourceBuilder> getImplementationMethodSourceBuilders() {
    return m_implementationBuilder.getMethodSourceBuilders();
  }

  public void addServiceMethodBuilder(ServiceMethod method) {
    getInterfaceSourceBuilder().addMethodSourceBuilder(method.getInterfaceSourceBuilder());
    getImplementationSourceBuilder().addMethodSourceBuilder(method.getImplementationSourceBuilder());
  }

  /**
   * @return the createdServiceInterface
   */
  public IType getCreatedServiceInterface() {
    return m_createdServiceInterface;
  }

  public void setCreatedServiceInterface(IType t) {
    m_createdServiceInterface = t;
  }

  /**
   * @return the createdServiceImplementation
   */
  public IType getCreatedServiceImplementation() {
    return m_createdServiceImplementation;
  }

  public boolean addProxyRegistrationProject(IJavaProject clientProject) {
    return m_proxyRegistrationProjects.add(clientProject);
  }

  public boolean removeProxyRegistrationProject(IJavaProject project) {
    return m_proxyRegistrationProjects.remove(project);
  }

  public void setProxyRegistrationProjects(List<IJavaProject> projects) {
    m_proxyRegistrationProjects.clear();
    m_proxyRegistrationProjects.addAll(projects);
  }

  public List<IJavaProject> getProxyRegistrationProjects() {
    return Collections.unmodifiableList(m_proxyRegistrationProjects);
  }

  public boolean addServiceRegistration(ServiceRegistrationDescription desc) {
    return m_serviceRegistrationDescriptions.add(desc);
  }

  public boolean removeServiceRegistration(ServiceRegistrationDescription desc) {
    return m_serviceRegistrationDescriptions.remove(desc);
  }

  public void setServiceRegistrations(List<ServiceRegistrationDescription> desc) {
    m_serviceRegistrationDescriptions.clear();
    m_serviceRegistrationDescriptions.addAll(desc);
  }

  public List<ServiceRegistrationDescription> getServiceRegistrations() {
    return Collections.unmodifiableList(m_serviceRegistrationDescriptions);
  }
}
