/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.sdk.operation.service;

import java.util.List;

import org.eclipse.jdt.core.Flags;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.sdk.sourcebuilder.ICommentSourceBuilder;
import org.eclipse.scout.sdk.sourcebuilder.annotation.AnnotationSourceBuilderFactory;
import org.eclipse.scout.sdk.sourcebuilder.comment.CommentSourceBuilderFactory;
import org.eclipse.scout.sdk.sourcebuilder.method.IMethodBodySourceBuilder;
import org.eclipse.scout.sdk.sourcebuilder.method.IMethodSourceBuilder;
import org.eclipse.scout.sdk.sourcebuilder.method.MethodBodySourceBuilderFactory;
import org.eclipse.scout.sdk.sourcebuilder.method.MethodSourceBuilder;
import org.eclipse.scout.sdk.util.type.MethodParameter;

/**
 * <h3>{@link ServiceMethod}</h3>
 * 
 * @author Andreas Hoegger
 * @since 3.9.0 09.04.2013
 */
public class ServiceMethod {

  private IMethodSourceBuilder m_interfaceSourceBuilder;
  private IMethodSourceBuilder m_implementationSourceBuilder;

  public ServiceMethod(String methodName, String interfaceFqn) {
    m_interfaceSourceBuilder = new MethodSourceBuilder(methodName);
    m_interfaceSourceBuilder.setCommentSourceBuilder(CommentSourceBuilderFactory.createPreferencesMethodCommentBuilder());
    m_interfaceSourceBuilder.setFlags(Flags.AccInterface);
    m_implementationSourceBuilder = new MethodSourceBuilder(methodName);
    m_implementationSourceBuilder.setFlags(Flags.AccPublic);
    if (StringUtility.hasText(interfaceFqn)) {
      m_implementationSourceBuilder.addAnnotationSourceBuilder(AnnotationSourceBuilderFactory.createOverrideAnnotationSourceBuilder());
    }
    else {
      m_implementationSourceBuilder.setCommentSourceBuilder(CommentSourceBuilderFactory.createPreferencesMethodCommentBuilder());
    }
    m_implementationSourceBuilder.setMethodBodySourceBuilder(MethodBodySourceBuilderFactory.createAutoGeneratedMethodBody());
  }

  IMethodSourceBuilder getInterfaceSourceBuilder() {
    return m_interfaceSourceBuilder;
  }

  IMethodSourceBuilder getImplementationSourceBuilder() {
    return m_implementationSourceBuilder;
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.IJavaElementSourceBuilder#getCommentSourceBuilder()
   */
  public ICommentSourceBuilder getCommentSourceBuilder() {
    return m_interfaceSourceBuilder.getCommentSourceBuilder();
  }

  /**
   * @return
   * @see org.eclipse.scout.sdk.sourcebuilder.IJavaElementSourceBuilder#getElementName()
   */
  public String getElementName() {
    return m_interfaceSourceBuilder.getElementName();
  }

  public void setReturnTypeSignature(String returnTypeSignature) {
    getInterfaceSourceBuilder().setReturnTypeSignature(returnTypeSignature);
    getImplementationSourceBuilder().setReturnTypeSignature(returnTypeSignature);
  }

  public String getReturnTypeSignature() {
    return getInterfaceSourceBuilder().getReturnTypeSignature();
  }

  public List<String> getExceptionSignatures() {
    return m_interfaceSourceBuilder.getExceptionSignatures();
  }

  public List<MethodParameter> getParameters() {
    return m_interfaceSourceBuilder.getParameters();
  }

  public IMethodBodySourceBuilder getMethodBodySourceBuilder() {
    return m_implementationSourceBuilder.getMethodBodySourceBuilder();
  }

  public void setCommentSourceBuilder(ICommentSourceBuilder commentSourceBuilder) {
    m_interfaceSourceBuilder.setCommentSourceBuilder(commentSourceBuilder);
  }

  public void setParameters(List<MethodParameter> parameters) {
    m_interfaceSourceBuilder.setParameters(parameters);
    m_implementationSourceBuilder.setParameters(parameters);
  }

  public boolean addParameter(MethodParameter parameter) {
    m_implementationSourceBuilder.addParameter(parameter);
    return m_interfaceSourceBuilder.addParameter(parameter);
  }

  public boolean removeParameter(MethodParameter parameter) {
    m_implementationSourceBuilder.removeParameter(parameter);
    return m_interfaceSourceBuilder.removeParameter(parameter);
  }

  public void addExceptionSignature(String exceptionSignature) {
    m_implementationSourceBuilder.addExceptionSignature(exceptionSignature);
    m_interfaceSourceBuilder.addExceptionSignature(exceptionSignature);
  }

  public boolean removeExceptionSignature(String exceptionSignature) {
    m_implementationSourceBuilder.removeExceptionSignature(exceptionSignature);
    return m_interfaceSourceBuilder.removeExceptionSignature(exceptionSignature);
  }

  public void setExceptionSignatures(List<String> exceptionSignatures) {
    m_interfaceSourceBuilder.setExceptionSignatures(exceptionSignatures);
    m_implementationSourceBuilder.setExceptionSignatures(exceptionSignatures);
  }

  public void setMethodBodySourceBuilder(IMethodBodySourceBuilder methodBodySourceBuilder) {
    m_implementationSourceBuilder.setMethodBodySourceBuilder(methodBodySourceBuilder);
  }

}
