/*******************************************************************************
 * Copyright (c) 2010, 2016 THALES GLOBAL SERVICES.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.sirius.tests.unit.table.tests.srs;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.sirius.table.business.api.refresh.DTableSynchronizer;
import org.eclipse.sirius.table.business.internal.refresh.DTableSynchronizerImpl;
import org.eclipse.sirius.table.metamodel.table.DTable;
import org.eclipse.sirius.table.metamodel.table.TableFactory;
import org.eclipse.sirius.table.metamodel.table.description.TableDescription;
import org.eclipse.sirius.table.ui.business.api.helper.TableUIHelper;
import org.eclipse.sirius.table.ui.tools.internal.editor.action.CreateRepresentationFromRepresentationCreationDescription;
import org.eclipse.sirius.table.ui.tools.internal.editor.action.HideColumnAction;
import org.eclipse.sirius.table.ui.tools.internal.editor.action.ShowAllColumnsAction;
import org.eclipse.sirius.tests.unit.table.unit.common.TableTestCase;
import org.eclipse.sirius.tools.api.command.ui.NoUICallback;
import org.eclipse.sirius.viewpoint.DRepresentation;
import org.eclipse.sirius.viewpoint.DView;

/**
 * @author <a href="mailto:laurent.redor@obeo.fr">Laurent Redor</a>
 * 
 */
public class ExploitArrays_SimpleTest extends TableTestCase {

    private static final String NEW_ELEMENT_NAME = "NewElementFromTestJUnit";

    /**
     * REQ-155 : Sirius shall, in case of having a hierarchy of Elements to
     * manage, represent the hierarchy in the Array.
     * 
     * This test checks that lines contains sub-lines as expected.
     * 
     * @throws Exception
     *             if the test has problems not handle
     */
    public void test_REQ_155() throws Exception {
        final TableDescription desc = find("Simple hierarchy");
        assertNotNull("Unit test data is not correct", desc);

        final DTableSynchronizer sync = new DTableSynchronizerImpl(desc, accessor, interpreter);

        final DTable newTable = TableFactory.eINSTANCE.createDTable();
        newTable.setDescription(desc);
        newTable.setTarget(semanticModel);
        sync.setTable(newTable);
        sync.refresh(new NullProgressMonitor());
        assertEquals("We have 3 top level classes so we should get 3 lines under tables", 3, newTable.getLines().size());
        assertEquals("We should have one sub-class of the AbstractClass1", 1, newTable.getLines().get(1).getLines().size());
        assertEquals("We should have one sub-class of the AbstractClass2", 1, newTable.getLines().get(2).getLines().size());
        assertEquals("We should have one sub-class of the Class2", 1, newTable.getLines().get(2).getLines().get(0).getLines().size());
    }

    /**
     * REQ-158 : Sirius shall enable to automatically build an Edition Array
     * based on a set of selected ModelElements in the Model corresponding to
     * the Elements.
     * 
     * This test checks that the table creation is possible from a table edition
     * elements.
     * 
     * @throws Exception
     *             if the test has problems not handle
     */
    public void test_REQ_158() throws Exception {
        // We change the userInterfaceCallBack of the table command factory
        getCommandFactory().setUserInterfaceCallBack(new NoUICallback() {
            @Override
            public String askForDetailName(final String defaultName) throws InterruptedException {
                return NEW_ELEMENT_NAME;
            }

            @Override
            public String askForDetailName(String defaultName, String representationDescription) throws InterruptedException {
                return NEW_ELEMENT_NAME;
            }

            @Override
            public String askForDetailName(String defaultName, String representationDescriptionName, String representationDescriptionDoc) throws InterruptedException {
                return NEW_ELEMENT_NAME;
            }
        });

        final TableDescription desc = find("Colored Classes Table");
        assertNotNull("Unit test data is not correct", desc);

        final DTableSynchronizer sync = new DTableSynchronizerImpl(desc, accessor, interpreter);

        final DTable newTable = TableFactory.eINSTANCE.createDTable();
        newTable.setDescription(desc);
        newTable.setTarget(semanticModel);
        sync.setTable(newTable);
        sync.refresh(new NullProgressMonitor());

        final TransactionalEditingDomain domain = session.getTransactionalEditingDomain();
        final TableDescription targetDescription = find("Class and subClass with some attributes");
        final CreateRepresentationFromRepresentationCreationDescription createRepresentationAction = new CreateRepresentationFromRepresentationCreationDescription(
                targetDescription.getOwnedRepresentationCreationDescriptions().get(0), newTable.getLines().get(0), domain, getCommandFactory());
        createRepresentationAction.run();

        boolean representationFound = false;
        for (final DView view : session.getOwnedViews()) {
            for (final DRepresentation representation : view.getOwnedRepresentations()) {
                if (NEW_ELEMENT_NAME.equals(representation.getName())) {
                    representationFound = true;
                }
            }
        }
        assertTrue("We should have a table named " + NEW_ELEMENT_NAME, representationFound);
    }

    /**
     * REQ-159 : Sirius shall enable to mask/unmask attributes.
     * 
     * This test checks that the column can be hide and reveal contains
     * sub-lines as expected.
     * 
     * @throws Exception
     *             if the test has problems not handle
     */
    public void test_REQ_159() throws Exception {
        final TableDescription desc = find("Property");
        assertNotNull("Unit test data is not correct", desc);

        final DTableSynchronizer sync = new DTableSynchronizerImpl(desc, accessor, interpreter);

        final DTable newTable = TableFactory.eINSTANCE.createDTable();
        newTable.setDescription(desc);
        newTable.setTarget(semanticModel);
        sync.setTable(newTable);
        sync.refresh(new NullProgressMonitor());
        assertEquals("We have 7 properties so we should get 7 lines", 7, newTable.getLines().size());
        assertEquals("We have 3 features columns so we should get 3 columns", 3, newTable.getColumns().size());

        final List<List<String>> expected = new ArrayList<List<String>>();
        TableUIHelper.addLineToTable(expected, new String[] { "", "Name", "Lower", "Upper" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property P1", "P1", "1", "1" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property P2", "P2", "1", "1" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property P3", "P3", "1", "1" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property ", "", "1", "1" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property ", "", "1", "1" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property ", "", "1", "1" });
        TableUIHelper.addLineToTable(expected, new String[] { "Property ", "", "1", "1" });

        assertEquals("The original display is KO", TableUIHelper.toHTML(expected), TableUIHelper.toContentHTMl(newTable, false));

        final TransactionalEditingDomain domain = session.getTransactionalEditingDomain();
        final HideColumnAction hideColumnAction = new HideColumnAction(newTable, domain, getCommandFactory());
        hideColumnAction.setColumn(newTable.getColumns().get(0));
        hideColumnAction.run();

        final List<List<String>> expectedWithHiding = new ArrayList<List<String>>();
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "", "Lower", "Upper" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property P1", "1", "1" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property P2", "1", "1" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property P3", "1", "1" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property ", "1", "1" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property ", "1", "1" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property ", "1", "1" });
        TableUIHelper.addLineToTable(expectedWithHiding, new String[] { "Property ", "1", "1" });

        assertEquals("The display after hiding the Name column is KO", TableUIHelper.toHTML(expectedWithHiding), TableUIHelper.toContentHTMl(newTable, false));

        final ShowAllColumnsAction showAllColumnsAction = new ShowAllColumnsAction(newTable, domain, getCommandFactory());
        showAllColumnsAction.run();

        assertEquals("The display after reveal all columns is KO", TableUIHelper.toHTML(expected), TableUIHelper.toContentHTMl(newTable, false));
    }

    /**
     * REQ-162 : Sirius shall enable to automatically build a Cross Array based
     * on a set of selected ModelElements in the Model corresponding to the
     * LineElements and ColumnElements, or to the Relationships.
     * 
     * This test checks that the table creation is possible from a cross table
     * elements.
     * 
     * @throws Exception
     *             if the test has problems not handle
     */
    public void test_REQ_162() throws Exception {
        // We change the userInterfaceCallBack of the table command factory
        getCommandFactory().setUserInterfaceCallBack(new NoUICallback() {

            @Override
            public String askForDetailName(final String defaultName) throws InterruptedException {
                return NEW_ELEMENT_NAME;
            }

            @Override
            public String askForDetailName(String defaultName, String representationDescription) throws InterruptedException {
                return NEW_ELEMENT_NAME;
            }

            @Override
            public String askForDetailName(String defaultName, String representationDescriptionName, String representationDescriptionDoc) throws InterruptedException {
                return NEW_ELEMENT_NAME;
            }
        });

        final TableDescription desc = find("Cross Table Colors");
        assertNotNull("Unit test data is not correct", desc);

        final DTableSynchronizer sync = new DTableSynchronizerImpl(desc, accessor, interpreter);

        final DTable newTable = TableFactory.eINSTANCE.createDTable();
        newTable.setDescription(desc);
        newTable.setTarget(semanticModel);
        sync.setTable(newTable);
        sync.refresh(new NullProgressMonitor());

        final TransactionalEditingDomain domain = session.getTransactionalEditingDomain();
        final TableDescription targetDescription = find("Model all associations");
        CreateRepresentationFromRepresentationCreationDescription createRepresentationAction = new CreateRepresentationFromRepresentationCreationDescription(
                targetDescription.getOwnedRepresentationCreationDescriptions().get(0), newTable.getLines().get(0), domain, getCommandFactory());
        createRepresentationAction.run();

        boolean representationFound = false;
        for (final DView view : session.getOwnedViews()) {
            for (final DRepresentation representation : view.getOwnedRepresentations()) {
                if (NEW_ELEMENT_NAME.equals(representation.getName())) {
                    representationFound = true;
                }
            }
        }
        assertTrue("We should have a table named " + NEW_ELEMENT_NAME, representationFound);
    }
}
