/*******************************************************************************
 * Copyright (c) 2007, 2018 THALES GLOBAL SERVICES.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.sirius.editor.properties.sections.common;

import org.eclipse.emf.common.command.CompoundCommand;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.sirius.editor.properties.ViewpointPropertySheetPage;
import org.eclipse.sirius.editor.utils.TextChangeHelper;
import org.eclipse.sirius.ext.swt.TextChangeListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertyConstants;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;

/**
 * An abstract implementation of a section displaying a text field.
 */
public abstract class AbstractMultilinePropertySection extends AbstractViewpointPropertySection {
    /** Text control of the section. */
    protected Text text;

    /** Label control of the section. */
    protected CLabel nameLabel;

    /** Main composite **/
    protected Composite composite;

    /**
     * @see org.eclipse.ui.views.properties.tabbed.ITabbedPropertySection#createControls(org.eclipse.swt.widgets.Composite,
     *      org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage)
     */
    @Override
    public void createControls(Composite parent, TabbedPropertySheetPage tabbedPropertySheetPage) {
        if (tabbedPropertySheetPage instanceof ViewpointPropertySheetPage) {
            super.createControls(parent, (ViewpointPropertySheetPage) tabbedPropertySheetPage);
        } else {
            super.createControls(parent, tabbedPropertySheetPage);
        }
        composite = getWidgetFactory().createFlatFormComposite(parent);
        FormData data;

        text = getWidgetFactory().createText(composite, "", SWT.MULTI); //$NON-NLS-1$
        data = new FormData();
        data.left = new FormAttachment(0, AbstractViewpointPropertySection.LABEL_WIDTH);
        data.right = new FormAttachment(100, 0);
        data.top = new FormAttachment(0, ITabbedPropertyConstants.VSPACE);
        data.bottom = new FormAttachment(0, 150);
        text.setLayoutData(data);

        nameLabel = getWidgetFactory().createCLabel(composite, getLabelText());
        data = new FormData();
        data.left = new FormAttachment(0, 0);
        data.right = new FormAttachment(text, -ITabbedPropertyConstants.HSPACE - 20);
        data.top = new FormAttachment(text, 0, SWT.TOP);
        nameLabel.setLayoutData(data);

        TextChangeListener listener = new TextChangeHelper(false) {
            @Override
            public void textChanged(Text control) {
                handleTextModified();
            }
        };
        listener.startListeningTo(text);
        listener.startListeningForEnter(text);
    }

    /**
     * @see org.eclipse.ui.views.properties.tabbed.AbstractPropertySection#aboutToBeShown()
     * @Override
     */
    @Override
    public void aboutToBeShown() {
        super.aboutToBeShown();
        PlatformUI.getWorkbench().getHelpSystem().setHelp(composite, "org.eclipse.sirius." + eObject.eClass().getName());
    }

    /**
     * {@inheritDoc}
     *
     * @see org.eclipse.sirius.editor.properties.sections.common.AbstractViewpointPropertySection#setInput(org.eclipse.ui.IWorkbenchPart,
     *      org.eclipse.jface.viewers.ISelection)
     */
    @Override
    public void setInput(IWorkbenchPart part, ISelection selection) {
        super.setInput(part, selection);
        nameLabel.setText(getLabelText());
    }

    /**
     * Handle the text modified event.
     */
    protected void handleTextModified() {
        String newText = text.getText();
        boolean equals = isEqual(newText);

        if (!equals) {
            EditingDomain editingDomain = ((IEditingDomainProvider) getPart()).getEditingDomain();
            Object value = getFeatureValue(newText);
            if (eObjectList.size() == 1) {
                // apply the property change to single selected object
                editingDomain.getCommandStack().execute(SetCommand.create(editingDomain, eObject, getFeature(), value));
            } else {
                CompoundCommand compoundCommand = new CompoundCommand();
                /* apply the property change to all selected elements */
                for (EObject nextObject : eObjectList) {
                    compoundCommand.append(SetCommand.create(editingDomain, nextObject, getFeature(), value));
                }
                editingDomain.getCommandStack().execute(compoundCommand);
            }
        }
    }

    /**
     * @see org.eclipse.ui.views.properties.tabbed.view.ITabbedPropertySection#refresh()
     */
    @Override
    public void refresh() {
        if (getFeatureAsText() != null) {
            text.setText(getFeatureAsText());
        }
    }

    /**
     * Determine if the provided string value is an equal representation of the current setting of the text property.
     *
     * @param newText
     *            the new string value.
     * @return <code>True</code> if the new string value is equal to the current property setting, <code>False</code>
     *         otherwise.
     */
    protected abstract boolean isEqual(String newText);

    /**
     * Get the feature for the text field of this section.
     *
     * @return The feature for the text.
     */
    @Override
    protected abstract EAttribute getFeature();

    /**
     * Get the base description.
     *
     * @return The description for the feature.
     */
    protected abstract String getPropertyDescription();

    protected String getToolTipText() {
        return getPropertyDescription();
    }

    /**
     * Get the value of the feature as text for the text field of the section.
     *
     * @return The value of the feature as text.
     */
    protected String getFeatureAsText() {
        String value = new String();
        if (eObject.eGet(getFeature()) != null) {
            value = eObject.eGet(getFeature()).toString();
        }
        return value;
    }

    /**
     * Get the new value of the feature for the text field of the section.
     *
     * @param newText
     *            The new value of the feature as a string.
     * @return The new value of the feature.
     */
    protected abstract Object getFeatureValue(String newText);

    protected abstract String getDefaultLabelText();

    /**
     * Get the label for the text field of the section.
     *
     * @return The label for the text field.
     */
    protected String getLabelText() {
        if (eObject != null) {
            final EStructuralFeature eFeature = getFeature();
            final IItemPropertyDescriptor propertyDescriptor = getPropertyDescriptor(eFeature);
            if (propertyDescriptor != null) {
                return propertyDescriptor.getDisplayName(eObject);
            }
        }
        return getDefaultLabelText();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void makeReadonly() {
        text.setEnabled(false);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void makeWrittable() {
        text.setEnabled(true);
    }
}
