/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.openweathermap.internal.connection;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.HttpResponseException;
import org.eclipse.jetty.client.api.ContentResponse;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.smarthome.binding.openweathermap.internal.config.OpenWeatherMapAPIConfiguration;
import org.eclipse.smarthome.binding.openweathermap.internal.connection.OpenWeatherMapCommunicationException;
import org.eclipse.smarthome.binding.openweathermap.internal.connection.OpenWeatherMapConfigurationException;
import org.eclipse.smarthome.binding.openweathermap.internal.handler.OpenWeatherMapAPIHandler;
import org.eclipse.smarthome.binding.openweathermap.internal.model.OpenWeatherMapJsonDailyForecastData;
import org.eclipse.smarthome.binding.openweathermap.internal.model.OpenWeatherMapJsonHourlyForecastData;
import org.eclipse.smarthome.binding.openweathermap.internal.model.OpenWeatherMapJsonWeatherData;
import org.eclipse.smarthome.binding.openweathermap.internal.utils.ByteArrayFileCache;
import org.eclipse.smarthome.core.cache.ExpiringCacheMap;
import org.eclipse.smarthome.core.library.types.PointType;
import org.eclipse.smarthome.core.library.types.RawType;
import org.eclipse.smarthome.io.net.http.HttpUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public class OpenWeatherMapConnection {
    private final Logger logger = LoggerFactory.getLogger(OpenWeatherMapConnection.class);
    private static final String PROPERTY_MESSAGE = "message";
    private static final String PNG_CONTENT_TYPE = "image/png";
    private static final String PARAM_APPID = "appid";
    private static final String PARAM_UNITS = "units";
    private static final String PARAM_LAT = "lat";
    private static final String PARAM_LON = "lon";
    private static final String PARAM_LANG = "lang";
    private static final String PARAM_FORECAST_CNT = "cnt";
    private static final String WEATHER_URL = "https://api.openweathermap.org/data/2.5/weather";
    private static final String THREE_HOUR_FORECAST_URL = "https://api.openweathermap.org/data/2.5/forecast";
    private static final String DAILY_FORECAST_URL = "https://api.openweathermap.org/data/2.5/forecast/daily";
    private static final String ICON_URL = "https://openweathermap.org/img/w/%s.png";
    private final OpenWeatherMapAPIHandler handler;
    private final HttpClient httpClient;
    private static final ByteArrayFileCache IMAGE_CACHE = new ByteArrayFileCache("org.eclipse.smarthome.binding.openweathermap");
    private final ExpiringCacheMap<String, String> cache;
    private final JsonParser parser = new JsonParser();
    private final Gson gson = new Gson();

    public OpenWeatherMapConnection(OpenWeatherMapAPIHandler handler, HttpClient httpClient) {
        this.handler = handler;
        this.httpClient = httpClient;
        OpenWeatherMapAPIConfiguration config = handler.getOpenWeatherMapAPIConfig();
        this.cache = new ExpiringCacheMap(TimeUnit.MINUTES.toMillis(config.getRefreshInterval()));
    }

    public synchronized @Nullable OpenWeatherMapJsonWeatherData getWeatherData(@Nullable PointType location) throws JsonSyntaxException, OpenWeatherMapCommunicationException, OpenWeatherMapConfigurationException {
        return (OpenWeatherMapJsonWeatherData)this.gson.fromJson(this.getResponseFromCache(this.buildURL(WEATHER_URL, this.getRequestParams(this.handler.getOpenWeatherMapAPIConfig(), location))), OpenWeatherMapJsonWeatherData.class);
    }

    public synchronized @Nullable OpenWeatherMapJsonHourlyForecastData getHourlyForecastData(@Nullable PointType location, int count) throws JsonSyntaxException, OpenWeatherMapCommunicationException, OpenWeatherMapConfigurationException {
        if (count <= 0) {
            throw new OpenWeatherMapConfigurationException("@text/offline.conf-error-not-supported-number-of-hours");
        }
        Map<String, String> params = this.getRequestParams(this.handler.getOpenWeatherMapAPIConfig(), location);
        params.put(PARAM_FORECAST_CNT, Integer.toString(count));
        return (OpenWeatherMapJsonHourlyForecastData)this.gson.fromJson(this.getResponseFromCache(this.buildURL(THREE_HOUR_FORECAST_URL, params)), OpenWeatherMapJsonHourlyForecastData.class);
    }

    public synchronized @Nullable OpenWeatherMapJsonDailyForecastData getDailyForecastData(@Nullable PointType location, int count) throws JsonSyntaxException, OpenWeatherMapCommunicationException, OpenWeatherMapConfigurationException {
        if (count <= 0) {
            throw new OpenWeatherMapConfigurationException("@text/offline.conf-error-not-supported-number-of-days");
        }
        Map<String, String> params = this.getRequestParams(this.handler.getOpenWeatherMapAPIConfig(), location);
        params.put(PARAM_FORECAST_CNT, Integer.toString(count));
        return (OpenWeatherMapJsonDailyForecastData)this.gson.fromJson(this.getResponseFromCache(this.buildURL(DAILY_FORECAST_URL, params)), OpenWeatherMapJsonDailyForecastData.class);
    }

    public static @Nullable RawType getWeatherIcon(String iconId) {
        if (StringUtils.isEmpty((String)iconId)) {
            throw new IllegalArgumentException("Cannot download weather icon as icon id is null.");
        }
        return OpenWeatherMapConnection.downloadWeatherIconFromCache(String.format(ICON_URL, iconId));
    }

    private static @Nullable RawType downloadWeatherIconFromCache(String url) {
        if (IMAGE_CACHE.containsKey(url)) {
            return new RawType(IMAGE_CACHE.get(url), PNG_CONTENT_TYPE);
        }
        RawType image = OpenWeatherMapConnection.downloadWeatherIcon(url);
        if (image != null) {
            IMAGE_CACHE.put(url, image.getBytes());
            return image;
        }
        return null;
    }

    private static @Nullable RawType downloadWeatherIcon(String url) {
        return HttpUtil.downloadImage((String)url);
    }

    private Map<String, String> getRequestParams(OpenWeatherMapAPIConfiguration config, @Nullable PointType location) {
        if (location == null) {
            throw new OpenWeatherMapConfigurationException("@text/offline.conf-error-missing-location");
        }
        HashMap<String, String> params = new HashMap<String, String>();
        params.put(PARAM_APPID, StringUtils.trimToEmpty((String)config.getApikey()));
        params.put(PARAM_UNITS, "metric");
        params.put(PARAM_LAT, location.getLatitude().toString());
        params.put(PARAM_LON, location.getLongitude().toString());
        String language = StringUtils.trimToEmpty((String)config.getLanguage());
        if (!language.isEmpty()) {
            params.put(PARAM_LANG, language.toLowerCase());
        }
        return params;
    }

    private String buildURL(String url, Map<String, String> requestParams) {
        return requestParams.keySet().stream().map(key -> String.valueOf(key) + "=" + this.encodeParam((String)requestParams.get(key))).collect(Collectors.joining("&", String.valueOf(url) + "?", ""));
    }

    private String encodeParam(String value) {
        try {
            return URLEncoder.encode(value, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            this.logger.debug("UnsupportedEncodingException occurred during execution: {}", (Object)e.getLocalizedMessage(), (Object)e);
            return "";
        }
    }

    private @Nullable String getResponseFromCache(String url) {
        return (String)this.cache.putIfAbsentAndGet((Object)url, () -> this.getResponse(url));
    }

    private String getResponse(String url) {
        try {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("OpenWeatherMap request: URL = '{}'", (Object)this.uglifyApikey(url));
            }
            ContentResponse contentResponse = this.httpClient.newRequest(url).method(HttpMethod.GET).timeout(10L, TimeUnit.SECONDS).send();
            int httpStatus = contentResponse.getStatus();
            String content = contentResponse.getContentAsString();
            String errorMessage = "";
            this.logger.trace("OpenWeatherMap response: status = {}, content = '{}'", (Object)httpStatus, (Object)content);
            switch (httpStatus) {
                case 200: {
                    return content;
                }
                case 400: 
                case 401: 
                case 404: {
                    errorMessage = this.getErrorMessage(content);
                    this.logger.debug("OpenWeatherMap server responded with status code {}: {}", (Object)httpStatus, (Object)errorMessage);
                    throw new OpenWeatherMapConfigurationException(errorMessage);
                }
            }
            errorMessage = this.getErrorMessage(content);
            this.logger.debug("OpenWeatherMap server responded with status code {}: {}", (Object)httpStatus, (Object)errorMessage);
            throw new OpenWeatherMapCommunicationException(errorMessage);
        }
        catch (ExecutionException e) {
            String errorMessage = e.getLocalizedMessage();
            this.logger.trace("Exception occurred during execution: {}", (Object)errorMessage, (Object)e);
            if (e.getCause() instanceof HttpResponseException) {
                this.logger.debug("OpenWeatherMap server responded with status code {}: Invalid API key.", (Object)401);
                throw new OpenWeatherMapConfigurationException("@text/offline.conf-error-invalid-apikey", e.getCause());
            }
            throw new OpenWeatherMapCommunicationException(errorMessage, e.getCause());
        }
        catch (InterruptedException | TimeoutException e) {
            this.logger.debug("Exception occurred during execution: {}", (Object)e.getLocalizedMessage(), (Object)e);
            throw new OpenWeatherMapCommunicationException(e.getLocalizedMessage(), e.getCause());
        }
    }

    private String uglifyApikey(String url) {
        return url.replaceAll("(appid=)+\\w+", "appid=*****");
    }

    private String getErrorMessage(String response) {
        JsonObject jsonResponse = this.parser.parse(response).getAsJsonObject();
        if (jsonResponse.has(PROPERTY_MESSAGE)) {
            return jsonResponse.get(PROPERTY_MESSAGE).getAsString();
        }
        return response;
    }
}

