/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.core.thing.firmware;

import com.google.common.base.Preconditions;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.eclipse.smarthome.core.common.SafeMethodCaller;
import org.eclipse.smarthome.core.common.ThreadPoolManager;
import org.eclipse.smarthome.core.events.Event;
import org.eclipse.smarthome.core.events.EventPublisher;
import org.eclipse.smarthome.core.i18n.I18nProvider;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingUID;
import org.eclipse.smarthome.core.thing.binding.firmware.Firmware;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUID;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUpdateBackgroundTransferHandler;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUpdateHandler;
import org.eclipse.smarthome.core.thing.firmware.FirmwareEventFactory;
import org.eclipse.smarthome.core.thing.firmware.FirmwareRegistry;
import org.eclipse.smarthome.core.thing.firmware.FirmwareStatus;
import org.eclipse.smarthome.core.thing.firmware.FirmwareStatusInfo;
import org.eclipse.smarthome.core.thing.firmware.ProgressCallbackImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FirmwareUpdateService {
    private final Logger logger = LoggerFactory.getLogger(FirmwareUpdateService.class);
    private final ScheduledExecutorService scheduler = ThreadPoolManager.getScheduledPool((String)FirmwareUpdateService.class.getSimpleName());
    private TimeUnit firmwareStatusInfoJobTimeUnit = TimeUnit.HOURS;
    private final int firmwareStatusInfoJobPeriod = 1;
    private ScheduledFuture<?> firmwareStatusInfoJob;
    private int timeout = 1800000;
    private final Map<ThingUID, FirmwareStatusInfo> firmwareStatusInfoMap = new HashMap<ThingUID, FirmwareStatusInfo>();
    private final List<FirmwareUpdateHandler> firmwareUpdateHandlers = new CopyOnWriteArrayList<FirmwareUpdateHandler>();
    private FirmwareRegistry firmwareRegistry;
    private EventPublisher eventPublisher;
    private I18nProvider i18nProvider;
    private Runnable firmwareStatusRunnable = new Runnable(){

        @Override
        public void run() {
            FirmwareUpdateService.this.logger.debug("Running firmware status check.");
            for (FirmwareUpdateHandler firmwareUpdateHandler : FirmwareUpdateService.this.firmwareUpdateHandlers) {
                try {
                    FirmwareUpdateService.this.logger.debug("Executing firmware status check for thing with UID {}.", (Object)firmwareUpdateHandler.getThing().getUID());
                    Firmware latestFirmware = FirmwareUpdateService.this.firmwareRegistry.getLatestFirmware(firmwareUpdateHandler.getThing().getThingTypeUID());
                    FirmwareStatusInfo newFirmwareStatusInfo = FirmwareUpdateService.this.getFirmwareStatusInfo(firmwareUpdateHandler, latestFirmware);
                    FirmwareUpdateService.this.processFirmwareStatusInfo(firmwareUpdateHandler, newFirmwareStatusInfo, latestFirmware);
                }
                catch (Exception e) {
                    FirmwareUpdateService.this.logger.debug("Exception occurred during firmware status check.", (Throwable)e);
                }
            }
        }
    };

    protected void activate() {
        if (this.firmwareStatusInfoJob == null || this.firmwareStatusInfoJob.isCancelled()) {
            this.firmwareStatusInfoJob = this.scheduler.scheduleAtFixedRate(this.firmwareStatusRunnable, 0L, 1L, this.firmwareStatusInfoJobTimeUnit);
        }
    }

    protected void deactivate() {
        if (this.firmwareStatusInfoJob != null && !this.firmwareStatusInfoJob.isCancelled()) {
            this.firmwareStatusInfoJob.cancel(true);
            this.firmwareStatusInfoJob = null;
        }
    }

    public FirmwareStatusInfo getFirmwareStatusInfo(ThingUID thingUID) {
        Preconditions.checkNotNull((Object)thingUID, (Object)"Thing UID must not be null.");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            this.logger.debug("No firmware update handler available for thing with UID {}.", (Object)thingUID);
            return null;
        }
        Firmware latestFirmware = this.firmwareRegistry.getLatestFirmware(firmwareUpdateHandler.getThing().getThingTypeUID());
        FirmwareStatusInfo firmwareStatusInfo = this.getFirmwareStatusInfo(firmwareUpdateHandler, latestFirmware);
        this.processFirmwareStatusInfo(firmwareUpdateHandler, firmwareStatusInfo, latestFirmware);
        return firmwareStatusInfo;
    }

    public void updateFirmware(final ThingUID thingUID, final FirmwareUID firmwareUID, final Locale locale) {
        Preconditions.checkNotNull((Object)thingUID, (Object)"Thing UID must not be null.");
        Preconditions.checkNotNull((Object)firmwareUID, (Object)"Firmware UID must not be null.");
        final FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            throw new IllegalStateException(String.format("There is no firmware update handler for thing with UID %s.", thingUID));
        }
        final Firmware firmware = this.getFirmware(firmwareUID);
        this.validateFirmwareUpdateConditions(firmware, firmwareUpdateHandler);
        this.logger.debug("Starting firmware update for thing with UID {} and firmware with UID {}", (Object)thingUID, (Object)firmwareUID);
        this.scheduler.submit(new Runnable(){

            @Override
            public void run() {
                final ProgressCallbackImpl progressCallback = new ProgressCallbackImpl(firmwareUpdateHandler, FirmwareUpdateService.this.eventPublisher, FirmwareUpdateService.this.i18nProvider, thingUID, firmwareUID, locale != null ? locale : Locale.getDefault());
                try {
                    SafeMethodCaller.call((SafeMethodCaller.ActionWithException)new SafeMethodCaller.ActionWithException<Void>(){

                        public Void call() {
                            firmwareUpdateHandler.updateFirmware(firmware, progressCallback);
                            return null;
                        }
                    }, (int)FirmwareUpdateService.this.timeout);
                }
                catch (ExecutionException e) {
                    FirmwareUpdateService.this.logger.error(String.format("Unexpected exception occurred for firmware update of thing with UID %s and firmware with UID %s.", thingUID, firmwareUID), e.getCause());
                    progressCallback.failedInternal("unexpected-handler-error");
                }
                catch (TimeoutException e) {
                    FirmwareUpdateService.this.logger.error(String.format("Timeout occurred for firmware update of thing with UID %s and firmware with UID %s.", thingUID, firmwareUID), (Throwable)e);
                    progressCallback.failedInternal("timeout-error");
                }
            }
        });
    }

    private FirmwareStatusInfo getFirmwareStatusInfo(FirmwareUpdateHandler firmwareUpdateHandler, Firmware latestFirmware) {
        String thingFirmwareVersion = this.getThingFirmwareVersion(firmwareUpdateHandler);
        if (latestFirmware == null || thingFirmwareVersion == null) {
            return FirmwareStatusInfo.createUnknownInfo();
        }
        if (latestFirmware.isSuccessorVersion(thingFirmwareVersion)) {
            if (firmwareUpdateHandler.isUpdateExecutable()) {
                return FirmwareStatusInfo.createUpdateExecutableInfo(latestFirmware.getUID());
            }
            return FirmwareStatusInfo.createUpdateAvailableInfo();
        }
        return FirmwareStatusInfo.createUpToDateInfo();
    }

    private synchronized void processFirmwareStatusInfo(FirmwareUpdateHandler firmwareUpdateHandler, FirmwareStatusInfo newFirmwareStatusInfo, Firmware latestFirmware) {
        ThingUID thingUID = firmwareUpdateHandler.getThing().getUID();
        FirmwareStatusInfo previousFirmwareStatusInfo = this.firmwareStatusInfoMap.put(thingUID, newFirmwareStatusInfo);
        if (previousFirmwareStatusInfo == null || !previousFirmwareStatusInfo.equals(newFirmwareStatusInfo)) {
            this.eventPublisher.post((Event)FirmwareEventFactory.createFirmwareStatusInfoEvent(newFirmwareStatusInfo, thingUID));
            if (newFirmwareStatusInfo.getFirmwareStatus() == FirmwareStatus.UPDATE_AVAILABLE && firmwareUpdateHandler instanceof FirmwareUpdateBackgroundTransferHandler && !firmwareUpdateHandler.isUpdateExecutable()) {
                this.transferLatestFirmware((FirmwareUpdateBackgroundTransferHandler)firmwareUpdateHandler, latestFirmware, previousFirmwareStatusInfo);
            }
        }
    }

    private void transferLatestFirmware(final FirmwareUpdateBackgroundTransferHandler fubtHandler, final Firmware latestFirmware, final FirmwareStatusInfo previousFirmwareStatusInfo) {
        this.scheduler.submit(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                try {
                    fubtHandler.transferFirmware(latestFirmware);
                }
                catch (Exception e) {
                    FirmwareUpdateService.this.logger.error("Exception occurred during background firmware transfer.", (Throwable)e);
                    3 var2_2 = this;
                    synchronized (var2_2) {
                        FirmwareUpdateService.this.firmwareStatusInfoMap.put(fubtHandler.getThing().getUID(), previousFirmwareStatusInfo);
                    }
                }
            }
        });
    }

    private void validateFirmwareUpdateConditions(Firmware firmware, FirmwareUpdateHandler firmwareUpdateHandler) {
        if (!firmwareUpdateHandler.isUpdateExecutable()) {
            throw new IllegalStateException(String.format("The firmware update of thing with UID %s is not executable.", firmwareUpdateHandler.getThing().getUID()));
        }
        this.validateFirmwareSuitability(firmware, firmwareUpdateHandler);
    }

    private void validateFirmwareSuitability(Firmware firmware, FirmwareUpdateHandler firmwareUpdateHandler) {
        Thing thing = firmwareUpdateHandler.getThing();
        if (!firmware.getUID().getThingTypeUID().equals(thing.getThingTypeUID())) {
            throw new IllegalArgumentException(String.format("Firmware with UID %s is not suitable for thing with UID %s.", firmware.getUID(), thing.getUID()));
        }
        String firmwareVersion = this.getThingFirmwareVersion(firmwareUpdateHandler);
        if (firmware.getPrerequisiteVersion() != null && !firmware.isPrerequisteVersion(firmwareVersion)) {
            throw new IllegalArgumentException(String.format("Firmware with UID %s requires at least firmware version %s to get installed. But the current firmware version of the thing with UID %s is %s.", firmware.getUID(), firmware.getPrerequisiteVersion(), thing.getUID(), firmwareVersion));
        }
    }

    private Firmware getFirmware(FirmwareUID firmwareUID) {
        Firmware firmware = this.firmwareRegistry.getFirmware(firmwareUID);
        if (firmware == null) {
            throw new IllegalArgumentException(String.format("Firmware with UID %s was not found.", firmwareUID));
        }
        return firmware;
    }

    private FirmwareUpdateHandler getFirmwareUpdateHandler(ThingUID thingUID) {
        for (FirmwareUpdateHandler firmwareUpdateHandler : this.firmwareUpdateHandlers) {
            if (!thingUID.equals(firmwareUpdateHandler.getThing().getUID())) continue;
            return firmwareUpdateHandler;
        }
        return null;
    }

    private String getThingFirmwareVersion(FirmwareUpdateHandler firmwareUpdateHandler) {
        return firmwareUpdateHandler.getThing().getProperties().get("firmwareVersion");
    }

    protected void addFirmwareUpdateHandler(FirmwareUpdateHandler firmwareUpdateHandler) {
        this.firmwareUpdateHandlers.add(firmwareUpdateHandler);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void removeFirmwareUpdateHandler(FirmwareUpdateHandler firmwareUpdateHandler) {
        FirmwareUpdateService firmwareUpdateService = this;
        synchronized (firmwareUpdateService) {
            this.firmwareStatusInfoMap.remove(firmwareUpdateHandler.getThing().getUID());
        }
        this.firmwareUpdateHandlers.remove(firmwareUpdateHandler);
    }

    protected void setFirmwareRegistry(FirmwareRegistry firmwareRegistry) {
        this.firmwareRegistry = firmwareRegistry;
    }

    protected void unsetFirmwareRegistry(FirmwareRegistry firmwareRegistry) {
        this.firmwareRegistry = null;
    }

    protected void setEventPublisher(EventPublisher eventPublisher) {
        this.eventPublisher = eventPublisher;
    }

    protected void unsetEventPublisher(EventPublisher eventPublisher) {
        this.eventPublisher = null;
    }

    protected void setI18nProvider(I18nProvider i18nProvider) {
        this.i18nProvider = i18nProvider;
    }

    protected void unsetI18nProvider(I18nProvider i18nProvider) {
        this.i18nProvider = null;
    }
}

