package org.eclipse.smarthome.model.script.jvmmodel;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.smarthome.core.items.Item;
import org.eclipse.smarthome.core.items.ItemRegistry;
import org.eclipse.smarthome.core.types.Type;
import org.eclipse.smarthome.model.script.engine.IItemRegistryProvider;
import org.eclipse.smarthome.model.script.scoping.StateAndCommandProvider;
import org.eclipse.smarthome.model.script.script.Script;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.jvmmodel.AbstractModelInferrer;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 * 
 * @author Oliver Libutzki - Xtext 2.5.0 migration
 */
@SuppressWarnings("all")
public class ScriptJvmModelInferrer extends AbstractModelInferrer {
  private final static Logger logger = LoggerFactory.getLogger(ScriptJvmModelInferrer.class);
  
  /**
   * conveninence API to build and initialize JvmTypes and their members.
   */
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;
  
  @Inject
  private IItemRegistryProvider itemRegistryProvider;
  
  @Inject
  private StateAndCommandProvider stateAndCommandProvider;
  
  /**
   * Is called for each instance of the first argument's type contained in a resource.
   * 
   * @param element - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if iPrelinkingPhase is <code>true</code>
   */
  protected void _infer(final Script script, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    Resource _eResource = script.eResource();
    URI _uRI = _eResource.getURI();
    String _lastSegment = _uRI.lastSegment();
    String[] _split = _lastSegment.split("\\.");
    String _head = IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(_split)));
    String _firstUpper = StringExtensions.toFirstUpper(_head);
    final String className = (_firstUpper + "Script");
    JvmGenericType _class = this._jvmTypesBuilder.toClass(script, className);
    IJvmDeclaredTypeAcceptor.IPostIndexingInitializing<JvmGenericType> _accept = acceptor.<JvmGenericType>accept(_class);
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        final Set<String> fieldNames = CollectionLiterals.<String>newHashSet();
        final Iterable<Type> types = ScriptJvmModelInferrer.this.stateAndCommandProvider.getAllTypes();
        final Procedure1<Type> _function = new Procedure1<Type>() {
          @Override
          public void apply(final Type type) {
            final String name = type.toString();
            boolean _add = fieldNames.add(name);
            if (_add) {
              EList<JvmMember> _members = it.getMembers();
              Class<? extends Type> _class = type.getClass();
              JvmTypeReference _newTypeRef = ScriptJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(script, _class);
              final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
                @Override
                public void apply(final JvmField it) {
                  it.setStatic(true);
                }
              };
              JvmField _field = ScriptJvmModelInferrer.this._jvmTypesBuilder.toField(script, name, _newTypeRef, _function);
              ScriptJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
            } else {
              Class<? extends Type> _class_1 = type.getClass();
              String _name = _class_1.getName();
              ScriptJvmModelInferrer.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", name, _name);
            }
          }
        };
        IterableExtensions.<Type>forEach(types, _function);
        final ItemRegistry itemRegistry = ScriptJvmModelInferrer.this.itemRegistryProvider.get();
        Collection<Item> _items = null;
        if (itemRegistry!=null) {
          _items=itemRegistry.getItems();
        }
        if (_items!=null) {
          final Procedure1<Item> _function_1 = new Procedure1<Item>() {
            @Override
            public void apply(final Item item) {
              final String name = item.getName();
              boolean _add = fieldNames.add(name);
              if (_add) {
                EList<JvmMember> _members = it.getMembers();
                String _name = item.getName();
                Class<? extends Item> _class = item.getClass();
                JvmTypeReference _newTypeRef = ScriptJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(script, _class);
                final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
                  @Override
                  public void apply(final JvmField it) {
                    it.setStatic(true);
                  }
                };
                JvmField _field = ScriptJvmModelInferrer.this._jvmTypesBuilder.toField(script, _name, _newTypeRef, _function);
                ScriptJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
              } else {
                String _name_1 = item.getName();
                Class<? extends Item> _class_1 = item.getClass();
                String _name_2 = _class_1.getName();
                ScriptJvmModelInferrer.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", _name_1, _name_2);
              }
            }
          };
          IterableExtensions.<Item>forEach(_items, _function_1);
        }
        EList<JvmMember> _members = it.getMembers();
        final Procedure1<JvmOperation> _function_2 = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            it.setStatic(true);
            ScriptJvmModelInferrer.this._jvmTypesBuilder.setBody(it, script);
          }
        };
        JvmOperation _method = ScriptJvmModelInferrer.this._jvmTypesBuilder.toMethod(script, "_script", null, _function_2);
        ScriptJvmModelInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
      }
    };
    _accept.initializeLater(_function);
  }
  
  public void infer(final EObject script, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (script instanceof Script) {
      _infer((Script)script, acceptor, isPreIndexingPhase);
      return;
    } else if (script != null) {
      _infer(script, acceptor, isPreIndexingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(script, acceptor, isPreIndexingPhase).toString());
    }
  }
}
