/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.hue.handler;

import java.util.AbstractMap;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.smarthome.binding.hue.HueBindingConstants;
import org.eclipse.smarthome.binding.hue.handler.HueBridgeHandler;
import org.eclipse.smarthome.binding.hue.handler.LightStateConverter;
import org.eclipse.smarthome.binding.hue.handler.LightStatusListener;
import org.eclipse.smarthome.binding.hue.internal.FullLight;
import org.eclipse.smarthome.binding.hue.internal.HueBridge;
import org.eclipse.smarthome.binding.hue.internal.State;
import org.eclipse.smarthome.binding.hue.internal.StateUpdate;
import org.eclipse.smarthome.core.library.types.HSBType;
import org.eclipse.smarthome.core.library.types.IncreaseDecreaseType;
import org.eclipse.smarthome.core.library.types.OnOffType;
import org.eclipse.smarthome.core.library.types.PercentType;
import org.eclipse.smarthome.core.library.types.StringType;
import org.eclipse.smarthome.core.thing.Bridge;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.ThingStatusDetail;
import org.eclipse.smarthome.core.thing.ThingTypeUID;
import org.eclipse.smarthome.core.thing.binding.BaseThingHandler;
import org.eclipse.smarthome.core.thing.binding.BridgeHandler;
import org.eclipse.smarthome.core.types.Command;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public class HueLightHandler
extends BaseThingHandler
implements LightStatusListener {
    public static final Set<ThingTypeUID> SUPPORTED_THING_TYPES = Stream.of(HueBindingConstants.THING_TYPE_COLOR_LIGHT, HueBindingConstants.THING_TYPE_COLOR_TEMPERATURE_LIGHT, HueBindingConstants.THING_TYPE_DIMMABLE_LIGHT, HueBindingConstants.THING_TYPE_EXTENDED_COLOR_LIGHT, HueBindingConstants.THING_TYPE_ON_OFF_LIGHT, HueBindingConstants.THING_TYPE_ON_OFF_PLUG, HueBindingConstants.THING_TYPE_DIMMABLE_PLUG).collect(Collectors.toSet());
    private static final Map<String, List<String>> VENDOR_MODEL_MAP = Stream.of(new AbstractMap.SimpleEntry<String, List<String>>("Philips", Arrays.asList("LCT001", "LCT002", "LCT003", "LCT007", "LLC001", "LLC006", "LLC007", "LLC010", "LLC011", "LLC012", "LLC013", "LLC020", "LST001", "LST002", "LWB004", "LWB006", "LWB007", "LWL001")), new AbstractMap.SimpleEntry<String, List<String>>("OSRAM", Arrays.asList("Classic_A60_RGBW", "PAR16_50_TW", "Surface_Light_TW", "Plug_01"))).collect(Collectors.toMap(e -> (String)e.getKey(), e -> (List)e.getValue()));
    private static final String OSRAM_PAR16_50_TW_MODEL_ID = "PAR16_50_TW";
    public static final String NORMALIZE_ID_REGEX = "[^a-zA-Z0-9_]";
    @NonNullByDefault(value={})
    private String lightId;
    private @Nullable Integer lastSentColorTemp;
    private @Nullable Integer lastSentBrightness;
    private final Logger logger = LoggerFactory.getLogger(HueLightHandler.class);
    private boolean isOsramPar16 = false;
    private boolean propertiesInitializedSuccessfully = false;
    private @Nullable HueBridgeHandler bridgeHandler;
    @Nullable ScheduledFuture<?> scheduledFuture;

    public HueLightHandler(Thing hueLight) {
        super(hueLight);
    }

    public void initialize() {
        this.logger.debug("Initializing hue light handler.");
        Bridge bridge = this.getBridge();
        this.initializeThing(bridge == null ? null : bridge.getStatus());
    }

    private void initializeThing(@Nullable ThingStatus bridgeStatus) {
        this.logger.debug("initializeThing thing {} bridge status {}", (Object)this.getThing().getUID(), (Object)bridgeStatus);
        String configLightId = (String)this.getConfig().get("lightId");
        if (configLightId != null) {
            this.lightId = configLightId;
            if (this.getHueBridgeHandler() != null) {
                if (bridgeStatus == ThingStatus.ONLINE) {
                    this.initializeProperties();
                    this.updateStatus(ThingStatus.ONLINE);
                } else {
                    this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.BRIDGE_OFFLINE);
                }
            } else {
                this.updateStatus(ThingStatus.OFFLINE);
            }
        } else {
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, "@text/offline.conf-error-no-light-id");
        }
    }

    private synchronized void initializeProperties() {
        FullLight fullLight;
        if (!this.propertiesInitializedSuccessfully && (fullLight = this.getLight()) != null) {
            String modelId = fullLight.getModelID().replaceAll(NORMALIZE_ID_REGEX, "_");
            this.updateProperty("modelId", modelId);
            this.updateProperty("firmwareVersion", fullLight.getSoftwareVersion());
            String vendor = this.getVendor(modelId);
            if (vendor != null) {
                this.updateProperty("vendor", vendor);
            }
            this.updateProperty("uniqueId", fullLight.getUniqueID());
            this.isOsramPar16 = OSRAM_PAR16_50_TW_MODEL_ID.equals(modelId);
            this.updateThing(this.thing);
            this.propertiesInitializedSuccessfully = true;
        }
    }

    private @Nullable String getVendor(String modelId) {
        for (String vendor : VENDOR_MODEL_MAP.keySet()) {
            if (!VENDOR_MODEL_MAP.get(vendor).contains(modelId)) continue;
            return vendor;
        }
        return null;
    }

    public void dispose() {
        this.logger.debug("Handler disposes. Unregistering listener.");
        if (this.lightId != null) {
            HueBridgeHandler bridgeHandler = this.getHueBridgeHandler();
            if (bridgeHandler != null) {
                bridgeHandler.unregisterLightStatusListener(this);
                this.bridgeHandler = null;
            }
            this.lightId = null;
        }
    }

    private @Nullable FullLight getLight() {
        HueBridgeHandler bridgeHandler = this.getHueBridgeHandler();
        if (bridgeHandler != null) {
            return bridgeHandler.getLightById(this.lightId);
        }
        return null;
    }

    public void handleCommand(ChannelUID channelUID, Command command) {
        HueBridgeHandler hueBridge = this.getHueBridgeHandler();
        if (hueBridge == null) {
            this.logger.warn("hue bridge handler not found. Cannot handle command without bridge.");
            return;
        }
        FullLight light = this.getLight();
        if (light == null) {
            this.logger.debug("hue light not known on bridge. Cannot handle command.");
            return;
        }
        StateUpdate lightState = null;
        switch (channelUID.getId()) {
            case "color_temperature": {
                if (command instanceof PercentType) {
                    lightState = LightStateConverter.toColorTemperatureLightState((PercentType)command);
                    break;
                }
                if (command instanceof OnOffType) {
                    lightState = LightStateConverter.toOnOffLightState((OnOffType)command);
                    if (!this.isOsramPar16) break;
                    lightState = this.addOsramSpecificCommands(lightState, (OnOffType)command);
                    break;
                }
                if (!(command instanceof IncreaseDecreaseType)) break;
                lightState = this.convertColorTempChangeToStateUpdate((IncreaseDecreaseType)command, light);
                break;
            }
            case "brightness": {
                if (command instanceof PercentType) {
                    lightState = LightStateConverter.toBrightnessLightState((PercentType)command);
                    break;
                }
                if (command instanceof OnOffType) {
                    lightState = LightStateConverter.toOnOffLightState((OnOffType)command);
                    if (!this.isOsramPar16) break;
                    lightState = this.addOsramSpecificCommands(lightState, (OnOffType)command);
                    break;
                }
                if (!(command instanceof IncreaseDecreaseType)) break;
                lightState = this.convertBrightnessChangeToStateUpdate((IncreaseDecreaseType)command, light);
                break;
            }
            case "switch": {
                this.logger.trace("CHANNEL_SWITCH handling command {}", (Object)command);
                if (!(command instanceof OnOffType)) break;
                lightState = LightStateConverter.toOnOffLightState((OnOffType)command);
                if (!this.isOsramPar16) break;
                lightState = this.addOsramSpecificCommands(lightState, (OnOffType)command);
                break;
            }
            case "color": {
                if (command instanceof HSBType) {
                    HSBType hsbCommand = (HSBType)command;
                    if (hsbCommand.getBrightness().intValue() == 0) {
                        lightState = LightStateConverter.toOnOffLightState(OnOffType.OFF);
                        break;
                    }
                    lightState = LightStateConverter.toColorLightState(hsbCommand);
                    break;
                }
                if (command instanceof PercentType) {
                    lightState = LightStateConverter.toBrightnessLightState((PercentType)command);
                    break;
                }
                if (command instanceof OnOffType) {
                    lightState = LightStateConverter.toOnOffLightState((OnOffType)command);
                    break;
                }
                if (!(command instanceof IncreaseDecreaseType)) break;
                lightState = this.convertBrightnessChangeToStateUpdate((IncreaseDecreaseType)command, light);
                break;
            }
            case "alert": {
                if (!(command instanceof StringType)) break;
                lightState = LightStateConverter.toAlertState((StringType)command);
                if (lightState == null) {
                    this.logger.warn("Unsupported String command: {}. Supported commands are: {}, {}, {} ", new Object[]{command, "NONE", "SELECT", "LSELECT"});
                    return;
                }
                this.scheduleAlertStateRestore(command);
                break;
            }
            case "effect": {
                if (!(command instanceof OnOffType)) break;
                lightState = LightStateConverter.toOnOffEffectState((OnOffType)command);
            }
        }
        if (lightState != null) {
            hueBridge.updateLightState(light, lightState);
        } else {
            this.logger.warn("Command sent to an unknown channel id: {}", (Object)channelUID);
        }
    }

    private StateUpdate addOsramSpecificCommands(StateUpdate lightState, OnOffType actionType) {
        if (actionType.equals((Object)OnOffType.ON)) {
            lightState.setBrightness(254);
        } else {
            lightState.setTransitionTime(0);
        }
        return lightState;
    }

    private @Nullable StateUpdate convertColorTempChangeToStateUpdate(IncreaseDecreaseType command, FullLight light) {
        StateUpdate stateUpdate = null;
        Integer currentColorTemp = this.getCurrentColorTemp(light.getState());
        if (currentColorTemp != null) {
            int newColorTemp = LightStateConverter.toAdjustedColorTemp(command, currentColorTemp);
            stateUpdate = new StateUpdate().setColorTemperature(newColorTemp);
            this.lastSentColorTemp = newColorTemp;
        }
        return stateUpdate;
    }

    private @Nullable Integer getCurrentColorTemp(@Nullable State lightState) {
        Integer colorTemp = this.lastSentColorTemp;
        if (colorTemp == null && lightState != null) {
            colorTemp = lightState.getColorTemperature();
        }
        return colorTemp;
    }

    private @Nullable StateUpdate convertBrightnessChangeToStateUpdate(IncreaseDecreaseType command, FullLight light) {
        StateUpdate stateUpdate = null;
        Integer currentBrightness = this.getCurrentBrightness(light.getState());
        if (currentBrightness != null) {
            int newBrightness = LightStateConverter.toAdjustedBrightness(command, currentBrightness);
            stateUpdate = this.createBrightnessStateUpdate(currentBrightness, newBrightness);
            this.lastSentBrightness = newBrightness;
        }
        return stateUpdate;
    }

    private @Nullable Integer getCurrentBrightness(@Nullable State lightState) {
        Integer brightness = this.lastSentBrightness;
        if (brightness == null && lightState != null) {
            brightness = !lightState.isOn() ? Integer.valueOf(0) : Integer.valueOf(lightState.getBrightness());
        }
        return brightness;
    }

    private StateUpdate createBrightnessStateUpdate(int currentBrightness, int newBrightness) {
        StateUpdate lightUpdate = new StateUpdate();
        if (newBrightness == 0) {
            lightUpdate.turnOff();
        } else {
            lightUpdate.setBrightness(newBrightness);
            if (currentBrightness == 0) {
                lightUpdate.turnOn();
            }
        }
        return lightUpdate;
    }

    private synchronized @Nullable HueBridgeHandler getHueBridgeHandler() {
        if (this.bridgeHandler == null) {
            Bridge bridge = this.getBridge();
            if (bridge == null) {
                return null;
            }
            BridgeHandler handler = bridge.getHandler();
            if (handler instanceof HueBridgeHandler) {
                this.bridgeHandler = (HueBridgeHandler)handler;
                this.bridgeHandler.registerLightStatusListener(this);
            } else {
                return null;
            }
        }
        return this.bridgeHandler;
    }

    @Override
    public void onLightStateChanged(@Nullable HueBridge bridge, FullLight fullLight) {
        this.logger.trace("onLightStateChanged() was called");
        if (!fullLight.getId().equals(this.lightId)) {
            this.logger.trace("Received state change for another handler's light ({}). Will be ignored.", (Object)fullLight.getId());
            return;
        }
        this.initializeProperties();
        this.lastSentColorTemp = null;
        this.lastSentBrightness = null;
        if (fullLight.getState().isReachable()) {
            this.updateStatus(ThingStatus.ONLINE);
        } else {
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.NONE, "@text/offline.light-not-reachable");
        }
        HSBType hsbType = LightStateConverter.toHSBType(fullLight.getState());
        if (!fullLight.getState().isOn()) {
            hsbType = new HSBType(hsbType.getHue(), hsbType.getSaturation(), new PercentType(0));
        }
        this.updateState("color", (org.eclipse.smarthome.core.types.State)hsbType);
        PercentType percentType = LightStateConverter.toColorTemperaturePercentType(fullLight.getState());
        this.updateState("color_temperature", (org.eclipse.smarthome.core.types.State)percentType);
        percentType = LightStateConverter.toBrightnessPercentType(fullLight.getState());
        if (!fullLight.getState().isOn()) {
            percentType = new PercentType(0);
        }
        this.updateState("brightness", (org.eclipse.smarthome.core.types.State)percentType);
        if (fullLight.getState().isOn()) {
            this.updateState("switch", (org.eclipse.smarthome.core.types.State)OnOffType.ON);
        } else {
            this.updateState("switch", (org.eclipse.smarthome.core.types.State)OnOffType.OFF);
        }
        StringType stringType = LightStateConverter.toAlertStringType(fullLight.getState());
        if (!stringType.toString().equals("NULL")) {
            this.updateState("alert", (org.eclipse.smarthome.core.types.State)stringType);
            this.scheduleAlertStateRestore((Command)stringType);
        }
    }

    public void channelLinked(ChannelUID channelUID) {
        FullLight light;
        HueBridgeHandler handler = this.getHueBridgeHandler();
        if (handler != null && (light = handler.getLightById(this.lightId)) != null) {
            this.onLightStateChanged(null, light);
        }
    }

    @Override
    public void onLightRemoved(@Nullable HueBridge bridge, FullLight light) {
        if (light.getId().equals(this.lightId)) {
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.NONE, "offline.light-removed");
        }
    }

    @Override
    public void onLightAdded(@Nullable HueBridge bridge, FullLight light) {
        if (light.getId().equals(this.lightId)) {
            this.onLightStateChanged(bridge, light);
        }
    }

    private void scheduleAlertStateRestore(Command command) {
        this.cancelScheduledFuture();
        int delay = this.getAlertDuration(command);
        if (delay > 0) {
            this.scheduledFuture = this.scheduler.schedule(() -> this.updateState("alert", (org.eclipse.smarthome.core.types.State)new StringType("NONE")), (long)delay, TimeUnit.MILLISECONDS);
        }
    }

    private void cancelScheduledFuture() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(true);
        }
    }

    private int getAlertDuration(Command command) {
        int delay;
        switch (command.toString()) {
            case "LSELECT": {
                delay = 15000;
                break;
            }
            case "SELECT": {
                delay = 2000;
                break;
            }
            default: {
                delay = -1;
            }
        }
        return delay;
    }
}

