/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.lifx.internal;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.eclipse.smarthome.binding.lifx.internal.LifxLightCommunicationHandler;
import org.eclipse.smarthome.binding.lifx.internal.LifxLightContext;
import org.eclipse.smarthome.binding.lifx.internal.LifxLightState;
import org.eclipse.smarthome.binding.lifx.internal.fields.HSBK;
import org.eclipse.smarthome.binding.lifx.internal.listener.LifxLightStateListener;
import org.eclipse.smarthome.binding.lifx.internal.protocol.AcknowledgementResponse;
import org.eclipse.smarthome.binding.lifx.internal.protocol.ApplicationRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.GetColorZonesRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.GetLightInfraredRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.GetLightPowerRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.GetRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.Packet;
import org.eclipse.smarthome.binding.lifx.internal.protocol.PowerState;
import org.eclipse.smarthome.binding.lifx.internal.protocol.Products;
import org.eclipse.smarthome.binding.lifx.internal.protocol.SetColorRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.SetColorZonesRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.SetLightInfraredRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.SetLightPowerRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.SetPowerRequest;
import org.eclipse.smarthome.binding.lifx.internal.protocol.SignalStrength;
import org.eclipse.smarthome.binding.lifx.internal.util.LifxMessageUtil;
import org.eclipse.smarthome.core.library.types.PercentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LifxLightStateChanger
implements LifxLightStateListener {
    private static final int PACKET_ACKNOWLEDGE_INTERVAL = 250;
    private static final int MAX_RETRIES = 3;
    private final Logger logger = LoggerFactory.getLogger(LifxLightStateChanger.class);
    private final String logId;
    private final Products product;
    private final Duration fadeTime;
    private final LifxLightState pendingLightState;
    private final ScheduledExecutorService scheduler;
    private final LifxLightCommunicationHandler communicationHandler;
    private final ReentrantLock lock = new ReentrantLock();
    private ScheduledFuture<?> sendJob;
    private Map<Integer, List<PendingPacket>> pendingPacketsMap = new ConcurrentHashMap<Integer, List<PendingPacket>>();

    public LifxLightStateChanger(LifxLightContext context, LifxLightCommunicationHandler communicationHandler) {
        this.logId = context.getLogId();
        this.product = context.getProduct();
        this.fadeTime = context.getConfiguration().getFadeTime();
        this.pendingLightState = context.getPendingLightState();
        this.scheduler = context.getScheduler();
        this.communicationHandler = communicationHandler;
    }

    private void sendPendingPackets() {
        try {
            try {
                this.lock.lock();
                this.removeFailedPackets();
                PendingPacket pendingPacket = this.findPacketToSend();
                if (pendingPacket != null) {
                    Packet packet = pendingPacket.packet;
                    if (pendingPacket.sendCount == 0) {
                        this.logger.debug("{} : Sending {} packet", (Object)this.logId, (Object)packet.getClass().getSimpleName());
                        this.communicationHandler.sendPacket(packet);
                    } else {
                        this.logger.debug("{} : Resending {} packet", (Object)this.logId, (Object)packet.getClass().getSimpleName());
                        this.communicationHandler.resendPacket(packet);
                    }
                    pendingPacket.lastSend = System.currentTimeMillis();
                    ++pendingPacket.sendCount;
                }
            }
            catch (Exception e) {
                this.logger.error("Error occurred while sending packet", (Throwable)e);
                this.lock.unlock();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void start() {
        try {
            try {
                this.lock.lock();
                this.communicationHandler.addResponsePacketListener(this::handleResponsePacket);
                this.pendingLightState.addListener(this);
                if (this.sendJob == null || this.sendJob.isCancelled()) {
                    this.sendJob = this.scheduler.scheduleWithFixedDelay(this::sendPendingPackets, 0L, 50L, TimeUnit.MILLISECONDS);
                }
            }
            catch (Exception e) {
                this.logger.error("Error occurred while starting send packets job", (Throwable)e);
                this.lock.unlock();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void stop() {
        try {
            try {
                this.lock.lock();
                this.communicationHandler.removeResponsePacketListener(this::handleResponsePacket);
                this.pendingLightState.removeListener(this);
                if (this.sendJob != null && !this.sendJob.isCancelled()) {
                    this.sendJob.cancel(true);
                    this.sendJob = null;
                }
                this.pendingPacketsMap.clear();
            }
            catch (Exception e) {
                this.logger.error("Error occurred while stopping send packets job", (Throwable)e);
                this.lock.unlock();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private List<PendingPacket> createPendingPackets(Packet ... packets) {
        Integer packetType = null;
        ArrayList<PendingPacket> pendingPackets = new ArrayList<PendingPacket>();
        Packet[] packetArray = packets;
        int n = packets.length;
        int n2 = 0;
        while (n2 < n) {
            Packet packet = packetArray[n2];
            packet.setAckRequired(true);
            packet.setResponseRequired(false);
            pendingPackets.add(new PendingPacket(packet));
            if (packetType == null) {
                packetType = packet.getPacketType();
            } else if (packetType.intValue() != packet.getPacketType()) {
                throw new IllegalArgumentException("Packets should have same packet type");
            }
            ++n2;
        }
        return pendingPackets;
    }

    private void addPacketsToMap(Packet ... packets) {
        List<PendingPacket> pendingPackets = this.createPendingPackets(packets);
        int packetType = packets[0].getPacketType();
        try {
            this.lock.lock();
            if (this.pendingPacketsMap.get(packetType) == null) {
                this.pendingPacketsMap.put(packetType, pendingPackets);
            } else {
                this.pendingPacketsMap.get(packetType).addAll(pendingPackets);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void replacePacketsInMap(Packet ... packets) {
        List<PendingPacket> pendingPackets = this.createPendingPackets(packets);
        int packetType = packets[0].getPacketType();
        try {
            this.lock.lock();
            this.pendingPacketsMap.put(packetType, pendingPackets);
        }
        finally {
            this.lock.unlock();
        }
    }

    private PendingPacket findPacketToSend() {
        PendingPacket result = null;
        for (List<PendingPacket> pendingPackets : this.pendingPacketsMap.values()) {
            for (PendingPacket pendingPacket : pendingPackets) {
                if (!pendingPacket.hasAcknowledgeIntervalElapsed() || result != null && pendingPacket.lastSend >= result.lastSend) continue;
                result = pendingPacket;
            }
        }
        return result;
    }

    private void removePacketsByType(int packetType) {
        try {
            this.lock.lock();
            this.pendingPacketsMap.remove(packetType);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void removeFailedPackets() {
        for (Integer key : this.pendingPacketsMap.keySet()) {
            List<PendingPacket> pendingPackets = this.pendingPacketsMap.get(key);
            Iterator<PendingPacket> it = pendingPackets.iterator();
            while (it.hasNext()) {
                PendingPacket pendingPacket = it.next();
                if (pendingPacket.sendCount <= 3 || !pendingPacket.hasAcknowledgeIntervalElapsed()) continue;
                this.logger.warn("{} failed (unacknowledged {} times)", (Object)pendingPacket.packet.getClass().getSimpleName(), (Object)pendingPacket.sendCount);
                it.remove();
            }
        }
    }

    private PendingPacket removeAcknowledgedPacket(int sequenceNumber) {
        for (Integer key : this.pendingPacketsMap.keySet()) {
            List<PendingPacket> pendingPackets = this.pendingPacketsMap.get(key);
            Iterator<PendingPacket> it = pendingPackets.iterator();
            while (it.hasNext()) {
                PendingPacket pendingPacket = it.next();
                if (pendingPacket.packet.getSequence() != sequenceNumber) continue;
                it.remove();
                return pendingPacket;
            }
        }
        return null;
    }

    @Override
    public void handleColorsChange(HSBK[] oldColors, HSBK[] newColors) {
        if (LifxMessageUtil.sameColors(newColors)) {
            SetColorRequest packet = new SetColorRequest(this.pendingLightState.getColors()[0], this.fadeTime.toMillis());
            this.removePacketsByType(501);
            this.replacePacketsInMap(packet);
        } else {
            ArrayList<SetColorZonesRequest> packets = new ArrayList<SetColorZonesRequest>();
            int i = 0;
            while (i < newColors.length) {
                if (newColors[i] != null && !newColors[i].equals(oldColors[i])) {
                    packets.add(new SetColorZonesRequest(i, newColors[i], this.fadeTime.toMillis(), ApplicationRequest.APPLY));
                }
                ++i;
            }
            if (!packets.isEmpty()) {
                this.removePacketsByType(102);
                this.addPacketsToMap(packets.toArray(new SetColorZonesRequest[packets.size()]));
            }
        }
    }

    @Override
    public void handlePowerStateChange(PowerState oldPowerState, PowerState newPowerState) {
        if (newPowerState != null && !newPowerState.equals((Object)oldPowerState)) {
            SetLightPowerRequest packet = new SetLightPowerRequest(this.pendingLightState.getPowerState());
            this.replacePacketsInMap(packet);
        }
    }

    @Override
    public void handleInfraredChange(PercentType oldInfrared, PercentType newInfrared) {
        int infrared = LifxMessageUtil.percentTypeToInfrared(this.pendingLightState.getInfrared());
        SetLightInfraredRequest packet = new SetLightInfraredRequest(infrared);
        this.replacePacketsInMap(packet);
    }

    @Override
    public void handleSignalStrengthChange(SignalStrength oldSignalStrength, SignalStrength newSignalStrength) {
    }

    public void handleResponsePacket(Packet packet) {
        if (packet instanceof AcknowledgementResponse) {
            PendingPacket pendingPacket;
            long ackTimestamp = System.currentTimeMillis();
            try {
                this.lock.lock();
                pendingPacket = this.removeAcknowledgedPacket(packet.getSequence());
            }
            finally {
                this.lock.unlock();
            }
            if (pendingPacket != null) {
                Packet sentPacket = pendingPacket.packet;
                this.logger.debug("{} : {} packet was acknowledged in {}ms", new Object[]{this.logId, sentPacket.getClass().getSimpleName(), ackTimestamp - pendingPacket.lastSend});
                if (sentPacket instanceof SetPowerRequest) {
                    GetLightPowerRequest powerPacket = new GetLightPowerRequest();
                    this.communicationHandler.sendPacket(powerPacket);
                } else if (sentPacket instanceof SetColorRequest) {
                    GetRequest colorPacket = new GetRequest();
                    this.communicationHandler.sendPacket(colorPacket);
                    this.getZonesIfZonesAreSet();
                } else if (sentPacket instanceof SetColorZonesRequest) {
                    this.getZonesIfZonesAreSet();
                } else if (sentPacket instanceof SetLightInfraredRequest) {
                    GetLightInfraredRequest infraredPacket = new GetLightInfraredRequest();
                    this.communicationHandler.sendPacket(infraredPacket);
                }
            } else {
                this.logger.debug("{} : No pending packet found for ack with sequence number: {}", (Object)this.logId, (Object)packet.getSequence());
            }
        }
    }

    private void getZonesIfZonesAreSet() {
        List<PendingPacket> pending;
        if (this.product.isMultiZone() && ((pending = this.pendingPacketsMap.get(501)) == null || pending.isEmpty())) {
            GetColorZonesRequest zoneColorPacket = new GetColorZonesRequest();
            this.communicationHandler.sendPacket(zoneColorPacket);
        }
    }

    private class PendingPacket {
        long lastSend;
        int sendCount;
        final Packet packet;

        private PendingPacket(Packet packet) {
            this.packet = packet;
        }

        private boolean hasAcknowledgeIntervalElapsed() {
            long millisSinceLastSend = System.currentTimeMillis() - this.lastSend;
            return millisSinceLastSend > 250L;
        }
    }
}

