/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.weatherunderground.handler;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.URL;
import java.util.Calendar;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang.StringUtils;
import org.eclipse.smarthome.binding.weatherunderground.internal.config.WeatherUndergroundConfiguration;
import org.eclipse.smarthome.binding.weatherunderground.internal.json.WeatherUndergroundJsonData;
import org.eclipse.smarthome.binding.weatherunderground.internal.json.WeatherUndergroundJsonUtils;
import org.eclipse.smarthome.core.i18n.LocaleProvider;
import org.eclipse.smarthome.core.library.types.DateTimeType;
import org.eclipse.smarthome.core.library.types.DecimalType;
import org.eclipse.smarthome.core.library.types.StringType;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.ThingStatusDetail;
import org.eclipse.smarthome.core.thing.binding.BaseThingHandler;
import org.eclipse.smarthome.core.types.Command;
import org.eclipse.smarthome.core.types.RefreshType;
import org.eclipse.smarthome.core.types.State;
import org.eclipse.smarthome.core.types.UnDefType;
import org.eclipse.smarthome.io.net.http.HttpUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WeatherUndergroundHandler
extends BaseThingHandler {
    private final Logger logger = LoggerFactory.getLogger(WeatherUndergroundHandler.class);
    private static final String URL = "http://api.wunderground.com/api/%APIKEY%/%FEATURES%/%SETTINGS%/q/%QUERY%.json";
    private static final String FEATURE_CONDITIONS = "conditions";
    private static final String FEATURE_FORECAST10DAY = "forecast10day";
    private static final String FEATURE_GEOLOOKUP = "geolookup";
    private static final Set<String> USUAL_FEATURES = Stream.of("conditions", "forecast10day").collect(Collectors.toSet());
    private static final int DEFAULT_REFRESH_PERIOD = 30;
    private static final int FETCH_TIMEOUT_MS = 30000;
    private LocaleProvider localeProvider;
    private WeatherUndergroundJsonData weatherData;
    private ScheduledFuture<?> refreshJob;
    private Gson gson;

    public WeatherUndergroundHandler(Thing thing, LocaleProvider localeProvider) {
        super(thing);
        this.localeProvider = localeProvider;
        this.gson = new Gson();
    }

    public void initialize() {
        String lang;
        this.logger.debug("Initializing WeatherUnderground handler.");
        WeatherUndergroundConfiguration config = (WeatherUndergroundConfiguration)this.getConfigAs(WeatherUndergroundConfiguration.class);
        this.logger.debug("config apikey = {}", (Object)config.apikey);
        this.logger.debug("config location = {}", (Object)config.location);
        this.logger.debug("config language = {}", (Object)config.language);
        this.logger.debug("config refresh = {}", (Object)config.refresh);
        boolean validConfig = true;
        String errors = "";
        String statusDescr = null;
        if (StringUtils.trimToNull((String)config.apikey) == null) {
            errors = String.valueOf(errors) + " Parameter 'apikey' must be configured.";
            statusDescr = "@text/offline.conf-error-missing-apikey";
            validConfig = false;
        }
        if (StringUtils.trimToNull((String)config.location) == null) {
            errors = String.valueOf(errors) + " Parameter 'location' must be configured.";
            statusDescr = "@text/offline.conf-error-missing-location";
            validConfig = false;
        }
        if (config.language != null && (lang = StringUtils.trimToEmpty((String)config.language)).length() != 2) {
            errors = String.valueOf(errors) + " Parameter 'language' must be 2 letters.";
            statusDescr = "@text/offline.conf-error-syntax-language";
            validConfig = false;
        }
        if (config.refresh != null && config.refresh < 5) {
            errors = String.valueOf(errors) + " Parameter 'refresh' must be at least 5 minutes.";
            statusDescr = "@text/offline.conf-error-min-refresh";
            validConfig = false;
        }
        errors = errors.trim();
        if (validConfig) {
            this.startAutomaticRefresh();
        } else {
            this.logger.debug("Disabling thing '{}': {}", (Object)this.getThing().getUID(), (Object)errors);
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.CONFIGURATION_ERROR, statusDescr);
        }
    }

    private void startAutomaticRefresh() {
        if (this.refreshJob == null || this.refreshJob.isCancelled()) {
            Runnable runnable = new Runnable(){

                @Override
                public void run() {
                    try {
                        WeatherUndergroundHandler.this.updateWeatherData();
                        for (Channel channel : WeatherUndergroundHandler.this.getThing().getChannels()) {
                            WeatherUndergroundHandler.this.updateChannel(channel.getUID().getId());
                        }
                    }
                    catch (Exception e) {
                        WeatherUndergroundHandler.this.logger.debug("Exception occurred during execution: {}", (Object)e.getMessage(), (Object)e);
                    }
                }
            };
            WeatherUndergroundConfiguration config = (WeatherUndergroundConfiguration)this.getConfigAs(WeatherUndergroundConfiguration.class);
            int period = config.refresh != null ? config.refresh : 30;
            this.refreshJob = this.scheduler.scheduleWithFixedDelay(runnable, 0L, period, TimeUnit.MINUTES);
        }
    }

    public void dispose() {
        this.logger.debug("Disposing WeatherUnderground handler.");
        if (this.refreshJob != null && !this.refreshJob.isCancelled()) {
            this.refreshJob.cancel(true);
            this.refreshJob = null;
        }
    }

    public void handleCommand(ChannelUID channelUID, Command command) {
        if (command instanceof RefreshType) {
            this.updateChannel(channelUID.getId());
        } else {
            this.logger.debug("The Weather Underground binding is a read-only binding and cannot handle command {}", (Object)command);
        }
    }

    private void updateChannel(String channelId) {
        Channel channel = this.getThing().getChannel(channelId);
        if (channel != null && this.isLinked(channelId)) {
            Object value;
            String sourceUnit = (String)channel.getConfiguration().get("SourceUnit");
            if (sourceUnit == null) {
                sourceUnit = "";
            } else if (sourceUnit.length() > 0) {
                sourceUnit = String.valueOf(sourceUnit.substring(0, 1).toUpperCase()) + sourceUnit.substring(1);
            }
            try {
                value = WeatherUndergroundJsonUtils.getValue(String.valueOf(channelId) + sourceUnit, this.weatherData);
            }
            catch (Exception e) {
                this.logger.debug("Update channel {}: Can't get value: {}", (Object)channelId, (Object)e.getMessage());
                return;
            }
            UnDefType state = null;
            if (value == null) {
                state = UnDefType.UNDEF;
            } else if (value instanceof Calendar) {
                state = new DateTimeType((Calendar)value);
            } else if (value instanceof BigDecimal) {
                state = new DecimalType((BigDecimal)value);
            } else if (value instanceof Integer) {
                state = new DecimalType(BigDecimal.valueOf(((Integer)value).longValue()));
            } else if (value instanceof String) {
                state = new StringType(value.toString());
            } else if (value instanceof URL) {
                state = HttpUtil.downloadImage((String)((URL)value).toExternalForm(), (int)30000);
                if (state == null) {
                    this.logger.debug("Failed to download the content of URL {}", (Object)((URL)value).toExternalForm());
                    state = UnDefType.UNDEF;
                }
            } else {
                this.logger.debug("Update channel {}: Unsupported value type {}", (Object)channelId, (Object)value.getClass().getSimpleName());
            }
            this.logger.debug("Update channel {} with state {} ({})", new Object[]{channelId, state == null ? "null" : state.toString(), value == null ? "null" : value.getClass().getSimpleName()});
            if (state != null) {
                this.updateState(channelId, (State)state);
            }
        }
    }

    private boolean updateWeatherData() {
        WeatherUndergroundConfiguration config = (WeatherUndergroundConfiguration)this.getConfigAs(WeatherUndergroundConfiguration.class);
        this.weatherData = this.getWeatherData(USUAL_FEATURES, StringUtils.trimToEmpty((String)config.location));
        return this.weatherData != null;
    }

    private boolean checkWeatherLocation(String location) {
        WeatherUndergroundJsonData data = this.getWeatherData(Collections.singleton(FEATURE_GEOLOOKUP), location);
        return data != null;
    }

    private WeatherUndergroundJsonData getWeatherData(Set<String> features, String location) {
        WeatherUndergroundJsonData result = null;
        boolean resultOk = false;
        String error = null;
        String errorDetail = null;
        String statusDescr = null;
        try {
            WeatherUndergroundConfiguration config = (WeatherUndergroundConfiguration)this.getConfigAs(WeatherUndergroundConfiguration.class);
            String urlStr = URL.replace("%APIKEY%", StringUtils.trimToEmpty((String)config.apikey));
            urlStr = urlStr.replace("%FEATURES%", String.join((CharSequence)"/", features));
            String lang = StringUtils.trimToEmpty((String)config.language);
            if (lang.isEmpty()) {
                lang = this.localeProvider.getLocale().getLanguage();
                this.logger.debug("Use language from system locale: {}", (Object)lang);
            }
            urlStr = lang.isEmpty() ? urlStr.replace("%SETTINGS%", "") : urlStr.replace("%SETTINGS%", "lang:" + lang.toUpperCase());
            urlStr = urlStr.replace("%QUERY%", location);
            this.logger.debug("URL = {}", (Object)urlStr);
            String response = null;
            try {
                response = HttpUtil.executeUrl((String)"GET", (String)urlStr, (int)30000);
                this.logger.debug("weatherData = {}", (Object)response);
            }
            catch (IllegalArgumentException e) {
                error = "Error creating URI with location parameter: '" + location + "'";
                errorDetail = e.getMessage();
                statusDescr = "@text/offline.uri-error";
            }
            if (response != null) {
                result = (WeatherUndergroundJsonData)this.gson.fromJson(response, WeatherUndergroundJsonData.class);
                if (result == null) {
                    errorDetail = "no data returned";
                } else if (result.getResponse() == null) {
                    errorDetail = "missing response sub-object";
                } else if (result.getResponse().getErrorDescription() != null) {
                    if ("keynotfound".equals(result.getResponse().getErrorType())) {
                        error = "API key has to be fixed";
                        statusDescr = "@text/offline.comm-error-invalid-api-key";
                    }
                    errorDetail = result.getResponse().getErrorDescription();
                } else {
                    resultOk = true;
                    for (String feature : features) {
                        if (feature.equals(FEATURE_CONDITIONS) && result.getCurrent() == null) {
                            resultOk = false;
                            errorDetail = "missing current_observation sub-object";
                            continue;
                        }
                        if (feature.equals(FEATURE_FORECAST10DAY) && result.getForecast() == null) {
                            resultOk = false;
                            errorDetail = "missing forecast sub-object";
                            continue;
                        }
                        if (!feature.equals(FEATURE_GEOLOOKUP) || result.getLocation() != null) continue;
                        resultOk = false;
                        errorDetail = "missing location sub-object";
                    }
                }
            }
            if (!resultOk && error == null) {
                error = "Error in Weather Underground response";
                statusDescr = "@text/offline.comm-error-response";
            }
        }
        catch (IOException e) {
            error = "Error running Weather Underground request";
            errorDetail = e.getMessage();
            statusDescr = "@text/offline.comm-error-running-request";
        }
        catch (JsonSyntaxException e) {
            error = "Error parsing Weather Underground response";
            errorDetail = e.getMessage();
            statusDescr = "@text/offline.comm-error-parsing-response";
        }
        if (resultOk) {
            this.updateStatus(ThingStatus.ONLINE);
        } else {
            this.logger.debug("{}: {}", (Object)error, (Object)errorDetail);
            this.updateStatus(ThingStatus.OFFLINE, ThingStatusDetail.OFFLINE.COMMUNICATION_ERROR, statusDescr);
        }
        return resultOk ? result : null;
    }
}

