/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.io.transport.mqtt.internal;

import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.net.SocketFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import org.apache.commons.lang.StringUtils;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttClientPersistence;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.MqttTopic;
import org.eclipse.paho.client.mqttv3.persist.MqttDefaultFilePersistence;
import org.eclipse.smarthome.io.transport.mqtt.MqttMessageConsumer;
import org.eclipse.smarthome.io.transport.mqtt.MqttMessageProducer;
import org.eclipse.smarthome.io.transport.mqtt.MqttSenderChannel;
import org.eclipse.smarthome.io.transport.mqtt.MqttWillAndTestament;
import org.eclipse.smarthome.io.transport.mqtt.internal.MqttBrokerConnectionHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MqttBrokerConnection
implements MqttCallback {
    private static Logger logger = LoggerFactory.getLogger(MqttBrokerConnection.class);
    private static final int RECONNECT_FREQUENCY = 60000;
    private String name;
    private String url;
    private String user;
    private String password;
    private int qos = 0;
    private boolean retain = false;
    private boolean async = true;
    private MqttWillAndTestament lastWill;
    private String clientId;
    private MqttClient client;
    private boolean started;
    private int keepAliveInterval = 60;
    private List<MqttMessageConsumer> consumers = new CopyOnWriteArrayList<MqttMessageConsumer>();
    private List<MqttMessageProducer> producers = new CopyOnWriteArrayList<MqttMessageProducer>();
    private Timer reconnectTimer;

    public MqttBrokerConnection(String name) {
        this.name = name;
    }

    public synchronized void start() throws Exception {
        if (StringUtils.isEmpty((String)this.url)) {
            logger.debug("No url defined for MQTT broker connection '{}'. Not starting.", (Object)this.name);
            return;
        }
        logger.info("Starting MQTT broker connection '{}'", (Object)this.name);
        this.openConnection();
        if (this.reconnectTimer != null) {
            this.reconnectTimer.cancel();
        }
        for (MqttMessageConsumer c : this.consumers) {
            this.startConsumer(c);
        }
        for (MqttMessageProducer p : this.producers) {
            this.startProducer(p);
        }
        this.started = true;
    }

    public String getName() {
        return this.name;
    }

    public String getUrl() {
        return this.url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public String getUser() {
        return this.user;
    }

    public void setUser(String user) {
        this.user = user;
    }

    public String getPassword() {
        return this.password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public int getQos() {
        return this.qos;
    }

    public void setQos(int qos) {
        if (qos >= 0 && qos <= 2) {
            this.qos = qos;
        }
    }

    public boolean isRetain() {
        return this.retain;
    }

    public void setRetain(boolean retain) {
        this.retain = retain;
    }

    public MqttWillAndTestament getLastWill() {
        return this.lastWill;
    }

    public void setLastWill(MqttWillAndTestament lastWill) {
        this.lastWill = lastWill;
    }

    public boolean isAsync() {
        return this.async;
    }

    public void setAsync(boolean async) {
        this.async = async;
    }

    public void setClientId(String value) {
        this.clientId = value;
    }

    private void openConnection() throws Exception {
        if (this.client != null && this.client.isConnected()) {
            return;
        }
        if (StringUtils.isBlank((String)this.url)) {
            throw new Exception("Missing url");
        }
        if (this.client == null) {
            if (StringUtils.isBlank((String)this.clientId) || this.clientId.length() > 23) {
                this.clientId = MqttClient.generateClientId();
            }
            String tmpDir = String.valueOf(System.getProperty("java.io.tmpdir")) + "/" + this.name;
            MqttDefaultFilePersistence dataStore = new MqttDefaultFilePersistence(tmpDir);
            logger.debug("Creating new client for '{}' using id '{}' and file store '{}'", new Object[]{this.url, this.clientId, tmpDir});
            this.client = new MqttClient(this.url, this.clientId, (MqttClientPersistence)dataStore);
            this.client.setCallback((MqttCallback)this);
        }
        MqttConnectOptions options = new MqttConnectOptions();
        if (!StringUtils.isBlank((String)this.user)) {
            options.setUserName(this.user);
        }
        if (!StringUtils.isBlank((String)this.password)) {
            options.setPassword(this.password.toCharArray());
        }
        if (this.url.toLowerCase().contains("ssl")) {
            if (StringUtils.isNotBlank((String)System.getProperty("com.ibm.ssl.protocol"))) {
                Properties sslProps = new Properties();
                this.addSystemProperty("com.ibm.ssl.protocol", sslProps);
                this.addSystemProperty("com.ibm.ssl.contextProvider", sslProps);
                this.addSystemProperty("com.ibm.ssl.keyStore", sslProps);
                this.addSystemProperty("com.ibm.ssl.keyStorePassword", sslProps);
                this.addSystemProperty("com.ibm.ssl.keyStoreType", sslProps);
                this.addSystemProperty("com.ibm.ssl.keyStoreProvider", sslProps);
                this.addSystemProperty("com.ibm.ssl.trustStore", sslProps);
                this.addSystemProperty("com.ibm.ssl.trustStorePassword", sslProps);
                this.addSystemProperty("com.ibm.ssl.trustStoreType", sslProps);
                this.addSystemProperty("com.ibm.ssl.trustStoreProvider", sslProps);
                this.addSystemProperty("com.ibm.ssl.enabledCipherSuites", sslProps);
                this.addSystemProperty("com.ibm.ssl.keyManager", sslProps);
                this.addSystemProperty("com.ibm.ssl.trustManager", sslProps);
                options.setSSLProperties(sslProps);
            } else {
                SSLContext sslContext = SSLContext.getInstance("TLSv1.2");
                sslContext.init(null, new TrustManager[]{this.getVeryTrustingTrustManager()}, new SecureRandom());
                SSLSocketFactory socketFactory = sslContext.getSocketFactory();
                options.setSocketFactory((SocketFactory)socketFactory);
            }
        }
        if (this.lastWill != null) {
            options.setWill(this.lastWill.getTopic(), this.lastWill.getPayload(), this.lastWill.getQos(), this.lastWill.isRetain());
        }
        options.setKeepAliveInterval(this.keepAliveInterval);
        this.client.connect(options);
    }

    private TrustManager getVeryTrustingTrustManager() {
        return new X509TrustManager(){

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return new X509Certificate[0];
            }

            @Override
            public void checkClientTrusted(X509Certificate[] certs, String authType) {
            }

            @Override
            public void checkServerTrusted(X509Certificate[] certs, String authType) {
            }
        };
    }

    private Properties addSystemProperty(String key, Properties props) {
        String value = System.getProperty(key);
        if (StringUtils.isNotBlank((String)value)) {
            props.put(key, value);
        }
        return props;
    }

    public synchronized void addProducer(MqttMessageProducer publisher) {
        this.producers.add(publisher);
        if (this.started) {
            this.startProducer(publisher);
        }
    }

    private void startProducer(MqttMessageProducer publisher) {
        logger.trace("Starting message producer for broker '{}'", (Object)this.name);
        publisher.setSenderChannel(new MqttSenderChannel(){

            @Override
            public void publish(String topic, byte[] payload) throws Exception {
                if (!MqttBrokerConnection.this.started) {
                    logger.warn("Broker connection not started. Cannot publish message to topic '{}'", (Object)topic);
                    return;
                }
                MqttMessage message = new MqttMessage(payload);
                message.setQos(MqttBrokerConnection.this.qos);
                message.setRetained(MqttBrokerConnection.this.retain);
                MqttTopic mqttTopic = MqttBrokerConnection.this.client.getTopic(topic);
                MqttDeliveryToken deliveryToken = mqttTopic.publish(message);
                logger.debug("Publishing message {} to topic '{}'", (Object)deliveryToken.getMessageId(), (Object)topic);
                if (!MqttBrokerConnection.this.async) {
                    deliveryToken.waitForCompletion(10000L);
                    if (!deliveryToken.isComplete()) {
                        logger.error("Did not receive completion message within timeout limit whilst publishing to topic '{}'", (Object)topic);
                    }
                }
            }
        });
    }

    public synchronized void addConsumer(MqttMessageConsumer subscriber) {
        this.consumers.add(subscriber);
        if (this.started) {
            this.startConsumer(subscriber);
        }
    }

    private void startConsumer(MqttMessageConsumer subscriber) {
        String topic = subscriber.getTopic();
        logger.debug("Starting message consumer for broker '{}' on topic '{}'", (Object)this.name, (Object)topic);
        try {
            this.client.subscribe(topic, this.qos);
        }
        catch (Exception e) {
            logger.error("Error starting consumer", (Throwable)e);
        }
    }

    public synchronized void removeProducer(MqttMessageProducer publisher) {
        logger.debug("Removing message producer for broker '{}'", (Object)this.name);
        publisher.setSenderChannel(null);
        this.producers.remove(publisher);
    }

    public synchronized void removeConsumer(MqttMessageConsumer subscriber) {
        logger.debug("Unsubscribing message consumer for topic '{}' from broker '{}'", (Object)subscriber.getTopic(), (Object)this.name);
        try {
            if (this.started) {
                this.client.unsubscribe(subscriber.getTopic());
            }
        }
        catch (Exception e) {
            logger.error("Error unsubscribing topic from broker", (Throwable)e);
        }
        this.consumers.remove(subscriber);
    }

    public synchronized void close() {
        logger.debug("Closing connection to broker '{}'", (Object)this.name);
        try {
            if (this.started) {
                this.client.disconnect();
            }
        }
        catch (MqttException e) {
            logger.error("Error closing connection to broker", (Throwable)e);
        }
        this.started = false;
    }

    public synchronized void connectionLost(Throwable t) {
        logger.error("MQTT connection to broker was lost", t);
        if (t instanceof MqttException) {
            MqttException e = (MqttException)t;
            logger.error("MQTT connection to '{}' was lost: {} : ReasonCode {} : Cause : {}", new Object[]{this.name, e.getMessage(), e.getReasonCode(), e.getCause() == null ? "Unknown" : e.getCause().getMessage()});
        } else {
            logger.error("MQTT connection to '{}' was lost: {}", (Object)this.name, (Object)t.getMessage());
        }
        this.started = false;
        logger.info("Starting connection helper to periodically try restore connection to broker '{}'", (Object)this.name);
        MqttBrokerConnectionHelper helper = new MqttBrokerConnectionHelper(this);
        this.reconnectTimer = new Timer(true);
        this.reconnectTimer.schedule((TimerTask)helper, 10000L, 60000L);
    }

    public void deliveryComplete(IMqttDeliveryToken token) {
        logger.trace("Message with id {} delivered.", (Object)token.getMessageId());
    }

    public void messageArrived(String topic, MqttMessage message) throws Exception {
        logger.trace("Received message on topic '{}' : {}", (Object)topic, (Object)new String(message.getPayload()));
        for (MqttMessageConsumer consumer : this.consumers) {
            if (!this.isTopicMatch(topic, consumer.getTopic())) continue;
            consumer.processMessage(topic, message.getPayload());
        }
    }

    private boolean isTopicMatch(String source, String target) {
        if (source.equals(target)) {
            return true;
        }
        if (target.indexOf(43) == -1 && target.indexOf(35) == -1) {
            return false;
        }
        String regex = target;
        regex = StringUtils.replace((String)regex, (String)"+", (String)"[^/]*");
        boolean result = source.matches(regex = StringUtils.replace((String)regex, (String)"#", (String)".*"));
        if (result) {
            logger.trace("Topic match for '{}' and '{}' using regex {}", new Object[]{source, target, regex});
            return true;
        }
        logger.trace("No topic match for '{}' and '{}' using regex {}", new Object[]{source, target, regex});
        return false;
    }

    public void setKeepAliveInterval(int keepAliveInterval) {
        this.keepAliveInterval = keepAliveInterval;
    }
}

