/**
 * Copyright (c) 2014 openHAB UG (haftungsbeschraenkt) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.smarthome.model.thing.internal;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smarthome.config.core.Configuration;
import org.eclipse.smarthome.core.common.registry.AbstractProvider;
import org.eclipse.smarthome.core.thing.Bridge;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingProvider;
import org.eclipse.smarthome.core.thing.ThingTypeUID;
import org.eclipse.smarthome.core.thing.ThingUID;
import org.eclipse.smarthome.core.thing.binding.builder.BridgeBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.ChannelBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.GenericThingBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.ThingBuilder;
import org.eclipse.smarthome.core.thing.type.ChannelDefinition;
import org.eclipse.smarthome.core.thing.type.ChannelType;
import org.eclipse.smarthome.core.thing.type.ThingType;
import org.eclipse.smarthome.core.thing.type.ThingTypeRegistry;
import org.eclipse.smarthome.core.thing.util.ThingHelper;
import org.eclipse.smarthome.model.core.ModelRepository;
import org.eclipse.smarthome.model.core.ModelRepositoryChangeListener;
import org.eclipse.smarthome.model.thing.thing.ModelBridge;
import org.eclipse.smarthome.model.thing.thing.ModelChannel;
import org.eclipse.smarthome.model.thing.thing.ModelProperty;
import org.eclipse.smarthome.model.thing.thing.ModelPropertyContainer;
import org.eclipse.smarthome.model.thing.thing.ModelThing;
import org.eclipse.smarthome.model.thing.thing.ThingModel;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link ThingProvider} implementation which computes *.things files.
 * 
 * @author Oliver Libutzki - Initial contribution
 */
@SuppressWarnings("all")
public class GenericThingProvider extends AbstractProvider<Thing> implements ThingProvider, ModelRepositoryChangeListener {
  private ModelRepository modelRepository;
  
  private ThingTypeRegistry thingTypeRegistry;
  
  private Map<String, Collection<Thing>> thingsMap = new ConcurrentHashMap<String, Collection<Thing>>();
  
  private final static Logger logger = LoggerFactory.getLogger(GenericThingProvider.class);
  
  public void activate() {
    Iterable<String> _allModelNamesOfType = this.modelRepository.getAllModelNamesOfType("things");
    final Procedure1<String> _function = new Procedure1<String>() {
      public void apply(final String it) {
        GenericThingProvider.this.createThingsFromModel(it);
      }
    };
    IterableExtensions.<String>forEach(_allModelNamesOfType, _function);
  }
  
  public Collection<Thing> getAll() {
    Collection<Collection<Thing>> _values = this.thingsMap.values();
    Iterable<Thing> _flatten = Iterables.<Thing>concat(_values);
    return IterableExtensions.<Thing>toList(_flatten);
  }
  
  private void createThingsFromModel(final String modelName) {
    GenericThingProvider.logger.debug("Read things from model \'{}\'", modelName);
    final ArrayList<Thing> things = CollectionLiterals.<Thing>newArrayList();
    boolean _notEquals = (!Objects.equal(this.modelRepository, null));
    if (_notEquals) {
      EObject _model = this.modelRepository.getModel(modelName);
      final ThingModel model = ((ThingModel) _model);
      boolean _notEquals_1 = (!Objects.equal(model, null));
      if (_notEquals_1) {
        EList<ModelThing> _things = model.getThings();
        final Procedure1<ModelThing> _function = new Procedure1<ModelThing>() {
          public void apply(final ModelThing it) {
            GenericThingProvider.this.createThing(it, null, things);
          }
        };
        IterableExtensions.<ModelThing>forEach(_things, _function);
      }
    }
    boolean _isEmpty = things.isEmpty();
    if (_isEmpty) {
      this.thingsMap.remove(modelName);
    } else {
      this.thingsMap.put(modelName, things);
    }
  }
  
  private void createThing(final ModelThing modelThing, final Bridge parentBridge, final List<Thing> thingList) {
    ThingTypeUID thingTypeUID = null;
    ThingUID thingUID = null;
    boolean _notEquals = (!Objects.equal(parentBridge, null));
    if (_notEquals) {
      ThingTypeUID _thingTypeUID = parentBridge.getThingTypeUID();
      final String bindingId = _thingTypeUID.getBindingId();
      final String thingTypeId = modelThing.getThingTypeId();
      final String thingId = modelThing.getThingId();
      ThingTypeUID _thingTypeUID_1 = new ThingTypeUID(bindingId, thingTypeId);
      thingTypeUID = _thingTypeUID_1;
      List<String> _parentPath = this.getParentPath(parentBridge);
      ThingUID _thingUID = new ThingUID(thingTypeUID, thingId, ((String[])Conversions.unwrapArray(_parentPath, String.class)));
      thingUID = _thingUID;
    } else {
      String _id = modelThing.getId();
      ThingUID _thingUID_1 = new ThingUID(_id);
      thingUID = _thingUID_1;
      String _bindingId = thingUID.getBindingId();
      String _thingTypeId = thingUID.getThingTypeId();
      ThingTypeUID _thingTypeUID_2 = new ThingTypeUID(_bindingId, _thingTypeId);
      thingTypeUID = _thingTypeUID_2;
    }
    GenericThingProvider.logger.debug("Creating thing for type \'{}\' with UID \'{}.", thingTypeUID, thingUID);
    final Configuration configuration = this.createConfiguration(modelThing);
    GenericThingBuilder<?> _xifexpression = null;
    if ((modelThing instanceof ModelBridge)) {
      _xifexpression = BridgeBuilder.create(thingUID);
    } else {
      _xifexpression = ThingBuilder.create(thingUID);
    }
    final GenericThingBuilder<?> thingBuilder = _xifexpression;
    thingBuilder.withConfiguration(configuration);
    boolean _notEquals_1 = (!Objects.equal(parentBridge, null));
    if (_notEquals_1) {
      ThingUID _uID = parentBridge.getUID();
      thingBuilder.withBridge(_uID);
    }
    final ThingType thingType = this.getThingType(thingTypeUID);
    EList<ModelChannel> _channels = modelThing.getChannels();
    List<ChannelDefinition> _elvis = null;
    List<ChannelDefinition> _channelDefinitions = null;
    if (thingType!=null) {
      _channelDefinitions=thingType.getChannelDefinitions();
    }
    if (_channelDefinitions != null) {
      _elvis = _channelDefinitions;
    } else {
      ArrayList<ChannelDefinition> _newArrayList = CollectionLiterals.<ChannelDefinition>newArrayList();
      _elvis = _newArrayList;
    }
    final List<Channel> channels = this.createChannels(thingUID, _channels, _elvis);
    thingBuilder.withChannels(channels);
    final Thing thing = thingBuilder.build();
    thingList.add(thing);
    if ((modelThing instanceof ModelBridge)) {
      EList<ModelThing> _things = ((ModelBridge)modelThing).getThings();
      final Procedure1<ModelThing> _function = new Procedure1<ModelThing>() {
        public void apply(final ModelThing it) {
          GenericThingProvider.this.createThing(it, ((Bridge) thing), thingList);
        }
      };
      IterableExtensions.<ModelThing>forEach(_things, _function);
    }
  }
  
  private List<String> getParentPath(final Bridge bridge) {
    ThingUID _uID = bridge.getUID();
    List<String> bridgeIds = _uID.getBridgeIds();
    ThingUID _uID_1 = bridge.getUID();
    String _id = _uID_1.getId();
    bridgeIds.add(_id);
    return bridgeIds;
  }
  
  private List<Channel> createChannels(final ThingUID thingUID, final List<ModelChannel> modelChannels, final List<ChannelDefinition> channelDefinitions) {
    List<Channel> _xblockexpression = null;
    {
      final Set<String> addedChannelIds = CollectionLiterals.<String>newHashSet();
      final List<Channel> channels = CollectionLiterals.<Channel>newArrayList();
      final Procedure1<ModelChannel> _function = new Procedure1<ModelChannel>() {
        public void apply(final ModelChannel it) {
          String _id = it.getId();
          boolean _add = addedChannelIds.add(_id);
          if (_add) {
            String _id_1 = it.getId();
            ChannelUID _channelUID = new ChannelUID(thingUID, _id_1);
            String _type = it.getType();
            ChannelBuilder _create = ChannelBuilder.create(_channelUID, _type);
            Configuration _createConfiguration = GenericThingProvider.this.createConfiguration(it);
            ChannelBuilder _withConfiguration = _create.withConfiguration(_createConfiguration);
            Channel _build = _withConfiguration.build();
            channels.add(_build);
          }
        }
      };
      IterableExtensions.<ModelChannel>forEach(modelChannels, _function);
      final Procedure1<ChannelDefinition> _function_1 = new Procedure1<ChannelDefinition>() {
        public void apply(final ChannelDefinition it) {
          String _id = it.getId();
          boolean _add = addedChannelIds.add(_id);
          if (_add) {
            String _id_1 = it.getId();
            ChannelUID _channelUID = new ChannelUID(thingUID, _id_1);
            ChannelType _type = it.getType();
            String _itemType = _type.getItemType();
            ChannelBuilder _create = ChannelBuilder.create(_channelUID, _itemType);
            Channel _build = _create.build();
            channels.add(_build);
          }
        }
      };
      IterableExtensions.<ChannelDefinition>forEach(channelDefinitions, _function_1);
      _xblockexpression = channels;
    }
    return _xblockexpression;
  }
  
  private Configuration createConfiguration(final ModelPropertyContainer propertyContainer) {
    Configuration _xblockexpression = null;
    {
      final Configuration configuration = new Configuration();
      EList<ModelProperty> _properties = propertyContainer.getProperties();
      final Procedure1<ModelProperty> _function = new Procedure1<ModelProperty>() {
        public void apply(final ModelProperty it) {
          String _key = it.getKey();
          Object _value = it.getValue();
          configuration.put(_key, _value);
        }
      };
      IterableExtensions.<ModelProperty>forEach(_properties, _function);
      _xblockexpression = configuration;
    }
    return _xblockexpression;
  }
  
  private ThingType getThingType(final ThingTypeUID thingTypeUID) {
    ThingType _thingType = null;
    if (this.thingTypeRegistry!=null) {
      _thingType=this.thingTypeRegistry.getThingType(thingTypeUID);
    }
    return _thingType;
  }
  
  protected void setModelRepository(final ModelRepository modelRepository) {
    this.modelRepository = modelRepository;
    modelRepository.addModelRepositoryChangeListener(this);
  }
  
  protected void unsetModelRepository(final ModelRepository modelRepository) {
    modelRepository.removeModelRepositoryChangeListener(this);
    this.modelRepository = null;
  }
  
  public void modelChanged(final String modelName, final org.eclipse.smarthome.model.core.EventType type) {
    boolean _endsWith = modelName.endsWith("things");
    if (_endsWith) {
      if (type != null) {
        switch (type) {
          case ADDED:
            this.createThingsFromModel(modelName);
            Collection<Thing> _elvis = null;
            Collection<Thing> _get = this.thingsMap.get(modelName);
            if (_get != null) {
              _elvis = _get;
            } else {
              ArrayList<Thing> _newArrayList = CollectionLiterals.<Thing>newArrayList();
              _elvis = _newArrayList;
            }
            final Collection<Thing> things = _elvis;
            final Procedure1<Thing> _function = new Procedure1<Thing>() {
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutAddedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(things, _function);
            break;
          case MODIFIED:
            Collection<Thing> _elvis_1 = null;
            Collection<Thing> _get_1 = this.thingsMap.get(modelName);
            if (_get_1 != null) {
              _elvis_1 = _get_1;
            } else {
              ArrayList<Thing> _newArrayList_1 = CollectionLiterals.<Thing>newArrayList();
              _elvis_1 = _newArrayList_1;
            }
            final Collection<Thing> oldThings = _elvis_1;
            final Function1<Thing, ThingUID> _function_1 = new Function1<Thing, ThingUID>() {
              public ThingUID apply(final Thing it) {
                return it.getUID();
              }
            };
            Iterable<ThingUID> _map = IterableExtensions.<Thing, ThingUID>map(oldThings, _function_1);
            final List<ThingUID> oldThingsUIDs = IterableExtensions.<ThingUID>toList(_map);
            this.createThingsFromModel(modelName);
            Collection<Thing> _elvis_2 = null;
            Collection<Thing> _get_2 = this.thingsMap.get(modelName);
            if (_get_2 != null) {
              _elvis_2 = _get_2;
            } else {
              ArrayList<Thing> _newArrayList_2 = CollectionLiterals.<Thing>newArrayList();
              _elvis_2 = _newArrayList_2;
            }
            final Collection<Thing> currentThings = _elvis_2;
            final Function1<Thing, ThingUID> _function_2 = new Function1<Thing, ThingUID>() {
              public ThingUID apply(final Thing it) {
                return it.getUID();
              }
            };
            Iterable<ThingUID> _map_1 = IterableExtensions.<Thing, ThingUID>map(currentThings, _function_2);
            final List<ThingUID> currentThingUIDs = IterableExtensions.<ThingUID>toList(_map_1);
            final Function1<Thing, Boolean> _function_3 = new Function1<Thing, Boolean>() {
              public Boolean apply(final Thing it) {
                ThingUID _uID = it.getUID();
                boolean _contains = currentThingUIDs.contains(_uID);
                return Boolean.valueOf((!_contains));
              }
            };
            final Iterable<Thing> removedThings = IterableExtensions.<Thing>filter(oldThings, _function_3);
            final Function1<Thing, Boolean> _function_4 = new Function1<Thing, Boolean>() {
              public Boolean apply(final Thing it) {
                ThingUID _uID = it.getUID();
                boolean _contains = oldThingsUIDs.contains(_uID);
                return Boolean.valueOf((!_contains));
              }
            };
            final Iterable<Thing> addedThings = IterableExtensions.<Thing>filter(currentThings, _function_4);
            final Procedure1<Thing> _function_5 = new Procedure1<Thing>() {
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutRemovedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(removedThings, _function_5);
            final Procedure1<Thing> _function_6 = new Procedure1<Thing>() {
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutAddedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(addedThings, _function_6);
            final Procedure1<Thing> _function_7 = new Procedure1<Thing>() {
              public void apply(final Thing newThing) {
                final Procedure1<Thing> _function = new Procedure1<Thing>() {
                  public void apply(final Thing oldThing) {
                    ThingUID _uID = newThing.getUID();
                    ThingUID _uID_1 = oldThing.getUID();
                    boolean _equals = Objects.equal(_uID, _uID_1);
                    if (_equals) {
                      boolean _equals_1 = ThingHelper.equals(oldThing, newThing);
                      boolean _not = (!_equals_1);
                      if (_not) {
                        GenericThingProvider.this.notifyListenersAboutUpdatedElement(oldThing, newThing);
                      }
                    }
                  }
                };
                IterableExtensions.<Thing>forEach(oldThings, _function);
              }
            };
            IterableExtensions.<Thing>forEach(currentThings, _function_7);
            break;
          case REMOVED:
            Collection<Thing> _elvis_3 = null;
            Collection<Thing> _remove = this.thingsMap.remove(modelName);
            if (_remove != null) {
              _elvis_3 = _remove;
            } else {
              ArrayList<Thing> _newArrayList_3 = CollectionLiterals.<Thing>newArrayList();
              _elvis_3 = _newArrayList_3;
            }
            final Collection<Thing> things_1 = _elvis_3;
            final Procedure1<Thing> _function_8 = new Procedure1<Thing>() {
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutRemovedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(things_1, _function_8);
            break;
          default:
            break;
        }
      }
    }
  }
  
  protected void setThingTypeRegistry(final ThingTypeRegistry thingTypeRegistry) {
    this.thingTypeRegistry = thingTypeRegistry;
  }
  
  protected void unsetThingTypeRegistry(final ThingTypeRegistry thingTypeRegistry) {
    this.thingTypeRegistry = null;
  }
}
