angular.module('PaperUI.controllers.setup', []).controller('SetupPageController', function($scope, $location, thingTypeRepository, bindingRepository) {
    $scope.navigateTo = function(path) {
        $location.path('inbox/' + path);
    }
    $scope.thingTypes = [];
    function getThingTypes() {
        thingTypeRepository.getAll(function(thingTypes) {
            $.each(thingTypes, function(i, thingType) {
                $scope.thingTypes[thingType.UID] = thingType;
            });
        });
    }
    $scope.getThingTypeLabel = function(key) {
        if ($scope.thingTypes && Object.keys($scope.thingTypes).length != 0) {
            if ($scope.thingTypes[key]) {
                return $scope.thingTypes[key].label;
            } else {
                return "Unkown device";
            }
        } else {
            thingTypeRepository.setDirty(false);
        }
    };
    getThingTypes();
}).controller('InboxController', function($scope, $timeout, $mdDialog, $q, inboxService, discoveryResultRepository, thingTypeRepository, thingService, toastService) {
    $scope.setHeaderText('Shows a list of found things in your home.')

    $scope.showScanDialog = function(event) {
        $mdDialog.show({
            controller : 'ScanDialogController',
            templateUrl : 'partials/dialog.scan.html',
            targetEvent : event,
        });
    }

    $scope.refresh = function() {
        discoveryResultRepository.getAll(true);
    };
}).controller('InboxEntryController', function($scope, $mdDialog, $q, inboxService, discoveryResultRepository, thingTypeRepository, thingService, toastService, thingRepository) {
    $scope.approve = function(thingUID, thingTypeUID, event) {
        $mdDialog.show({
            controller : 'ApproveInboxEntryDialogController',
            templateUrl : 'partials/dialog.approveinboxentry.html',
            targetEvent : event,
            locals : {
                discoveryResult : discoveryResultRepository.find(function(discoveryResult) {
                    return discoveryResult.thingUID === thingUID;
                })
            }
        }).then(function(result) {
            inboxService.approve({
                'thingUID' : thingUID,
                'enableChannels' : !$scope.advancedMode
            }, result.label).$promise.then(function() {
                thingRepository.setDirty(true);
                toastService.showDefaultToast('Thing added.', 'Show Thing', 'configuration/things/view/' + thingUID);
                var thingType = thingTypeRepository.find(function(thingType) {
                    return thingTypeUID === thingType.UID;
                });

                if (thingType && thingType.bridge) {
                    $scope.navigateTo('setup/search/' + thingUID.split(':')[0]);
                } else {
                    discoveryResultRepository.getAll(true);
                }
            });
        });
    };
    $scope.ignore = function(thingUID) {
        inboxService.ignore({
            'thingUID' : thingUID
        }, function() {
            $scope.refresh();
        });
    };
    $scope.unignore = function(thingUID) {
        inboxService.unignore({
            'thingUID' : thingUID
        }, function() {
            $scope.refresh();
        });
    };
    $scope.remove = function(thingUID, event) {
        var discoveryResult = discoveryResultRepository.find(function(discoveryResult) {
            return discoveryResult.thingUID === thingUID;
        });
        var confirm = $mdDialog.confirm().title('Remove ' + discoveryResult.label).content('Would you like to remove the discovery result from the inbox?').ariaLabel('Remove Discovery Result').ok('Remove').cancel('Cancel').targetEvent(event);
        $mdDialog.show(confirm).then(function() {
            inboxService.remove({
                'thingUID' : thingUID
            }, function() {
                $scope.refresh();
                toastService.showSuccessToast('Inbox entry removed');
            });
        });
    };
}).controller('ScanDialogController', function($scope, $rootScope, $timeout, $mdDialog, discoveryService, bindingRepository) {
    $scope.supportedBindings = [];
    $scope.activeScans = [];

    $scope.scan = function(bindingId) {
        $scope.activeScans.push(bindingId);
        discoveryService.scan({
            'bindingId' : bindingId
        }, function() {

        });
        setTimeout(function() {
            $scope.$apply(function() {
                $scope.activeScans.splice($scope.activeScans.indexOf(bindingId), 1)
            });
        }, 3000);
    };

    bindingRepository.getAll();

    $scope.getBindingById = function(bindingId) {
        for (var i = 0; i < $rootScope.data.bindings.length; i++) {
            var binding = $rootScope.data.bindings[i];
            if (binding.id === bindingId) {
                return binding;
            }
        }
        return {};
    }

    discoveryService.getAll(function(response) {
        $scope.supportedBindings = response;
    });

    $scope.close = function() {
        $mdDialog.hide();
    }
}).controller('ApproveInboxEntryDialogController', function($scope, $mdDialog, discoveryResult, thingTypeRepository) {
    $scope.discoveryResult = discoveryResult;
    $scope.label = discoveryResult.label;
    $scope.thingType = null;
    $scope.thingTypeUID = discoveryResult.thingTypeUID;
    thingTypeRepository.getOne(function(thingType) {
        return thingType.UID === $scope.thingTypeUID;
    }, function(thingType) {
        $scope.thingType = thingType;
    });

    $scope.close = function() {
        $mdDialog.cancel();
    }
    $scope.approve = function(label) {
        var selectedGroupNames = [];
        for ( var groupName in $scope.groupNames) {
            if ($scope.groupNames[groupName]) {
                selectedGroupNames.push(groupName);
            }
        }
        $mdDialog.hide({
            label : label,
            groupNames : selectedGroupNames
        });
    }
}).controller('ManualSetupChooseController', function($scope, bindingRepository, thingTypeRepository, thingService) {
    $scope.setSubtitle([ 'Manual Setup' ]);
    $scope.setHeaderText('Choose a thing, which should be aded manually to your Smart Home.')

    $scope.currentBindingId = null;
    $scope.setCurrentBindingId = function(bindingId) {
        $scope.currentBindingId = bindingId;
    };

    bindingRepository.getAll(function(data) {
    });

}).controller('ManualSetupConfigureController', function($scope, $routeParams, $mdDialog, toastService, bindingRepository, thingTypeRepository, thingService, thingRepository, configService, linkService) {

    var thingTypeUID = $routeParams.thingTypeUID;

    function generateUUID() {
        var d = new Date().getTime();
        var uuid = 'xxxxxxxx'.replace(/[x]/g, function(c) {
            var r = (d + Math.random() * 16) % 16 | 0;
            d = Math.floor(d / 16);
            return (c == 'x' ? r : (r & 0x3 | 0x8)).toString(16);
        });
        return uuid;
    }
    ;

    $scope.thingType = null;
    $scope.thing = {
        UID : null,
        configuration : {},
        item : {
            label : null,
            groupNames : []
        }
    };
    $scope.thingID = null;

    $scope.addThing = function(thing) {
        thing.thingTypeUID = thingTypeUID;
        thing.UID = thing.thingTypeUID + ":" + thing.ID;
        thingService.add(thing, function() {
            toastService.showDefaultToast('Thing added.', 'Show Thing', 'configuration/things/view/' + thing.UID);
            $scope.navigateTo('setup/search/' + $scope.thingType.UID.split(':')[0]);
        });
    };

    function linkChannel(channelUID) {
        var itemName = channelUID.replace(/:/g, "_");
        linkService.link({
            itemName : itemName,
            channelUID : channelUID
        });
    }

    $scope.needsBridge = false;
    $scope.bridges = [];
    $scope.getBridges = function() {
        $scope.bridges = [];
        thingRepository.getAll(function(things) {
            for (var i = 0; i < things.length; i++) {
                var thing = things[i];
                for (var j = 0; j < $scope.thingType.supportedBridgeTypeUIDs.length; j++) {
                    var supportedBridgeTypeUID = $scope.thingType.supportedBridgeTypeUIDs[j];
                    if (thing.thingTypeUID === supportedBridgeTypeUID) {
                        $scope.bridges.push(thing);
                    }
                }
            }
        });
    };

    thingTypeRepository.getOne(function(thingType) {
        return thingType.UID === thingTypeUID;
    }, function(thingType) {
        $scope.setTitle('Configure ' + thingType.label);
        $scope.setHeaderText(thingType.description);
        $scope.thingType = thingType;
        $scope.parameters = configService.getRenderingModel(thingType.configParameters, thingType.parameterGroups);
        $scope.thing.ID = generateUUID();
        $scope.thing.item.label = thingType.label;
        $scope.thing.label = thingType.label;
        $scope.needsBridge = $scope.thingType.supportedBridgeTypeUIDs && $scope.thingType.supportedBridgeTypeUIDs.length > 0;
        if ($scope.needsBridge) {
            $scope.getBridges();
        }
        configService.setDefaults($scope.thing, $scope.thingType)
    });
}).controller('SetupWizardController', function($scope, discoveryResultRepository) {
    $scope.showIgnored = false;
    $scope.toggleShowIgnored = function() {
        $scope.showIgnored = !$scope.showIgnored;
    }
    $scope.refresh = function() {
        discoveryResultRepository.getAll(true);
    };
    $scope.refresh();
    $scope.filter = function(discoveryResult) {
        return $scope.showIgnored || discoveryResult.flag === 'NEW';
    }
}).controller('SetupWizardBindingsController', function($scope, bindingRepository, discoveryService) {
    $scope.setSubtitle([ 'Choose Binding' ]);
    $scope.setHeaderText('Choose a Binding for which you want to add new things.');
    bindingRepository.getAll();
    $scope.selectBinding = function(bindingId) {
        discoveryService.getAll(function(supportedBindings) {
            if (supportedBindings.indexOf(bindingId) >= 0) {
                $scope.navigateTo('setup/search/' + bindingId);
            } else {
                $scope.navigateTo('setup/thing-types/' + bindingId);
            }
        });
    }
}).controller('SetupWizardSearchBindingController', function($scope, discoveryResultRepository, discoveryService, thingTypeRepository, bindingRepository) {
    $scope.showIgnored = false;
    $scope.toggleShowIgnored = function() {
        $scope.showIgnored = !$scope.showIgnored;
    }
    $scope.bindingId = $scope.path[4];
    var binding = bindingRepository.find(function(binding) {
        return binding.id === $scope.bindingId;
    });
    $scope.setSubtitle([ binding ? binding.name : '', 'Search' ]);
    $scope.setHeaderText('Searching for new things for the ' + (binding ? binding.name : '') + '.');

    $scope.discoverySupported = true;
    discoveryService.getAll(function(supportedBindings) {
        if (supportedBindings.indexOf($scope.bindingId) >= 0) {
            $scope.discoverySupported = true;
            $scope.scan($scope.bindingId);
        }
    });

    $scope.scanning = false;
    $scope.filter = function(discoveryResult) {
        return ($scope.showIgnored || discoveryResult.flag === 'NEW') && discoveryResult.thingUID.split(':')[0] === $scope.bindingId;
    }
    $scope.scan = function(bindingId) {
        $scope.scanning = true;
        discoveryService.scan({
            'bindingId' : bindingId
        }, function() {
        });
        setTimeout(function() {
            $scope.$apply(function() {
                $scope.scanning = false;
            });
        }, 10000);
    };

    $scope.refresh = function() {
        discoveryResultRepository.getAll(true);
    };
    $scope.refresh();
}).controller('SetupWizardThingTypesController', function($scope, bindingRepository) {
    $scope.bindingId = $scope.path[4];
    var binding = bindingRepository.find(function(binding) {
        return binding.id === $scope.bindingId;
    });
    $scope.setSubtitle([ binding ? binding.name : '', 'Choose Thing' ]);
    $scope.setHeaderText('Choose a Thing from the ' + (binding ? binding.name : '') + ' which you want to add.');

    $scope.selectThingType = function(thingTypeUID) {
        $scope.navigateTo('setup/add/' + thingTypeUID);
    }
    $scope.filter = function(thingType) {
        return (thingType.UID.split(':')[0] === $scope.bindingId) && (thingType.listed);
    }
});
'use strict';

angular.module('PaperUI.controllers.rules', []).controller('RulesPageController', function($scope, $location, $mdDialog) {
    $scope.navigateTo = function(path) {
        $location.path('rules/' + path);
    };

    $scope.openDialog = function(ctrl, url, params) {
        $mdDialog.show({
            controller : ctrl,
            templateUrl : url,
            targetEvent : params.event,
            hasBackdrop : true,
            locals : params
        });
    };

    $scope.getRuleJSON = function(sharedProperties, uid, name, desc) {
        var rule = {
            tags : [],
            conditions : sharedProperties.getModuleJSON('condition'),
            description : desc,
            name : name,
            triggers : sharedProperties.getModuleJSON('trigger'),
            configDescriptions : [],
            actions : sharedProperties.getModuleJSON('action')
        };

        if (uid) {
            rule.uid = uid;
        }

        return rule;
    }
}).controller('RulesController', function($scope, $timeout, ruleRepository, ruleService, toastService, sharedProperties) {
    $scope.setHeaderText('Shows all rules.');

    $scope.refresh = function() {
        ruleRepository.getAll(true);
    };

    $scope.configure = function(rule) {
        $scope.navigateTo('configure/' + rule.uid);
    };

    $scope.remove = function(rule, e) {
        e.stopImmediatePropagation();
        ruleService.remove({
            ruleUID : rule.uid
        }, function() {
            $scope.refresh();
            toastService.showDefaultToast('Rule removed.');
        });
    };

    ruleRepository.getAll(true);

    $scope.removePart = function(opt, id) {
        sharedProperties.removeFromArray(opt, id);
    };
    $scope.toggleEnabled = function(rule, e) {
        e.stopImmediatePropagation();
        ruleService.setEnabled({
            ruleUID : rule.uid
        }, (!rule.enabled).toString(), function() {
            $scope.refresh();
            if (rule.enabled) {
                toastService.showDefaultToast('Rule disabled.');
            } else {
                toastService.showDefaultToast('Rule enabled.');
            }
        });
    };

    $scope.ruleOptionSelected = function(event, value) {
        if (value == 0) {
            $scope.navigateTo('new');
        } else {
            $scope.openDialog('RuleTemplateController', 'partials/dialog.ruletemplate.html', {
                event : event
            });
        }
    };
}).controller('ViewRuleController', function($scope, ruleRepository) {
    var ruleUID = $scope.path[3];
    ruleRepository.getOne(function(rule) {
        return rule.uid === ruleUID;
    }, function(rule) {
        $scope.setSubtitle([ rule.name ]);
        $scope.rule = rule;
    });
}).controller('NewRuleController', function($scope, itemRepository, ruleService, toastService, $mdDialog, sharedProperties, moduleTypeService) {
    $scope.setSubtitle([ 'New Rule' ]);
    itemRepository.getAll();
    sharedProperties.reset();
    $scope.editing = false;
    var ruleUID = $scope.path[3];

    if ($scope.path[3]) {
        ruleService.getByUid({
            ruleUID : ruleUID
        }, function(data) {
            $scope.name = data.name;
            $scope.description = data.description;
            setModuleArrays(data);
        });
        $scope.setSubtitle([ 'Configure' ]);
        $scope.editing = true;
    } else if (sharedProperties.getParams().length > 0 && sharedProperties.getParams()[0]) {
        $scope.name = sharedProperties.getParams()[0].label;
        $scope.description = sharedProperties.getParams()[0].description;
        setModuleArrays(sharedProperties.getParams()[0]);
    }

    function setModuleArrays(data) {
        moduleTypeService.getByType({
            mtype : 'trigger'
        }).$promise.then(function(moduleData) {
            sharedProperties.setModuleTypes(moduleData);
            sharedProperties.addArray('trigger', data.triggers);
            sharedProperties.addArray('action', data.actions);
            sharedProperties.addArray('condition', data.conditions);
        });
    }

    $scope.saveUserRule = function() {

        var rule = $scope.getRuleJSON(sharedProperties, null, $scope.name, $scope.description);
        ruleService.add(rule).$promise.then(function() {
            toastService.showDefaultToast('Rule added.');
            $scope.navigateTo('');
        });

    };

    $scope.updateUserRule = function() {

        var rule = $scope.getRuleJSON(sharedProperties, $scope.path[3], $scope.name, $scope.description);
        ruleService.update({
            ruleUID : $scope.path[3]
        }, rule).$promise.then(function() {
            toastService.showDefaultToast('Rule updated.');
            $scope.navigateTo('');
        });
    };

    $scope.openNewModuleDialog = function(event, type) {
        $scope.openDialog('addModuleDialogController', 'partials/dialog.addmodule.html', {
            event : event,
            module : {},
            ruleID : $scope.path[3] || '',
            type : type
        });
    };

    $scope.openUpdateModuleDialog = function(event, type, module) {
        $scope.openDialog('addModuleDialogController', 'partials/dialog.addmodule.html', {
            event : event,
            module : module,
            ruleID : $scope.path[3] || '',
            type : type
        });
    };

    $scope.aTriggers = sharedProperties.getTriggersArray();
    $scope.aActions = sharedProperties.getActionsArray();
    $scope.aConditions = sharedProperties.getConditionsArray();

    $scope.sortableOptions = {
        handle : '.draggable',
        update : function(e, ui) {
        },
        axis : 'y'
    };

}).controller('RuleConfigureController', function($scope, ruleRepository, ruleService, toastService) {
    $scope.setSubtitle([ 'Configure' ]);
    var ruleUID = $scope.path[3];

    ruleRepository.getOne(function(rule) {
        return rule.uid === ruleUID;
    }, function(rule) {
        $scope.setSubtitle([ 'Configure ' + rule.name ]);
    });

    ruleService.getModuleConfigParameter({
        ruleUID : ruleUID
    }, function(data) {
        $scope.script = data.content;
    });

    $scope.save = function() {
        ruleService.setModuleConfigParameter({
            ruleUID : ruleUID
        }, $scope.script, function() {
            toastService.showDefaultToast('Rule updated successfully.');
            $scope.navigateTo('');
        });
    };
}).controller('RuleTemplateController', function($scope, $location, ruleService, configService, toastService, $mdDialog, sharedProperties) {
    $scope.templateData = ruleService.getRuleTemplates();
    $scope.templateStep = 1;

    $scope.selectTemplate = function() {
        var res = configService.getRenderingModel($scope.templateData[$scope.templateIndex].configDescriptions);
        $scope.name = $scope.templateData[$scope.templateIndex].label;
        $scope.description = $scope.templateData[$scope.templateIndex].description;
        angular.forEach(res, function(value) {
            sharedProperties.updateParams(value);
        });
        $scope.templateStep = 2;
        $scope.parameters = sharedProperties.getParams();
        $scope.configuration = configService.setConfigDefaults({}, $scope.parameters);
    };

    $scope.saveRule = function() {
        sharedProperties.resetParams();
        $scope.configuration = configService.replaceEmptyValues($scope.configuration);
        var rule = {
            templateUID : $scope.templateData[$scope.templateIndex].uid,
            name : $scope.name,
            description : $scope.description,
            configuration : $scope.configuration
        };
        ruleService.add(rule).$promise.then(function() {
            toastService.showDefaultToast('Rule added.');
            $mdDialog.hide();
            $location.path('rules/');
        });

    };

    $scope.close = function() {
        sharedProperties.resetParams();
        $mdDialog.hide();
    };
}).directive('dragdrop', function() {
    return {
        restrict : 'AE',
        replace : true,
        template : '<span class="draggable md-icon-reorder"></span>',
        link : function(scope, elem, attrs) {

            var touchHandler = function(event) {
                var touch = event.changedTouches[0];
                var simulatedEvent = document.createEvent("MouseEvent");
                simulatedEvent.initMouseEvent({
                    touchstart : "mousedown",
                    touchmove : "mousemove",
                    touchend : "mouseup"
                }[event.type], true, true, window, 1, touch.screenX, touch.screenY, touch.clientX, touch.clientY, false, false, false, false, 0, null);

                touch.target.dispatchEvent(simulatedEvent);
                event.preventDefault();
            };
            elem[0].addEventListener("touchstart", touchHandler, true);
            elem[0].addEventListener("touchmove", touchHandler, true);
            elem[0].addEventListener("touchend", touchHandler, true);
            elem[0].addEventListener("touchcancel", touchHandler, true);
        }
    };
}).directive('scriptarea', function() {
    return {
        restrict : 'A',
        require : 'ngModel',
        link : function(scope, elem, attrs, ngModel) {
            elem.ready(function() {
                setTimeout(function() {
                    elem[0].style.cssText = 'height:auto;';
                    elem[0].style.cssText = 'height:' + elem[0].scrollHeight + 'px';
                }, 500);
            });
            var localAttrs = attrs;
            var element = elem;
            var resizeHandler = function(event) {
                elem[0].style.cssText = 'height:auto;';
                if (elem[0].value.length < 1) {
                    elem[0].style.cssText = 'height:35px';
                } else {
                    elem[0].style.cssText = 'height:' + elem[0].scrollHeight + 'px';
                }
                validateAtrribute();
            };
            elem[0].addEventListener("keydown", resizeHandler, true);
            elem[0].addEventListener("input", resizeHandler, true);
            elem[0].addEventListener("cut", resizeHandler);
            elem[0].addEventListener("blur", function() {
                validateAtrribute();
            });
            function validateAtrribute() {
                var modelArr = document.getElementsByName(attrs.name);
                if (modelArr && modelArr.length > 0) {
                    var modelValue = modelArr[0].value;
                    if (modelValue && (modelValue.length < localAttrs.ngMinlength || modelValue.length > localAttrs.ngMaxlength)) {
                        element.addClass('border-invalid');
                    } else if ((modelValue === undefined || modelValue == "") && localAttrs.ngRequired) {
                        element.addClass('border-invalid');
                    } else {
                        element.removeClass('border-invalid');
                    }
                }
            }
        }
    }
});

'use strict';

angular.module('PaperUI.controllers.rules').controller('addModuleDialogController', function($rootScope, $scope, $mdDialog, moduleTypeService, sharedProperties, $filter, configService, module, ruleID, type) {

    var objectFilter = $filter('filter');
    $scope.moduleData;
    moduleTypeService.getByType({
        mtype : type
    }).$promise.then(function(data) {
        $scope.moduleData = objectFilter(data, {
            visibility : 'VISIBLE'
        });
        if ($scope.id) {
            setConfigurations();
        }
    });
    $scope.id = module.id;
    $scope.type = type;
    $scope.description = '';
    $scope.module = '';
    $scope.step = 1;
    $scope.editMode = false;
    $scope.configuration = {};
    var originalConfiguration = {};
    function setConfigurations() {
        if ($scope.moduleData) {
            var params = filterByUid($scope.moduleData, $scope.module);
            var res = configService.getRenderingModel(params[0].configDescriptions);
            angular.forEach(res, function(value) {
                sharedProperties.updateParams(value);
            });

            var index = sharedProperties.searchArray(sharedProperties.getModuleArray(type), $scope.id);
            if (index != -1) {
                $scope.configuration = configService.convertValues(sharedProperties.getModuleArray(type)[index].configuration);
                angular.copy($scope.configuration, originalConfiguration);
                $scope.configArray = configService.getConfigAsArray($scope.configuration);
            }
        }
    }

    if ($scope.id) {
        $scope.editMode = true;
        $scope.module = module.type;
        $scope.name = module.label;
        $scope.description = module.description;
        setConfigurations();
        $scope.step = 2;
    }

    $scope.parameters = sharedProperties.getParams();

    $scope.close = function() {
        sharedProperties.resetParams();
        var index = sharedProperties.searchArray(sharedProperties.getModuleArray(type), $scope.id);
        if (index != -1) {
            sharedProperties.getModuleArray(type)[index].configuration = originalConfiguration;
        }
        $mdDialog.hide();
    };

    $scope.saveModule = function() {
        var tempModule = filterByUid($scope.moduleData, $scope.module);
        if (tempModule != null && tempModule.length > 0) {
            tempModule[0].label = $scope.name;
            $scope.configuration = configService.replaceEmptyValues($scope.configuration);
            var obj = {
                id : $scope.id,
                label : $scope.name,
                description : $scope.description,
                type : tempModule[0].uid,
                configuration : $scope.configuration
            };
            sharedProperties.updateModule($scope.type, obj);
        }
        sharedProperties.resetParams();
        $mdDialog.hide();
    };

    $scope.deleteModule = function(opt) {
        sharedProperties.removeFromArray(opt, $scope.id);
        sharedProperties.resetParams();
        $mdDialog.hide();
    };

    $scope.secondStep = function() {
        var tempModule = filterByUid($scope.moduleData, $scope.module);
        if (tempModule != null && tempModule.length > 0) {
            $scope.name = tempModule[0].label;
            $scope.description = tempModule[0].description;
            setConfigurations();
            $scope.step = 2;
        }

    };

    function filterByUid(data, uid) {
        return objectFilter(data, {
            uid : uid
        });
    }

});
angular.module('PaperUI.controllers', [ 'PaperUI.constants' ]).controller('BodyController', function($rootScope, $scope, $http, eventService, toastService, discoveryResultRepository, thingTypeRepository, bindingRepository, restConfig) {
    $scope.scrollTop = 0;
    $(window).scroll(function() {
        $scope.$apply(function(scope) {
            $scope.scrollTop = $('body').scrollTop();
        });
    });
    $scope.isBigTitle = function() {
        return $scope.scrollTop < 80 && !$rootScope.simpleHeader;
    }
    $scope.setTitle = function(title) {
        $rootScope.title = title;
    }
    $scope.subtitles = [];
    $scope.setSubtitle = function(args) {
        $scope.subtitles = [];
        $.each(args, function(i, subtitle) {
            $scope.subtitles.push(subtitle);
        })
    }
    $scope.setHeaderText = function(headerText) {
        $scope.headerText = headerText;
    }
    $rootScope.$on('$routeChangeStart', function() {
        $scope.subtitles = [];
        $scope.headerText = null;
    });
    $scope.generateUUID = function() {
        var d = new Date().getTime();
        var uuid = 'xxxxxxxx'.replace(/[x]/g, function(c) {
            var r = (d + Math.random() * 16) % 16 | 0;
            d = Math.floor(d / 16);
            return (c == 'x' ? r : (r & 0x3 | 0x8)).toString(16);
        });
        return uuid;
    };

    var numberOfInboxEntries = -1;
    eventService.onEvent('smarthome/inbox/*/added', function(topic, discoveryResult) {
        toastService.showDefaultToast('New Inbox Entry: ' + discoveryResult.label, 'Show Inbox', 'inbox/setup');
    });
    eventService.onEvent('smarthome/items/*/state', function(topic, stateObject) {

        var itemName = topic.split('/')[2];
        var state = stateObject.value;

        console.log('Item ' + itemName + ' updated: ' + state);

        if ($rootScope.itemUpdates[itemName] + 500 > new Date().getTime()) {
            console.log('Ignoring update for ' + itemName + ', because update was probably triggered through UI.');
            return;
        }

        var changeStateRecursively = function(item) {
            var updateState = true;
            if (item.name === itemName) {
                // ignore ON and OFF update for Dimmer
                if (item.type === 'DimmerItem') {
                    if (state === 'ON' || state == 'OFF') {
                        updateState = false;
                    }
                }

                if (updateState) {
                    $scope.$apply(function(scope) {
                        item.state = state;
                    });
                } else {
                    console.log('Ignoring state ' + state + ' for ' + itemName)
                }
            }
            if (item.members) {
                $.each(item.members, function(i, memberItem) {
                    changeStateRecursively(memberItem);
                });
            }
        }

        if ($rootScope.data.items) {
            $.each($rootScope.data.items, function(i, item) {
                changeStateRecursively(item);
            });
        }
    });

    $scope.getNumberOfNewDiscoveryResults = function() {
        var numberOfNewDiscoveryResults = 0;
        if (!$scope.data.discoveryResults) {
            return numberOfNewDiscoveryResults;
        }
        for (var i = 0; i < $scope.data.discoveryResults.length; i++) {
            var discoveryResult = $scope.data.discoveryResults[i];
            if (discoveryResult.flag === 'NEW') {
                numberOfNewDiscoveryResults++;
            }
        }
        return numberOfNewDiscoveryResults;
    }

    $http.get(restConfig.restPath + "/links/auto").then(function(response) {
        if (response.data !== undefined) {
            $rootScope.advancedMode = !response.data;
            window.localStorage.setItem('paperui.advancedMode', !response.data);
        }
    });

    discoveryResultRepository.getAll();
    thingTypeRepository.getAll();
    bindingRepository.getAll();
}).controller('PreferencesPageController', function($rootScope, $scope, toastService) {
    $scope.setHeaderText('Edit user preferences.');

    var localStorage = window.localStorage;
    var language = localStorage.getItem('paperui.language');

    $scope.language = language ? language : 'english';
    $scope.advancedMode = $rootScope.advancedMode;
    $scope.save = function() {
        $rootScope.advancedMode = $scope.advancedMode;
        localStorage.setItem('paperui.language', $scope.language);
        localStorage.setItem('paperui.advancedMode', $rootScope.advancedMode);
        toastService.showSuccessToast('Preferences saved successfully. Please reload the page.');
    }

    $scope.getSelected = function(property) {
        return $('select#' + property + ' option:selected').val();
    }
}).controller('NavController', function($scope, $location, $http, restConfig, moduleConfig) {
    $scope.opened = null;
    $scope.extensionEnabled;
    $scope.ruleEnabled;
    $scope.open = function(viewLocation) {
        $scope.opened = viewLocation;
    }
    $scope.isActive = function(viewLocation) {
        var active = (viewLocation === $location.path().split('/')[1]);
        return active || $scope.opened === viewLocation;
    }
    $scope.isSubActive = function(viewLocation) {
        var active = (viewLocation === $location.path().split('/')[2]);
        return active;
    }
    $scope.isHidden = function(module) {
        return moduleConfig[module] === false;
    }
    $scope.$on('$routeChangeSuccess', function() {
        $('body').removeClass('sml-open');
        $('.mask').remove();
        $scope.opened = null;
    });
    $http.get(restConfig.restPath).then(function(response) {
        $scope.extensionEnabled = false;
        $scope.ruleEnabled = false;
        if (response.data && response.data.links) {
            for (var i = 0; i < response.data.links.length; i++) {
                if (response.data.links[i].type === 'extensions') {
                    $scope.extensionEnabled = true;
                } else if (response.data.links[i].type === 'rules') {
                    $scope.ruleEnabled = true;
                }
            }
        }
    });
});
angular.module('PaperUI.controllers.configuration', [ 'PaperUI.constants' ]).controller('ConfigurationPageController', function($scope, $location, thingTypeRepository) {
    $scope.navigateTo = function(path) {
        $location.path('configuration/' + path);
    }
    $scope.thingTypes = [];
    function getThingTypes() {
        thingTypeRepository.getAll(function(thingTypes) {
            $.each(thingTypes, function(i, thingType) {
                $scope.thingTypes[thingType.UID] = thingType;
            });
        });
    }
    $scope.getThingTypeLabel = function(key) {
        if ($scope.thingTypes && Object.keys($scope.thingTypes).length != 0) {
            if ($scope.thingTypes[key]) {
                return $scope.thingTypes[key].label;
            } else {
                return '';
            }
        } else {
            thingTypeRepository.setDirty(false);
        }
    };
    getThingTypes();
}).controller('BindingController', function($scope, $mdDialog, bindingRepository) {
    $scope.setSubtitle([ 'Bindings' ]);
    $scope.setHeaderText('Shows all installed bindings.');
    $scope.refresh = function() {
        bindingRepository.getAll(true);
    };
    $scope.openBindingInfoDialog = function(bindingId, event) {
        $mdDialog.show({
            controller : 'BindingInfoDialogController',
            templateUrl : 'partials/dialog.bindinginfo.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                bindingId : bindingId
            }
        });
    }
    $scope.configure = function(bindingId, configDescriptionURI, event) {
        $mdDialog.show({
            controller : 'ConfigureBindingDialogController',
            templateUrl : 'partials/dialog.configurebinding.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                bindingId : bindingId,
                configDescriptionURI : configDescriptionURI
            }
        });
    }
    bindingRepository.getAll();
}).controller('BindingInfoDialogController', function($scope, $mdDialog, thingTypeRepository, bindingRepository, bindingId) {
    $scope.binding = undefined;
    bindingRepository.getOne(function(binding) {
        return binding.id === bindingId;
    }, function(binding) {
        $scope.binding = binding;
        $scope.binding.thingTypes = [];
        thingTypeRepository.getAll(function(thingTypes) {
            $.each(thingTypes, function(index, thingType) {
                if (thingType.UID.split(':')[0] === binding.id) {
                    $scope.binding.thingTypes.push(thingType);
                }
            });
        });
    });
    $scope.close = function() {
        $mdDialog.hide();
    }
}).controller('ConfigureBindingDialogController', function($scope, $mdDialog, bindingRepository, bindingService, configService, configDescriptionService, toastService, bindingId, configDescriptionURI) {

    $scope.binding = null;
    $scope.parameters = [];
    $scope.config = {};

    if (configDescriptionURI) {
        $scope.expertMode = false;
        configDescriptionService.getByUri({
            uri : configDescriptionURI
        }, function(configDescription) {
            if (configDescription) {
                $scope.parameters = configService.getRenderingModel(configDescription.parameters, configDescription.parameterGroups);
            }
        });
    }
    if (bindingId) {
        bindingRepository.getOne(function(binding) {
            return binding.id === bindingId;
        }, function(binding) {
            $scope.binding = binding;
        });
        bindingService.getConfigById({
            id : bindingId
        }).$promise.then(function(config) {
            $scope.configuration = configService.convertValues(config);
            $scope.configArray = configService.getConfigAsArray($scope.configuration);
        }, function(failed) {
            $scope.configuration = {};
            $scope.configArray = configService.getConfigAsArray($scope.configuration);
        });
    } else {
        $scope.newConfig = true;
        $scope.serviceId = '';
        $scope.configuration = {
            '' : ''
        };
        $scope.configArray = [];
        $scope.expertMode = true;
    }
    $scope.close = function() {
        $mdDialog.hide();
    }
    $scope.addParameter = function() {
        $scope.configArray.push({
            name : '',
            value : undefined
        });
    }
    $scope.save = function() {
        if ($scope.expertMode) {
            $scope.configuration = configService.getConfigAsObject($scope.configArray, $scope.parameters);
        }
        $scope.configuration = configService.replaceEmptyValues($scope.configuration);
        bindingService.updateConfig({
            id : bindingId
        }, $scope.configuration, function() {
            $mdDialog.hide();
            toastService.showDefaultToast('Binding config updated.');
        });
    }
    $scope.$watch('expertMode', function() {
        if ($scope.expertMode) {
            $scope.configArray = configService.getConfigAsArray($scope.configuration);
        } else {
            $scope.configuration = configService.getConfigAsObject($scope.configArray, $scope.parameters);
        }
    });
}).controller('ServicesController', function($scope, $mdDialog, serviceConfigService, toastService) {
    $scope.setSubtitle([ 'Services' ]);
    $scope.setHeaderText('Shows all configurable services.');
    $scope.tabs = [];
    $scope.refresh = function() {
        serviceConfigService.getAll(function(services) {
            // $scope.services = services;
            var arrOfIndex = [];
            var index = 0;
            angular.forEach(services, function(value) {
                if (arrOfIndex[value.category] === undefined) {
                    arrOfIndex[value.category] = index++;
                }
                if ($scope.tabs[arrOfIndex[value.category]] === undefined) {
                    $scope.tabs[arrOfIndex[value.category]] = [];
                    $scope.tabs[arrOfIndex[value.category]].category = value.category;
                }
                $scope.tabs[arrOfIndex[value.category]].push(value);
            });
        });
    };
    $scope.add = function(serviceId, event) {
        $mdDialog.show({
            controller : 'ConfigureServiceDialogController',
            templateUrl : 'partials/dialog.configureservice.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                serviceId : undefined,
                configDescriptionURI : undefined
            }
        }).then(function() {
            $scope.refresh();
        });
    }
    $scope.configure = function(serviceId, configDescriptionURI, event) {
        $mdDialog.show({
            controller : 'ConfigureServiceDialogController',
            templateUrl : 'partials/dialog.configureservice.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                serviceId : serviceId,
                configDescriptionURI : configDescriptionURI
            }
        });
    }
    $scope.refresh();
}).controller('ConfigureServiceDialogController', function($scope, $mdDialog, configService, serviceConfigService, configDescriptionService, toastService, serviceId, configDescriptionURI) {

    $scope.service = null;
    $scope.parameters = [];
    $scope.config = {};

    if (configDescriptionURI) {
        $scope.expertMode = false;
        configDescriptionService.getByUri({
            uri : configDescriptionURI
        }, function(configDescription) {
            if (configDescription) {
                $scope.parameters = configService.getRenderingModel(configDescription.parameters, configDescription.parameterGroups);
                if (!jQuery.isEmptyObject($scope.configuration)) {
                    $scope.configuration = configService.setConfigDefaults($scope.configuration, $scope.parameters);
                }
            }
        });
    }
    if (serviceId) {
        serviceConfigService.getById({
            id : serviceId
        }, function(service) {
            $scope.service = service;
        });
        serviceConfigService.getConfigById({
            id : serviceId
        }).$promise.then(function(config) {
            if (config) {
                $scope.configuration = configService.convertValues(config);
                $scope.configArray = configService.getConfigAsArray($scope.configuration);
                if ($scope.parameters && $scope.parameters.length > 0) {
                    $scope.configuration = configService.setConfigDefaults($scope.configuration, $scope.parameters);
                }
            }
        });
    } else {
        $scope.newConfig = true;
        $scope.serviceId = '';
        $scope.configuration = {
            '' : ''
        };
        $scope.configArray = [];
        $scope.expertMode = true;
    }
    $scope.close = function() {
        $mdDialog.hide();
    }
    $scope.addParameter = function() {
        $scope.configArray.push({
            name : '',
            value : undefined
        });
    }
    $scope.save = function() {
        if ($scope.expertMode) {
            $scope.configuration = configService.getConfigAsObject($scope.configArray, $scope.parameters);
        }
        $scope.configuration = configService.setConfigDefaults($scope.configuration, $scope.parameters);
        serviceConfigService.updateConfig({
            id : (serviceId ? serviceId : $scope.serviceId)
        }, $scope.configuration, function() {
            $mdDialog.hide();
            toastService.showDefaultToast('Service config updated.');
        });
    }
    $scope.$watch('expertMode', function() {
        if ($scope.expertMode) {
            $scope.configArray = configService.getConfigAsArray($scope.configuration);
        } else {
            $scope.configuration = configService.getConfigAsObject($scope.configArray, $scope.parameters);
        }
    });
}).controller('AddGroupDialogController', function($scope, $mdDialog) {
    $scope.binding = undefined;

    $scope.close = function() {
        $mdDialog.cancel();
    }
    $scope.add = function(label) {
        $mdDialog.hide(label);
    }
}).controller('ThingController', function($scope, $timeout, $mdDialog, thingRepository, thingService, toastService) {
    $scope.setSubtitle([ 'Things' ]);
    $scope.setHeaderText('Shows all configured Things.');
    $scope.refresh = function() {
        thingRepository.getAll(true);
    }
    $scope.remove = function(thing, event) {
        event.stopImmediatePropagation();
        $mdDialog.show({
            controller : 'RemoveThingDialogController',
            templateUrl : 'partials/dialog.removething.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                thing : thing
            }
        }).then(function() {
            $scope.refresh();
        });
    }
    $scope.refresh();
}).controller('ViewThingController', function($scope, $mdDialog, toastService, thingTypeRepository, thingRepository, thingService, linkService, channelTypeService, configService, thingConfigService, util) {

    var thingUID = $scope.path[4];
    $scope.thingTypeUID = null;

    $scope.thing;
    $scope.thingType;
    $scope.thingChannels = [];
    $scope.showAdvanced = false;
    $scope.channelTypes;
    channelTypeService.getAll().$promise.then(function(channels) {
        $scope.channelTypes = channels;
        $scope.refreshChannels(false);
    });
    $scope.edit = function(thing, event) {
        $mdDialog.show({
            controller : 'EditThingDialogController',
            templateUrl : 'partials/dialog.editthing.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                thing : thing
            }
        });
    };
    $scope.remove = function(thing, event) {
        event.stopImmediatePropagation();
        $mdDialog.show({
            controller : 'RemoveThingDialogController',
            templateUrl : 'partials/dialog.removething.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                thing : thing
            }
        }).then(function() {
            $scope.navigateTo('things');
        });
    }

    $scope.enableChannel = function(thingUID, channelID, event) {
        var channel = $scope.getChannelById(channelID);
        if ($scope.advancedMode) {
            $scope.linkChannel(channelID, event);
        } else {
            linkService.link({
                itemName : $scope.thing.UID.replace(/[^a-zA-Z0-9_]/g, "_") + '_' + channelID.replace(/[^a-zA-Z0-9_]/g, "_"),
                channelUID : $scope.thing.UID + ':' + channelID
            }, function() {
                $scope.getThing(true);
                toastService.showDefaultToast('Channel linked');
            });
        }
    };

    $scope.disableChannel = function(thingUID, channelID, event) {
        var channel = $scope.getChannelById(channelID);
        var linkedItem = channel.linkedItems[0];
        if ($scope.advancedMode) {
            $scope.unlinkChannel(channelID, event);
        } else {
            linkService.unlink({
                itemName : $scope.thing.UID.replace(/[^a-zA-Z0-9_]/g, "_") + '_' + channelID.replace(/[^a-zA-Z0-9_]/g, "_"),
                channelUID : $scope.thing.UID + ':' + channelID
            }, function() {
                $scope.getThing(true);
                toastService.showDefaultToast('Channel unlinked');
            });
        }
    };

    $scope.linkChannel = function(channelID, event) {
        var channel = $scope.getChannelById(channelID);
        var channelType = $scope.getChannelTypeById(channelID);
        $mdDialog.show({
            controller : 'LinkChannelDialogController',
            templateUrl : 'partials/dialog.linkchannel.html',
            targetEvent : event,
            hasBackdrop : true,
            linkedItem : channel.linkedItems.length > 0 ? channel.linkedItems[0] : '',
            acceptedItemType : channel.itemType + 'Item',
            category : channelType.category ? channelType.category : ""
        }).then(function(itemName) {
            linkService.link({
                itemName : itemName,
                channelUID : $scope.thing.UID + ':' + channelID
            }, function() {
                $scope.getThing(true);
                toastService.showDefaultToast('Channel linked');
            });
        });
    }

    $scope.unlinkChannel = function(channelID) {
        var channel = $scope.getChannelById(channelID);
        $mdDialog.show({
            controller : 'UnlinkChannelDialogController',
            templateUrl : 'partials/dialog.unlinkchannel.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                itemName : channel.linkedItems[0]
            }
        }).then(function(itemName) {
            linkService.unlink({
                itemName : channel.linkedItems[0],
                channelUID : $scope.thing.UID + ':' + channelID
            }, function() {
                $scope.getThing(true);
                toastService.showDefaultToast('Channel unlinked');
            });
        });
    }
    $scope.getChannelById = function(channelId) {
        if (!$scope.thing) {
            return;
        }
        return $.grep($scope.thing.channels, function(channel, i) {
            return channelId == channel.id;
        })[0];
    }

    $scope.getChannelTypeById = function(channelId) {
        return thingConfigService.getChannelTypeById($scope.thingType, $scope.channelTypes, channelId);
    };

    $scope.getChannelFromChannelTypes = function(channelUID) {
        if (!$scope.channelTypes) {
            return;
        }
        return thingConfigService.getChannelFromChannelTypes($scope.channelTypes, channelUID);
    };

    var getChannels = function(advanced) {

        if (!$scope.thingType || !$scope.thing || !$scope.channelTypes) {
            return;
        }
        $scope.isAdvanced = checkAdvance($scope.thing.channels);
        return thingConfigService.getThingChannels($scope.thing, $scope.thingType, $scope.channelTypes, advanced);
    };
    $scope.refreshChannels = function(showAdvanced) {
        $scope.thingChannels = getChannels(showAdvanced);
    }

    function checkAdvance(channels) {
        if (channels) {
            for (var i = 0, len = channels.length; i < len; i++) {
                var channel = channels[i];
                var channelType = $scope.getChannelTypeById(channel.id);
                if (channelType && channelType.advanced) {
                    return true;
                }
            }
        }
    }

    $scope.getThing = function(refresh) {
        thingRepository.getOne(function(thing) {
            return thing.UID === thingUID;
        }, function(thing) {
            $scope.thing = thing;
            $scope.thingTypeUID = thing.thingTypeUID;
            getThingType();
            if (thing.item) {
                $scope.setTitle(thing.label);
            } else {
                $scope.setTitle(thing.UID);
            }
        }, refresh);
    }
    $scope.getThing(true);

    function getThingType() {
        thingTypeRepository.getOne(function(thingType) {
            return thingType.UID === $scope.thingTypeUID;
        }, function(thingType) {
            $scope.thingType = thingType;
            if (thingType) {
                $scope.thingTypeChannels = thingType.channels && thingType.channels.length > 0 ? thingType.channels : thingType.channelGroups;
                $scope.setHeaderText(thingType.description);
            }
            $scope.refreshChannels($scope.showAdvanced);
        });
    }

    $scope.configChannel = function(channel, thing, event) {
        var channelType = this.getChannelFromChannelTypes(channel.channelTypeUID);

        $mdDialog.show({
            controller : 'ChannelConfigController',
            templateUrl : 'partials/dialog.channelconfig.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                channelType : channelType,
                channel : channel,
                thing : thing
            }
        });
    };

    $scope.hasProperties = function(properties) {
        return util.hasProperties(properties);
    }
}).controller('RemoveThingDialogController', function($scope, $mdDialog, toastService, thingService, thing) {
    $scope.thing = thing;
    $scope.isRemoving = thing.statusInfo.status === 'REMOVING';
    $scope.close = function() {
        $mdDialog.cancel();
    }
    $scope.remove = function(thingUID) {
        var forceRemove = $scope.isRemoving ? true : false;
        thingService.remove({
            thingUID : thing.UID,
            force : forceRemove
        }, function() {
            if (forceRemove) {
                toastService.showDefaultToast('Thing removed (forced).');
            } else {
                toastService.showDefaultToast('Thing removal initiated.');
            }
            $mdDialog.hide();
        });
    }
}).controller('LinkChannelDialogController', function($scope, $mdDialog, $filter, toastService, itemRepository, itemService, linkedItem, acceptedItemType, category) {
    $scope.itemName = linkedItem;
    $scope.acceptedItemType = acceptedItemType;
    $scope.category = category;
    $scope.itemFormVisible = false;
    itemRepository.getAll(function(items) {
        $scope.items = items;
        $scope.items = $filter('filter')($scope.items, {
            type : $scope.acceptedItemType
        });
        $scope.items = $filter('orderBy')($scope.items, "name");
        $scope.items.push({
            name : "_createNew",
            type : $scope.acceptedItemType
        });
    });
    $scope.checkCreateOption = function() {
        if ($scope.itemName == "_createNew") {
            $scope.itemFormVisible = true;
        } else {
            $scope.itemFormVisible = false;
        }
    }
    $scope.createAndLink = function() {
        var item = {
            name : $scope.newItemName,
            label : $scope.itemLabel,
            type : $scope.acceptedItemType,
            category : $scope.category
        };
        itemService.create({
            itemName : $scope.newItemName
        }, item).$promise.then(function() {
            toastService.showDefaultToast("Item created");
            itemRepository.setDirty(true);
            $mdDialog.hide($scope.newItemName);
        });
    }
    $scope.close = function() {
        $mdDialog.cancel();
    }
    $scope.link = function(itemName) {
        $mdDialog.hide(itemName);
    }
}).controller('UnlinkChannelDialogController', function($scope, $mdDialog, toastService, linkService, itemName) {
    $scope.itemName = itemName;
    $scope.close = function() {
        $mdDialog.cancel();
    }
    $scope.unlink = function() {
        $mdDialog.hide();
    }
}).controller('EditThingController', function($scope, $mdDialog, toastService, thingTypeRepository, thingRepository, configService, thingService) {
    $scope.setHeaderText('Click the \'Save\' button to apply the changes.');

    var thingUID = $scope.path[4];
    $scope.thingTypeUID = null;

    $scope.thing;
    $scope.groups = [];
    $scope.thingType;
    $scope.isEditing = true;
    var originalThing = {};

    $scope.update = function(thing) {
        if (!thing.item) {
            thing.item = {};
        }
        if (JSON.stringify(originalThing.configuration) !== JSON.stringify(thing.configuration)) {
            thing.configuration = configService.replaceEmptyValues(thing.configuration);
            thingService.updateConfig({
                thingUID : thing.UID
            }, thing.configuration, function() {
                thingRepository.update(thing);
            });
        }
        var dict = {};
        var update = false;
        if (originalThing.label !== thing.label) {
            dict.label = thing.label;
            update = true;
        }
        if (originalThing.bridgeUID !== thing.bridgeUID) {
            dict.bridgeUID = thing.bridgeUID
            update = true;
        }
        if (update) {
            thingService.update({
                thingUID : thing.UID
            }, dict);
        }
        toastService.showDefaultToast('Thing updated');
        $scope.navigateTo('things/view/' + thing.UID);
    };

    $scope.needsBridge = false;
    $scope.bridges = [];
    $scope.getBridges = function() {
        $scope.bridges = [];
        thingRepository.getAll(function(things) {
            for (var i = 0; i < things.length; i++) {
                var thing = things[i];
                for (var j = 0; j < $scope.thingType.supportedBridgeTypeUIDs.length; j++) {
                    var supportedBridgeTypeUID = $scope.thingType.supportedBridgeTypeUIDs[j];
                    if (thing.thingTypeUID === supportedBridgeTypeUID) {
                        $scope.bridges.push(thing);
                    }
                }
            }
        });
    };
    $scope.getThingType = function() {
        thingTypeRepository.getOne(function(thingType) {
            return thingType.UID === $scope.thingTypeUID;
        }, function(thingType) {
            $scope.thingType = thingType;
            $scope.parameters = configService.getRenderingModel(thingType.configParameters, thingType.parameterGroups);
            $scope.needsBridge = $scope.thingType.supportedBridgeTypeUIDs && $scope.thingType.supportedBridgeTypeUIDs.length > 0;
            if ($scope.needsBridge) {
                $scope.getBridges();
            }
        });
    };
    $scope.getThing = function(refresh) {
        thingRepository.getOne(function(thing) {
            return thing.UID === thingUID;
        }, function(thing) {
            $scope.thing = thing;
            angular.copy(thing, originalThing);
            $scope.thingTypeUID = thing.thingTypeUID;
            $scope.getThingType();
            if (thing.item) {
                $scope.setTitle('Edit ' + thing.label);
            } else {
                $scope.setTitle('Edit ' + thing.UID);
            }
        }, refresh);
    }
    $scope.getThing(false);
}).controller('ChannelConfigController', function($scope, $mdDialog, toastService, thingRepository, thingService, configService, channelType, channel, thing) {
    $scope.parameters = configService.getRenderingModel(channelType.parameters, channelType.parameterGroups);
    $scope.configuration = channel.configuration;
    $scope.channel = channel;
    $scope.thing = thing;
    $scope.close = function() {
        $mdDialog.cancel();
    }
    $scope.save = function() {
        var updated = false;
        for (var i = 0; !updated && i < $scope.thing.channels.length; i++) {
            if ($scope.thing.channels[i].uid == $scope.channel.uid) {
                $scope.thing.channels[i].configuration = $scope.configuration;
                updated = true;
            }
        }
        thingService.update({
            thingUID : thing.UID
        }, $scope.thing, function() {
            thingRepository.update($scope.thing);
            $mdDialog.hide();
            toastService.showDefaultToast('Channel updated');
        });
    }
});

angular.module('PaperUI.controllers.configuration').controller('ItemSetupController', function($scope, $timeout, $mdDialog, $filter, itemService, toastService, sharedProperties) {
    $scope.setSubtitle([ 'Items' ]);
    $scope.setHeaderText('Shows all configured Items.');
    $scope.items = [];
    $scope.refresh = function() {
        itemService.getNonRecursiveAll(function(items) {
            $scope.items = $filter('filter')(items, {
                type : '!GroupItem'
            })
        });

    };
    $scope.remove = function(item, event) {
        event.stopImmediatePropagation();
        $mdDialog.show({
            controller : 'ItemRemoveController',
            templateUrl : 'partials/dialog.removeitem.html',
            targetEvent : event,
            hasBackdrop : true,
            locals : {
                item : item
            }
        }).then(function() {
            $scope.refresh();
        });
    }
    $scope.getSrcURL = function(category, type) {
        return category ? '../icon/' + category.toLowerCase() : type ? '../icon/' + type.toLowerCase().replace('item', '') : '';
    }
    $scope.refresh();
}).controller('ItemConfigController', function($scope, $mdDialog, $filter, $location, toastService, itemService, itemConfig, itemRepository) {
    $scope.items = [];
    $scope.oldCategory;
    $scope.types = itemConfig.types;
    var itemName;
    var originalItem = {};
    if ($scope.path && $scope.path.length > 4) {
        itemName = $scope.path[4];
    }
    itemService.getNonRecursiveAll(function(items) {
        $scope.items = items;
        if (itemName) {
            var items = $filter('filter')(items, {
                name : itemName
            });
            if (items.length > 0) {
                $scope.item = items[0];
                angular.copy($scope.item, originalItem);
                $scope.configMode = "edit";
                $scope.srcURL = $scope.getSrcURL($scope.item.category, $scope.item.type);
                $scope.oldCategory = $scope.item.category;
                $scope.setTitle('Edit ' + $scope.item.name);
                $scope.setSubtitle([]);
            }
        } else {
            $scope.item = {};
            $scope.setTitle('Configuration');
            $scope.setSubtitle([ 'New Item' ]);
            $scope.configMode = "create";
        }

    });

    $scope.update = function() {
        putItem("Item updated.");
    }
    $scope.create = function(item) {
        putItem("Item created.");
    }

    function putItem(text) {
        if (JSON.stringify($scope.item) !== JSON.stringify(originalItem)) {
            itemService.create({
                itemName : $scope.item.name
            }, $scope.item).$promise.then(function() {
                toastService.showDefaultToast(text);
            });
        }
        $location.path('configuration/items');
    }

    $scope.renderIcon = function() {
        $scope.oldCategory = $scope.item.category;
        $scope.srcURL = $scope.getSrcURL($scope.item.category, $scope.item.type);
    }

}).controller('ItemRemoveController', function($scope, $mdDialog, $filter, $location, toastService, itemService, item) {
    $scope.item = item;
    $scope.remove = function(itemName) {
        itemService.remove({
            itemName : itemName
        }, function() {
            toastService.showDefaultToast('Item removed.');
        });
        $mdDialog.hide();
    }

    $scope.close = function() {
        $mdDialog.cancel();
    }
}).directive('itemname', function() {
    return {
        restrict : 'A',
        require : 'ngModel',
        link : function(scope, element, attr, ctrl) {
            function customValidator(ngModelValue) {

                var items = getItems();
                if (!searchItemNameExists(ngModelValue, items)) {
                    ctrl.$setValidity('nameValidator', true);
                    if (ngModelValue != null && ngModelValue.length != 0) {
                        element.parent().removeClass('md-input-invalid');
                    }
                } else {
                    ctrl.$setValidity('nameValidator', false);
                    element.parent().addClass('md-input-invalid');
                }
                return ngModelValue;
            }
            function searchItemNameExists(value, arr) {
                for (var i = 0; i < arr.length; i++) {
                    if (arr[i].name == value) {
                        return true;
                    }
                }
                return false;
            }
            function getItems() {
                return scope.items;
            }
            ctrl.$parsers.push(customValidator);
        }
    };
});
angular.module('PaperUI.controllers.extension', [ 'PaperUI.constants' ]).controller('ExtensionPageController', function($scope, extensionService, bindingRepository, thingTypeRepository, eventService, toastService, $filter) {
    $scope.navigateTo = function(path) {
        $location.path('extensions/' + path);
    };
    $scope.extensionTypes = [];
    $scope.refresh = function() {
        extensionService.getAllTypes(function(extensionTypes) {
            $scope.extensionTypes = [];
            angular.forEach(extensionTypes, function(extensionType) {
                $scope.extensionTypes.push({
                    typeId : extensionType.id,
                    label : extensionType.label,
                    extensions : [],
                    inProgress : false
                });
            });
            extensionService.getAll(function(extensions) {
                angular.forEach(extensions, function(extension) {
                    var extensionType = $scope.getType(extension.type);
                    if (extensionType !== undefined) {
                        extensionType.extensions.push(extension);
                    }
                });
                angular.forEach($scope.extensionTypes, function(extensionType) {
                    extensionType.extensions = $filter('orderBy')(extensionType.extensions, "label")
                });
            });
        });
    }

    $scope.getType = function(extensionTypeId) {
        var result;
        angular.forEach($scope.extensionTypes, function(extensionType) {
            if (extensionType.typeId === extensionTypeId) {
                result = extensionType;
            }
        });
        return result;
    };
    $scope.getExtension = function(extensionId) {
        var result;
        angular.forEach($scope.extensionTypes, function(extensionType) {
            angular.forEach(extensionType.extensions, function(extension) {
                if (extension.id === extensionId) {
                    result = extension;
                }
            });
        });
        return result;
    };
    $scope.refresh();
    $scope.install = function(extensionId) {
        var extension = $scope.getExtension(extensionId);
        extension.inProgress = true;
        extensionService.install({
            id : extensionId
        });
        bindingRepository.setDirty(true);
        thingTypeRepository.setDirty(true);
    };
    $scope.uninstall = function(extensionId) {
        var extension = $scope.getExtension(extensionId);
        extension.inProgress = true;
        extensionService.uninstall({
            id : extensionId
        });
        bindingRepository.setDirty(true);
        thingTypeRepository.setDirty(true);
    };
    eventService.onEvent('smarthome/extensions/*', function(topic, extensionId) {
        var extension = $scope.getExtension(extensionId);
        if (extension) {
            extension.inProgress = false;
            if (topic.indexOf("uninstalled") > -1) {
                extension.installed = false;
                toastService.showDefaultToast('Extension ' + extension.label + ' uninstalled.');
            } else if (topic.indexOf("installed") > -1) {
                extension.installed = true;
                toastService.showDefaultToast('Extension ' + extension.label + ' installed.');
            } else {
                toastService.showDefaultToast('Install or uninstall of extension ' + extension.label + ' failed.');
            }
        }
    });
});

angular.module('PaperUI.controllers.control', []).controller('ControlPageController', function($scope, $routeParams, $location, $timeout, itemRepository, thingService, thingTypeService, channelTypeService, thingConfigService) {
    $scope.items = [];
    $scope.selectedIndex = 0;
    $scope.tabs = [];
    $scope.things = [];

    $scope.next = function() {
        var newIndex = $scope.selectedIndex + 1;
        if (newIndex > ($scope.tabs.length - 1)) {
            newIndex = 0;
        }
        $scope.selectedIndex = newIndex;
    }
    $scope.prev = function() {
        var newIndex = $scope.selectedIndex - 1;
        if (newIndex < 0) {
            newIndex = $scope.tabs.length - 1;
        }
        $scope.selectedIndex = newIndex;
    }

    $scope.refresh = function() {
        itemRepository.getAll(function(items) {
            $scope.tabs = [];
            $scope.tabs.push({
                name : 'all',
                label : 'All'
            });
            $scope.items['All'] = items;
            for (var int = 0; int < items.length; int++) {
                var item = items[int];
                if (item.tags.indexOf('home-group') > -1) {
                    $scope.tabs.push({
                        name : item.name,
                        label : item.label
                    });
                }
            }
        }, true);
    }
    $scope.channelTypes = [];
    $scope.thingTypes = [];
    $scope.thingChannels = [];
    $scope.isLoadComplete = false;
    function getThings() {
        thingService.getAll().$promise.then(function(things) {
            $scope.things = things;
            $scope.isLoadComplete = false;
            thingTypeService.getAll().$promise.then(function(thingTypes) {
                $scope.thingTypes = thingTypes;
                channelTypeService.getAll().$promise.then(function(channels) {
                    $scope.channelTypes = channels;
                    for (var i = 0; i < $scope.things.length;) {
                        var thingType = $.grep($scope.thingTypes, function(thingType, j) {
                            return thingType.UID == $scope.things[i].thingTypeUID;
                        })[0];
                        $scope.things[i].thingChannels = thingConfigService.getThingChannels($scope.things[i], thingType, $scope.channelTypes, true);
                        angular.forEach($scope.things[i].thingChannels, function(value, key) {
                            $scope.things[i].thingChannels[key].channels = $.grep($scope.things[i].thingChannels[key].channels, function(channel, i) {
                                return channel.linkedItems.length > 0;
                            });
                        });
                        if (!thingHasChannels(i)) {
                            $scope.things.splice(i, 1);
                        } else {
                            i++;
                        }
                    }
                    $scope.isLoadComplete = true;
                });
            });

        });
    }
    function thingHasChannels(index) {
        for (var i = 0; i < $scope.things[index].thingChannels.length; i++) {
            if ($scope.things[index].thingChannels[i].channels && $scope.things[index].thingChannels[i].channels.length > 0) {
                return true;
            }
        }
        return false;
    }

    $scope.getItem = function(itemName) {
        for (var int = 0; int < $scope.data.items.length; int++) {
            var item = $scope.data.items[int];
            if (item.name === itemName) {
                return item;
            }
        }
        return null;
    }

    $scope.getThingsForTab = function(tabName) {
        $scope.things = thingService.getAll();
    }

    $scope.getItemsForTab = function(tabName) {
        var items = []
        if (tabName === 'all') {
            for (var int = 0; int < $scope.data.items.length; int++) {
                var item = $scope.data.items[int];
                if (item.tags.indexOf('thing') > -1) {
                    items.push(item);
                }
            }
            return items;
        } else {
            return this.getItem(tabName).members;
        }
    }

    $scope.masonry = function() {
        if ($scope.data.items) {
            $timeout(function() {
                var itemContainer = '#items-' + $scope.selectedIndex;
                new Masonry(itemContainer, {});
            }, 100, true);
        }
    }
    $scope.$on('ngRepeatFinished', function(ngRepeatFinishedEvent) {
        $scope.masonry();
    });
    $scope.refresh();
    getThings();

}).controller('ControlController', function($scope, $timeout, $filter, itemService) {

    $scope.getItemName = function(itemName) {
        return itemName.replace(/_/g, ' ');
    }

    $scope.getStateText = function(item) {
        if (item.state === 'NULL' || item.state === 'UNDEF') {
            return '-';
        }
        var state = item.type === 'NumberItem' ? parseFloat(item.state) : item.state;

        if (item.type === 'DateTimeItem') {
            var date = new Date(item.state);
            return $filter('date')(date, "dd.MM.yyyy hh:mm:ss");
        } else if (!item.stateDescription || !item.stateDescription.pattern) {
            return state;
        } else {
            return sprintf(item.stateDescription.pattern, state);
        }
    }

    $scope.getMinText = function(item) {
        if (!item.stateDescription || isNaN(item.stateDescription.minimum)) {
            return '';
        } else if (!item.stateDescription.pattern) {
            return item.stateDescription.minimum;
        } else {
            return sprintf(item.stateDescription.pattern, item.stateDescription.minimum);
        }
    }

    $scope.getMaxText = function(item) {
        if (!item.stateDescription || isNaN(item.stateDescription.maximum)) {
            return '';
        } else if (!item.stateDescription.pattern) {
            return item.stateDescription.maximum;
        } else {
            return sprintf(item.stateDescription.pattern, item.stateDescription.maximum);
        }
    }

    var categories = {
        'Alarm' : {},
        'Battery' : {},
        'Blinds' : {},
        'ColorLight' : {
            label : 'Color',
            icon : 'wb_incandescent'
        },
        'Contact' : {},
        'DimmableLight' : {
            label : 'Brightness',
            icon : 'wb_incandescent'
        },
        'CarbonDioxide' : {
            label : 'CO2'
        },
        'Door' : {},
        'Energy' : {},
        'Fan' : {},
        'Fire' : {},
        'Flow' : {},
        'GarageDoor' : {},
        'Gas' : {},
        'Humidity' : {},
        'Light' : {},
        'Motion' : {},
        'MoveControl' : {},
        'Player' : {},
        'PowerOutlet' : {},
        'Pressure' : {
        // icon: 'home-icon-measure_pressure_bar'
        },
        'Rain' : {},
        'Recorder' : {},
        'Smoke' : {},
        'SoundVolume' : {
            label : 'Volume',
            icon : 'volume_up',
            hideSwitch : true
        },
        'Switch' : {},
        'Temperature' : {
            label : 'Temperature'
        },
        'Water' : {},
        'Wind' : {},
        'Window' : {},
        'Zoom' : {},
    }

    $scope.getLabel = function(itemCategory, label, defaultLabel) {
        if (label) {
            return label;
        }

        if (itemCategory) {
            var category = categories[itemCategory];
            if (category) {
                return category.label ? category.label : itemCategory;
            } else {
                return defaultLabel;
            }
        } else {
            return defaultLabel;
        }
    }
    $scope.getIcon = function(itemCategory, fallbackIcon) {
        var defaultIcon = fallbackIcon ? fallbackIcon : 'radio_button_unchecked';
        if (itemCategory) {
            var category = categories[itemCategory];
            if (category) {
                return category.icon ? category.icon : defaultIcon;
            } else {
                return defaultIcon;
            }
        } else {
            return defaultIcon;
        }
    }
    $scope.isHideSwitch = function(itemCategory) {
        if (itemCategory) {
            var category = categories[itemCategory];
            if (category) {
                return category.hideSwitch;
            }
        }
        return false;
    }
    $scope.isReadOnly = function(item) {
        return item.stateDescription ? item.stateDescription.readOnly : false;
    }

    /**
     * Check if the item has a configured option list. Returns true if there are options, otherwise false.
     * 
     * @param item
     *            the current item
     */
    $scope.isOptionList = function(item) {
        return (item.stateDescription != null && item.stateDescription.options.length > 0)
    }
}).controller('ItemController', function($rootScope, $scope, itemService) {
    $scope.editMode = false;
    $scope.sendCommand = function(command, updateState) {
        $rootScope.itemUpdates[$scope.item.name] = new Date().getTime();
        itemService.sendCommand({
            itemName : $scope.item.name
        }, command);
        if (updateState) {
            $scope.item.state = command;
        }
    };
    $scope.editState = function() {
        $scope.editMode = true;
    };
    $scope.updateState = function() {
        $scope.sendCommand($scope.item.state, false);
        $scope.editMode = false;
    };
}).controller('DefaultItemController', function($scope, itemService) {

    $scope.optionListChanged = function() {
        $scope.sendCommand($scope.item.state, false);
    };

}).controller('ImageItemController', function($scope, itemService) {

    $scope.refreshCameraImage = function() {
        itemService.sendCommand({
            itemName : $scope.item.name
        }, "REFRESH");
    };

}).controller('SwitchItemController', function($scope, $timeout, itemService) {
    $scope.setOn = function(state) {
        $scope.sendCommand(state);
    }
}).controller('DimmerItemController', function($scope, $timeout, itemService) {
    if ($scope.item.state === 'UNDEF' || $scope.item.state === 'NULL') {
        $scope.item.state = '-';
    }
    $scope.on = parseInt($scope.item.state) > 0 ? 'ON' : 'OFF';

    $scope.setOn = function(on) {
        $scope.on = on === 'ON' ? 'ON' : 'OFF';

        $scope.sendCommand(on);

        var brightness = parseInt($scope.item.state);
        if (on === 'ON' && brightness === 0) {
            $scope.item.state = 100;
        }
        if (on === 'OFF' && brightness > 0) {
            $scope.item.state = 0;
        }
    }
    $scope.pending = false;
    $scope.setBrightness = function(brightness) {
        // send updates every 300 ms only
        if (!$scope.pending) {
            $timeout(function() {
                var command = $scope.item.state === 0 ? '0' : $scope.item.state;
                $scope.sendCommand(command);
                $scope.pending = false;
            }, 300);
            $scope.pending = true;
        }
    }
    $scope.$watch('item.state', function() {
        var brightness = parseInt($scope.item.state);
        if (brightness > 0 && $scope.on === 'OFF') {
            $scope.on = 'ON';
        }
        if (brightness === 0 && $scope.on === 'ON') {
            $scope.on = 'OFF';
        }
    });
}).controller('ColorItemController', function($scope, $timeout, $element, itemService) {

    function getStateAsObject(state) {
        var stateParts = state.split(",");
        if (stateParts.length == 3) {
            return {
                h : parseInt(stateParts[0]),
                s : parseInt(stateParts[1]),
                b : parseInt(stateParts[2])
            }
        } else {
            return {
                h : 0,
                s : 0,
                b : 0
            }
        }
    }

    function toState(stateObject) {
        return Math.ceil(stateObject.h) + ',' + Math.ceil(stateObject.s) + ',' + Math.ceil(stateObject.b);
    }

    $scope.pending = false;

    $scope.setBrightness = function(brightness) {
        // send updates every 300 ms only
        if (!$scope.pending) {
            $timeout(function() {
                var stateObject = getStateAsObject($scope.item.state);
                stateObject.b = $scope.brightness === 0 ? '0' : $scope.brightness;
                stateObject.s = $scope.saturation === 0 ? '0' : $scope.saturation;
                stateObject.h = $scope.hue === 0 ? '0' : $scope.hue;
                $scope.item.state = toState(stateObject);
                $scope.sendCommand($scope.item.state);
                $scope.pending = false;
            }, 300);
            $scope.pending = true;
        }
    }

    $scope.setHue = function(hue) {
        // send updates every 300 ms only
        if (!$scope.pending) {
            $timeout(function() {
                var stateObject = getStateAsObject($scope.item.state);
                stateObject.h = $scope.hue === 0 ? '0' : $scope.hue;
                stateObject.b = $scope.brightness === 0 ? '0' : $scope.brightness;
                stateObject.s = $scope.saturation === 0 ? '0' : $scope.saturation;
                if ($scope.item.state == "UNDEF" || $scope.item.state === 'NULL') {
                    stateObject.b = 100;
                    stateObject.s = 100;
                    $scope.brightness = stateObject.b;
                    $scope.saturation = stateObject.s;
                }
                $scope.item.state = toState(stateObject);
                $scope.sendCommand($scope.item.state);
                $scope.pending = false;
            }, 300);
            $scope.pending = true;
        }

    }

    $scope.setSaturation = function(saturation) {
        // send updates every 300 ms only
        if (!$scope.pending) {
            $timeout(function() {
                var stateObject = getStateAsObject($scope.item.state);
                stateObject.s = $scope.saturation === 0 ? '0' : $scope.saturation;
                stateObject.b = $scope.brightness === 0 ? '0' : $scope.brightness;
                stateObject.h = $scope.hue === 0 ? '0' : $scope.hue;
                $scope.item.state = toState(stateObject);
                $scope.sendCommand($scope.item.state);
                $scope.pending = false;
            }, 300);
            $scope.pending = true;
        }
    }

    $scope.getHexColor = function(hue) {
        var hsv = tinycolor({
            h : hue,
            s : 1,
            v : 1
        }).toHsv();
        return tinycolor(hsv).toHexString();
    }

    var setStates = function() {
        var stateObject = getStateAsObject($scope.item.state);
        var hue = stateObject.h;
        var brightness = stateObject.b;
        var saturation = stateObject.s;

        $scope.hue = hue ? hue : 0;
        $scope.brightness = brightness ? brightness : 0;
        $scope.saturation = saturation ? saturation : 0;
    }

    setStates();

    $scope.$watch('item.state', function() {
        setStates();
    });

    $scope.$watch('hue', function() {
        var hexColor = $scope.getHexColor($scope.hue);
        $($element).find('.hue .md-thumb').css('background-color', hexColor);
    });

    var hexColor = $scope.getHexColor();
    $($element).find('.hue .md-thumb').css('background-color', hexColor);
}).controller('NumberItemController', function($scope) {
    $scope.shouldRenderSlider = function(item) {
        if (item.stateDescription) {
            var stateDescription = item.stateDescription;
            if (stateDescription.readOnly) {
                return false;
            } else {
                if (isNaN(stateDescription.minimum) || isNaN(stateDescription.maximum)) {
                    return false;
                } else {
                    return true;
                }
            }
        }
        return false;
    }
}).controller('RollershutterItemController', function($scope) {
    if ($scope.item.state === 'UNDEF' || $scope.item.state === 'NULL') {
        $scope.item.state = '-';
    }
}).controller('PlayerItemController', function($scope) {

}).controller('LocationItemController', function($scope, $sce) {
    $scope.init = function() {
        if ($scope.item.state !== 'UNDEF' && $scope.item.state !== 'NULL') {
            var latitude = parseFloat($scope.item.state.split(',')[0]);
            var longitude = parseFloat($scope.item.state.split(',')[1]);
            var bbox = (longitude - 0.01) + ',' + (latitude - 0.01) + ',' + (longitude + 0.01) + ',' + (latitude + 0.01);
            var marker = latitude + ',' + longitude;
            $scope.formattedState = latitude + '°N ' + longitude + '°E';
            $scope.url = $sce.trustAsResourceUrl('http://www.openstreetmap.org/export/embed.html?bbox=' + bbox + '&marker=' + marker);
        } else {
            $scope.formattedState = '- °N - °E';
        }
    };
    $scope.$watch('item.state', function() {
        $scope.init();
    });
    $scope.init();
});
