/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.ecore.base.basicAttributes;

import com.google.common.base.Objects;
import java.util.Collections;
import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class BasicAttributesTypeConformance {
  public static final String IMLICIT_DEFAULT_ENUM = "ENUM_VALUE_UNDEFINED";
  
  public static final List<String> cppKeywords = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("alignas", "alignof", "and", "and_eq", "asm", "auto", "bitand", "bitor", "bool", "break", "case", "catch", "char", "char16_t", "char32_t", "class", "compl", "const", "constexpr", "const_cast", "continue", "decltype", "default", "delete", "do", "double", "dynamic_cast", "else", "enum", "explicit", "export", "extern", "false", "float", "for", "friend", "goto", "if", "inline", "int", "long", "mutable", "namespace", "new", "noexcept", "not", "not_eq", "nullptr", "operator", "or", "or_eq", "private", "protected", "public", "register", "reinterpret_cast", "return", "short", "signed", "sizeof", "static", "static_assert", "static_cast", "struct", "switch", "template", "this", "thread_local", "throw", "true", "try", "typedef", "typeid", "typename", "union", "unsigned", "using", "virtual", "void", "volatile", "wchar_t", "while", "xor", "xor_eq"));
  
  public AbstractValue createPrimitiveDefaultValue(final AbstractAttributeType attrType) {
    SingleValue value = null;
    if ((attrType instanceof PrimitiveType)) {
      final PRIMITIVE_TYPE_NAME typeName = ((PrimitiveType)attrType).getTypeName();
      boolean _lessEqualsThan = (typeName.compareTo(PRIMITIVE_TYPE_NAME.INT64) <= 0);
      if (_lessEqualsThan) {
        final IntValue intVal = BasicAttributesFactory.eINSTANCE.createIntValue();
        intVal.setValue(0);
        value = intVal;
      } else {
        boolean _lessEqualsThan_1 = (typeName.compareTo(PRIMITIVE_TYPE_NAME.DOUBLE) <= 0);
        if (_lessEqualsThan_1) {
          final FloatingPointValue dblVal = BasicAttributesFactory.eINSTANCE.createFloatingPointValue();
          dblVal.setValue(0.0);
          value = dblVal;
        } else {
          boolean _equals = Objects.equal(typeName, PRIMITIVE_TYPE_NAME.STRING);
          if (_equals) {
            final StringValue strVal = BasicAttributesFactory.eINSTANCE.createStringValue();
            strVal.setValue("Text");
            value = strVal;
          } else {
            boolean _equals_1 = Objects.equal(typeName, PRIMITIVE_TYPE_NAME.BOOLEAN);
            if (_equals_1) {
              final BoolValue boolVal = BasicAttributesFactory.eINSTANCE.createBoolValue();
              boolVal.setValue(false);
              value = boolVal;
            }
          }
        }
      }
    } else {
      if ((attrType instanceof InlineEnumerationType)) {
        final EnumerationValue en = BasicAttributesFactory.eINSTANCE.createEnumerationValue();
        en.setValue(IterableExtensions.<EnumerationElement>head(((InlineEnumerationType)attrType).getEnums()));
        value = en;
      }
    }
    ArrayType _array = attrType.getArray();
    boolean _tripleNotEquals = (_array != null);
    if (_tripleNotEquals) {
      final ArrayValue arrayValue = BasicAttributesFactory.eINSTANCE.createArrayValue();
      if ((value != null)) {
        arrayValue.getValues().add(value);
      }
      return arrayValue;
    } else {
      return value;
    }
  }
  
  public boolean isCompatible(final AbstractValue av, final AbstractAttributeType attrType) {
    ArrayType _array = attrType.getArray();
    boolean _tripleNotEquals = (_array != null);
    if (_tripleNotEquals) {
      if ((av instanceof ArrayValue)) {
        return true;
      } else {
        EObject _eContainer = av.eContainer();
        if ((_eContainer instanceof ArrayValue)) {
          return this.checkCompatibility(attrType, av);
        }
      }
    } else {
      return this.checkCompatibility(attrType, av);
    }
    return false;
  }
  
  protected boolean checkCompatibility(final AbstractAttributeType attrType, final AbstractValue av) {
    boolean compatible = false;
    if ((attrType instanceof PrimitiveType)) {
      final PRIMITIVE_TYPE_NAME typeName = ((PrimitiveType)attrType).getTypeName();
      boolean _lessEqualsThan = (typeName.compareTo(PRIMITIVE_TYPE_NAME.UINT64) <= 0);
      if (_lessEqualsThan) {
        if ((av instanceof IntValue)) {
          int _value = ((IntValue)av).getValue();
          boolean _greaterEqualsThan = (_value >= 0);
          compatible = _greaterEqualsThan;
        }
      } else {
        boolean _lessEqualsThan_1 = (typeName.compareTo(PRIMITIVE_TYPE_NAME.INT64) <= 0);
        if (_lessEqualsThan_1) {
          compatible = (av instanceof IntValue);
        } else {
          boolean _lessEqualsThan_2 = (typeName.compareTo(PRIMITIVE_TYPE_NAME.DOUBLE) <= 0);
          if (_lessEqualsThan_2) {
            compatible = ((av instanceof FloatingPointValue) || (av instanceof IntValue));
          } else {
            boolean _equals = Objects.equal(typeName, PRIMITIVE_TYPE_NAME.STRING);
            if (_equals) {
              compatible = (av instanceof StringValue);
            } else {
              boolean _equals_1 = Objects.equal(typeName, PRIMITIVE_TYPE_NAME.BOOLEAN);
              if (_equals_1) {
                compatible = (av instanceof BoolValue);
              }
            }
          }
        }
      }
    } else {
      if ((attrType instanceof InlineEnumerationType)) {
        if ((av instanceof EnumerationValue)) {
          final Function1<EnumerationElement, Boolean> _function = (EnumerationElement en) -> {
            String _name = en.getName();
            String _name_1 = ((EnumerationValue)av).getValue().getName();
            return Boolean.valueOf(Objects.equal(_name, _name_1));
          };
          compatible = IterableExtensions.<EnumerationElement>exists(((InlineEnumerationType)attrType).getEnums(), _function);
        } else {
          compatible = false;
        }
      }
    }
    return compatible;
  }
  
  public String getValueTypeName(final AbstractValue av) {
    String _xblockexpression = null;
    {
      String arrayPrefix = "a value-type ";
      EObject _eContainer = av.eContainer();
      if ((_eContainer instanceof ArrayValue)) {
        arrayPrefix = "an array value-type ";
      }
      String _switchResult = null;
      boolean _matched = false;
      if (av instanceof IntValue) {
        int _value = ((IntValue)av).getValue();
        boolean _lessThan = (_value < 0);
        if (_lessThan) {
          _matched=true;
          _switchResult = (arrayPrefix + "\'negative number\'");
        }
      }
      if (!_matched) {
        if (av instanceof IntValue) {
          int _value = ((IntValue)av).getValue();
          boolean _greaterEqualsThan = (_value >= 0);
          if (_greaterEqualsThan) {
            _matched=true;
            _switchResult = (arrayPrefix + "\'number\'");
          }
        }
      }
      if (!_matched) {
        if (av instanceof FloatingPointValue) {
          _matched=true;
          _switchResult = (arrayPrefix + "\'floating point\'");
        }
      }
      if (!_matched) {
        if (av instanceof StringValue) {
          _matched=true;
          _switchResult = (arrayPrefix + "\'string\'");
        }
      }
      if (!_matched) {
        if (av instanceof BoolValue) {
          _matched=true;
          _switchResult = (arrayPrefix + "\'boolean\'");
        }
      }
      if (!_matched) {
        if (av instanceof EnumerationValue) {
          _matched=true;
          _switchResult = (arrayPrefix + "\'EnumerationValue\'");
        }
      }
      if (!_matched) {
        if (av instanceof ArrayValue) {
          _matched=true;
          SingleValue _head = IterableExtensions.<SingleValue>head(((ArrayValue)av).getValues());
          String _valueTypeName = null;
          if (_head!=null) {
            _valueTypeName=this.getValueTypeName(_head);
          }
          _switchResult = _valueTypeName;
        }
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  public String getCardinalityName(final AbstractAttributeType attrType) {
    String _switchResult = null;
    boolean _matched = false;
    if (attrType instanceof PrimitiveType) {
      ArrayType _array = ((PrimitiveType)attrType).getArray();
      boolean _tripleNotEquals = (_array != null);
      if (_tripleNotEquals) {
        _matched=true;
        PRIMITIVE_TYPE_NAME _typeName = ((PrimitiveType)attrType).getTypeName();
        String _plus = ("an array of type \'" + _typeName);
        _switchResult = (_plus + "\'");
      }
    }
    if (!_matched) {
      if (attrType instanceof PrimitiveType) {
        ArrayType _array = ((PrimitiveType)attrType).getArray();
        boolean _tripleEquals = (_array == null);
        if (_tripleEquals) {
          _matched=true;
          PRIMITIVE_TYPE_NAME _typeName = ((PrimitiveType)attrType).getTypeName();
          String _plus = ("an element of type \'" + _typeName);
          _switchResult = (_plus + "\'");
        }
      }
    }
    if (!_matched) {
      if (attrType instanceof InlineEnumerationType) {
        ArrayType _array = ((InlineEnumerationType)attrType).getArray();
        boolean _tripleNotEquals = (_array != null);
        if (_tripleNotEquals) {
          _matched=true;
          _switchResult = "an array of type \'InlineEnumeration\'";
        }
      }
    }
    if (!_matched) {
      if (attrType instanceof InlineEnumerationType) {
        ArrayType _array = ((InlineEnumerationType)attrType).getArray();
        boolean _tripleEquals = (_array == null);
        if (_tripleEquals) {
          _matched=true;
          _switchResult = "an element of type \'InlineEnumeration\'";
        }
      }
    }
    return _switchResult;
  }
}
