/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.service.communicationObject.validation;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import org.eclipse.emf.common.util.URI;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AbstractAttributeType;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AbstractValue;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.BasicAttributesTypeConformance;
import org.eclipse.smartmdsd.ecore.service.communicationObject.AbstractCommElement;
import org.eclipse.smartmdsd.ecore.service.communicationObject.CommObjectsRepository;
import org.eclipse.smartmdsd.ecore.service.communicationObject.CommunicationObjectPackage;
import org.eclipse.smartmdsd.ecore.service.communicationObject.CommunicationObjectTypeConformance;
import org.eclipse.xtext.validation.Check;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class CommunicationObjectValidator extends AbstractCommunicationObjectValidator {
  protected static final String ISSUE_PREFIX = "org.xtext.service.communicationObject.";
  
  public static final String INVALID_REPO_NAME = (CommunicationObjectValidator.ISSUE_PREFIX + "InvalidRepoName");
  
  public static final String CAPITAL_COMM_ELEM_NAME = (CommunicationObjectValidator.ISSUE_PREFIX + "CapitalCommElementName");
  
  public static final String CPP_KEYWORD_COMM_ELEM_NAME = (CommunicationObjectValidator.ISSUE_PREFIX + "CppKeywordCommElementName");
  
  public static final String CPP_KEYWORD_ATTR_NAME = (CommunicationObjectValidator.ISSUE_PREFIX + "CppKeywordAttributeName");
  
  public static final String RESERVED_ENUM_NAME = (CommunicationObjectValidator.ISSUE_PREFIX + "ReservedEnumName");
  
  @Inject
  private CommunicationObjectTypeConformance conf;
  
  @Override
  public Boolean isCompatible(final AbstractValue av, final AbstractAttributeType type) {
    return Boolean.valueOf(this.conf.isCompatible(av, type));
  }
  
  @Check
  public void checkRepositoryNameIsProjectName(final CommObjectsRepository repo) {
    final URI uri = repo.eResource().getURI();
    boolean _isPlatform = uri.isPlatform();
    if (_isPlatform) {
      final String projectName = uri.segment(1);
      String _name = repo.getName();
      boolean _notEquals = (!Objects.equal(_name, projectName));
      if (_notEquals) {
        this.warning("Repository name should match project name.", 
          CommunicationObjectPackage.Literals.COMM_OBJECTS_REPOSITORY__NAME, 
          CommunicationObjectValidator.INVALID_REPO_NAME, projectName);
      }
    }
  }
  
  @Check
  public void checkAbstractCommElemNameStartsWithCapital(final AbstractCommElement elem) {
    boolean _isUpperCase = Character.isUpperCase(elem.getName().charAt(0));
    boolean _not = (!_isUpperCase);
    if (_not) {
      this.warning("CommElement name should start with a capital!", 
        CommunicationObjectPackage.Literals.ABSTRACT_COMM_ELEMENT__NAME, 
        CommunicationObjectValidator.CAPITAL_COMM_ELEM_NAME);
    }
  }
  
  @Check
  public void checkCppReservedKeywords(final AbstractCommElement elem) {
    boolean _contains = BasicAttributesTypeConformance.cppKeywords.contains(elem.getName());
    if (_contains) {
      this.error("Element name must not be a C++ reserved keyword.", 
        CommunicationObjectPackage.Literals.ABSTRACT_COMM_ELEMENT__NAME, 
        CommunicationObjectValidator.CPP_KEYWORD_COMM_ELEM_NAME);
    }
  }
}
