/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.component.componentParameter.validation;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AbstractValue;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AttributeDefinition;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AttributeRefinement;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinition;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ComponentParameter;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ComponentParameterBase;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ComponentParameterPackage;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ParameterInstance;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ParameterSetInstance;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.CoordinationSlavePort;
import org.eclipse.smartmdsd.ecore.service.coordinationPattern.ParameterPattern;
import org.eclipse.smartmdsd.ecore.service.parameterDefinition.ParameterSetDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.CoordinationServiceDefinition;
import org.eclipse.smartmdsd.xtext.component.componentParameter.validation.AbstractComponentParameterValidator;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class ComponentParameterValidator extends AbstractComponentParameterValidator {
  @Inject
  private IQualifiedNameProvider fqn;
  
  protected static final String COMP_PARAM_ISSUE_PREFIX = "org.xtext.component.componentParameter.";
  
  public static final String MISSING_ATTRIBUTE_REFINEMENTS = (ComponentParameterValidator.COMP_PARAM_ISSUE_PREFIX + "MissingAttributeRefinements");
  
  public static final String SINGLE_PARAM_SET_INSTANCE = (ComponentParameterValidator.COMP_PARAM_ISSUE_PREFIX + "SingleParamSetInstance");
  
  public static final String PARAM_SET_INSTANCE_MATCH_BEHAVIOR_INTERFACE = (ComponentParameterValidator.COMP_PARAM_ISSUE_PREFIX + "ParameterSetInstanceMatchBehaviorInterface");
  
  public static final String MISSING_ATTRIBUTE_VALUE = (ComponentParameterValidator.COMP_PARAM_ISSUE_PREFIX + "MissingAttributeValue");
  
  @Check
  public void checkAllAttributesRefined(final ParameterInstance param) {
    final Function1<AttributeDefinition, Boolean> _function = (AttributeDefinition attr) -> {
      final Function1<AttributeRefinement, Boolean> _function_1 = (AttributeRefinement ref) -> {
        return Boolean.valueOf(ref.getAttribute().equals(attr));
      };
      return Boolean.valueOf(IterableExtensions.<AttributeRefinement>exists(param.getAttributes(), _function_1));
    };
    boolean _forall = IterableExtensions.<AttributeDefinition>forall(param.getParameterDef().getAttributes(), _function);
    boolean _not = (!_forall);
    if (_not) {
      this.warning("Missing attribute-refinements", 
        ComponentParameterPackage.Literals.PARAMETER_INSTANCE__PARAMETER_DEF, 
        ComponentParameterValidator.MISSING_ATTRIBUTE_REFINEMENTS);
    }
  }
  
  @Check
  public void checkSingleParamSetInstance(final ParameterSetInstance param) {
    final EObject parent = param.eContainer();
    if ((parent instanceof ComponentParameter)) {
      int _size = IterableExtensions.size(Iterables.<ParameterSetInstance>filter(((ComponentParameter)parent).getParameters(), ParameterSetInstance.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple ParameterSetInstances are defined, only one (at most) is allowed.", 
          ComponentParameterPackage.Literals.PARAMETER_SET_INSTANCE__PARAM_SET, 
          ComponentParameterValidator.SINGLE_PARAM_SET_INSTANCE, 
          param.getParamSet().getName());
      }
    }
  }
  
  @Check
  public void checkParameterSetInstanceMatchComponentBehaviorSlaveInterface(final ParameterSetInstance paramSetInstance) {
    final EObject parent = paramSetInstance.eContainer();
    if ((parent instanceof ComponentParameter)) {
      ComponentDefinition _component = ((ComponentParameter)parent).getComponent();
      boolean _tripleNotEquals = (_component != null);
      if (_tripleNotEquals) {
        final CoordinationSlavePort slavePort = IterableExtensions.<CoordinationSlavePort>head(Iterables.<CoordinationSlavePort>filter(((ComponentParameter)parent).getComponent().getElements(), CoordinationSlavePort.class));
        if ((slavePort != null)) {
          CoordinationServiceDefinition _service = slavePort.getService();
          ParameterPattern _parameterPattern = null;
          if (_service!=null) {
            _parameterPattern=_service.getParameterPattern();
          }
          ParameterSetDefinition _parameterSet = null;
          if (_parameterPattern!=null) {
            _parameterSet=_parameterPattern.getParameterSet();
          }
          final ParameterSetDefinition paramSet = _parameterSet;
          if ((paramSet != null)) {
            boolean _equals = paramSet.equals(paramSetInstance.getParamSet());
            boolean _not = (!_equals);
            if (_not) {
              String _name = paramSetInstance.getParamSet().getName();
              String _plus = ("ParameterSetInstance " + _name);
              String _plus_1 = (_plus + 
                " does not match ParameterSet ");
              String _name_1 = paramSet.getName();
              String _plus_2 = (_plus_1 + _name_1);
              String _plus_3 = (_plus_2 + " specified within BehaviorSlaveInterface of Component ");
              String _name_2 = ((ComponentParameter)parent).getComponent().getName();
              String _plus_4 = (_plus_3 + _name_2);
              this.warning(_plus_4, 
                ComponentParameterPackage.Literals.PARAMETER_SET_INSTANCE__PARAM_SET, 
                ComponentParameterValidator.PARAM_SET_INSTANCE_MATCH_BEHAVIOR_INTERFACE, 
                this.fqn.getFullyQualifiedName(paramSet).toString());
            }
          }
        }
      }
    }
  }
  
  @Check
  public void checkAttributeHasDefaultValue(final AttributeDefinition attr) {
    final EObject parent = attr.eContainer();
    if ((parent instanceof ComponentParameterBase)) {
      AbstractValue _defaultvalue = attr.getDefaultvalue();
      boolean _tripleEquals = (_defaultvalue == null);
      if (_tripleEquals) {
        this.warning("You should assign a default value.", 
          null, 
          ComponentParameterValidator.MISSING_ATTRIBUTE_VALUE);
      }
    }
  }
}
