/**
 * Copyright (c) 2019 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.datasheetPropertyChecks.validation;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Collection;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.SystemComponentArchitecture;
import org.eclipse.smartmdsd.ecore.system.datasheetPropertyChecks.DatasheetPropertyCheck;
import org.eclipse.smartmdsd.ecore.system.datasheetPropertyChecks.DatasheetPropertyChecksPackage;
import org.eclipse.smartmdsd.ecore.system.datasheetPropertyChecks.EvaluationFunction;
import org.eclipse.smartmdsd.ecore.system.datasheetPropertyChecks.EvaluationOperator;
import org.eclipse.smartmdsd.ecore.system.datasheetPropertyChecks.SystemDatasheetPropertyChecksModel;
import org.eclipse.smartmdsd.xtext.system.datasheetPropertyChecks.validation.AbstractSystemDatasheetPropertyChecksValidator;
import org.eclipse.smartmdsd.xtext.system.datasheetPropertyChecks.validation.DatasheetPropertyCheckEvaluator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class SystemDatasheetPropertyChecksValidator extends AbstractSystemDatasheetPropertyChecksValidator {
  @Inject
  private DatasheetPropertyCheckEvaluator evaluator;
  
  @Check
  public void evaluateDatasheetPropertyCheck(final DatasheetPropertyCheck check) {
    final EObject parent = check.eContainer();
    if ((parent instanceof SystemDatasheetPropertyChecksModel)) {
      final SystemComponentArchitecture system = ((SystemDatasheetPropertyChecksModel)parent).getSystem();
      if (((system != null) && (check.getPropertyValue() != null))) {
        EvaluationFunction _function = check.getFunction();
        boolean _equals = Objects.equal(_function, EvaluationFunction.FOR_ALL);
        if (_equals) {
          EList<ComponentInstance> _components = system.getComponents();
          for (final ComponentInstance instance : _components) {
            Collection<String> _incompatiblePropertyValues = this.evaluator.getIncompatiblePropertyValues(instance, check);
            for (final String value : _incompatiblePropertyValues) {
              {
                String _name = instance.getName();
                String _plus = ("Component " + _name);
                String _plus_1 = (_plus + " has incompatible ");
                String _propertyName = check.getPropertyName();
                String _plus_2 = (_plus_1 + _propertyName);
                String _plus_3 = (_plus_2 + " ");
                final String message = (_plus_3 + value);
                this.warning(message, DatasheetPropertyChecksPackage.Literals.DATASHEET_PROPERTY_CHECK__PROPERTY_NAME);
              }
            }
          }
        } else {
          EvaluationFunction _function_1 = check.getFunction();
          boolean _equals_1 = Objects.equal(_function_1, EvaluationFunction.EXISTS);
          if (_equals_1) {
            boolean _exists = this.evaluator.exists(system, check);
            boolean _not = (!_exists);
            if (_not) {
              final String message = "No components found that satisfy this constraint";
              this.warning(message, DatasheetPropertyChecksPackage.Literals.DATASHEET_PROPERTY_CHECK__PROPERTY_NAME);
            }
          } else {
            EvaluationFunction _function_2 = check.getFunction();
            boolean _equals_2 = Objects.equal(_function_2, EvaluationFunction.SUM);
            if (_equals_2) {
              final double left = this.evaluator.calculateSum(system, check.getPropertyName());
              try {
                final Double right = Double.valueOf(check.getPropertyValue());
                boolean _doubleCompare = this.evaluator.doubleCompare(left, check.getOperator(), (right).doubleValue());
                boolean _not_1 = (!_doubleCompare);
                if (_not_1) {
                  EvaluationOperator _operator = check.getOperator();
                  String _plus = ((("Sum constraint violation: Σ " + Double.valueOf(left)) + " ") + _operator);
                  String _plus_1 = (_plus + " ");
                  final String message_1 = (_plus_1 + right);
                  this.warning(message_1, DatasheetPropertyChecksPackage.Literals.DATASHEET_PROPERTY_CHECK__PROPERTY_NAME);
                }
              } catch (final Throwable _t) {
                if (_t instanceof NumberFormatException) {
                  final NumberFormatException ex = (NumberFormatException)_t;
                  ex.printStackTrace();
                } else {
                  throw Exceptions.sneakyThrow(_t);
                }
              }
            }
          }
        }
      }
    }
  }
}
