/**
 * Copyright (c) 2019 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.base.genericDatasheet.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.DatasheetProperty;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.DefaultDatasheetProperties;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.GenericDatasheetModel;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.GenericDatasheetPackage;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.MandatoryDatasheetElement;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.MandatoryDatasheetElementNames;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.TechnologyReadinessLevel;
import org.eclipse.smartmdsd.xtext.base.genericDatasheet.GenericDatasheetUtils;
import org.eclipse.smartmdsd.xtext.base.genericDatasheet.validation.AbstractGenericDatasheetValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class GenericDatasheetValidator extends AbstractGenericDatasheetValidator {
  public static final String INVALID_SPDX_ID = "invalidSpdxId";
  
  public static final String INVALID_SPDX_URI = "invalidSpdxURI";
  
  public static final String INVALID_TRL_VALUE = "invalidTRLValue";
  
  public static final String UNDEFINED_BASE_URI = "undefinedBaseURI";
  
  public static final String UNDEFINED_SHORT_DESCRIPTION = "undefinedShortDescription";
  
  @Check
  public void checkMandatoryElements(final GenericDatasheetModel model) {
    final Iterable<MandatoryDatasheetElement> mandatory_elements = Iterables.<MandatoryDatasheetElement>filter(model.getElements(), MandatoryDatasheetElement.class);
    final Function1<MandatoryDatasheetElement, Boolean> _function = (MandatoryDatasheetElement it) -> {
      return Boolean.valueOf(it.getName().equals(MandatoryDatasheetElementNames.BASE_URI));
    };
    boolean _exists = IterableExtensions.<MandatoryDatasheetElement>exists(mandatory_elements, _function);
    boolean _not = (!_exists);
    if (_not) {
      this.warning("Mandatory Element BaseURI is not defined", null, GenericDatasheetValidator.UNDEFINED_BASE_URI);
    }
    final Function1<MandatoryDatasheetElement, Boolean> _function_1 = (MandatoryDatasheetElement it) -> {
      return Boolean.valueOf(it.getName().equals(MandatoryDatasheetElementNames.SHORT_DESCRIPTION));
    };
    boolean _exists_1 = IterableExtensions.<MandatoryDatasheetElement>exists(mandatory_elements, _function_1);
    boolean _not_1 = (!_exists_1);
    if (_not_1) {
      this.warning("Mandatory Element ShortDescription is not defined", null, GenericDatasheetValidator.UNDEFINED_SHORT_DESCRIPTION);
    }
  }
  
  @Check
  public void checkSpdxId(final DatasheetProperty property) {
    boolean _equals = property.getName().equals(DefaultDatasheetProperties.SPDX_LICENSE.getLiteral());
    if (_equals) {
      String _value = property.getValue();
      boolean _tripleNotEquals = (_value != null);
      if (_tripleNotEquals) {
        final Function1<String, Boolean> _function = (String it) -> {
          return Boolean.valueOf(it.equals(property.getValue()));
        };
        boolean _exists = IterableExtensions.<String>exists(GenericDatasheetUtils.getSpdxLicenseNames(), _function);
        boolean _not = (!_exists);
        if (_not) {
          this.warning("Invalid SPDX ID, please use the completion helper by pressing CTRL+Space to get valid IDs", 
            GenericDatasheetPackage.Literals.DATASHEET_PROPERTY__VALUE, GenericDatasheetValidator.INVALID_SPDX_ID);
        }
        String _semanticID = property.getSemanticID();
        boolean _tripleNotEquals_1 = (_semanticID != null);
        if (_tripleNotEquals_1) {
          boolean _contains = property.getSemanticID().contains(property.getValue());
          boolean _not_1 = (!_contains);
          if (_not_1) {
            this.warning("The semantic URI seems not to match the value", 
              GenericDatasheetPackage.Literals.DATASHEET_PROPERTY__SEMANTIC_ID, 
              GenericDatasheetValidator.INVALID_SPDX_URI, property.getValue());
          }
        }
      }
    }
  }
  
  @Check
  public void checkTRLValue(final DatasheetProperty property) {
    boolean _equals = property.getName().equals(DefaultDatasheetProperties.TECHNOLOGY_READINESS_LEVEL.getLiteral());
    if (_equals) {
      final Function1<TechnologyReadinessLevel, Boolean> _function = (TechnologyReadinessLevel it) -> {
        String _literal = it.getLiteral();
        String _value = property.getValue();
        return Boolean.valueOf(Objects.equal(_literal, _value));
      };
      boolean _exists = IterableExtensions.<TechnologyReadinessLevel>exists(TechnologyReadinessLevel.VALUES, _function);
      boolean _not = (!_exists);
      if (_not) {
        this.error("Invalid TRL value", 
          GenericDatasheetPackage.Literals.DATASHEET_PROPERTY__VALUE, GenericDatasheetValidator.INVALID_TRL_VALUE);
      }
    }
  }
}
