/**
 * Copyright (c) 2018 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Vineet Nagrath, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtend.open62541.compiler;

import com.google.inject.Inject;
import org.eclipse.smartmdsd.xtend.open62541.compiler.AbstractModelViewController;
import org.eclipse.smartmdsd.xtend.open62541.compiler.CopyrightHelpers;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;

@SuppressWarnings("all")
public abstract class AbstractModelViewControllerImpl implements AbstractModelViewController {
  @Inject
  @Extension
  private CopyrightHelpers _copyrightHelpers;
  
  @Override
  public String getAbstractModelHeader() {
    return "AbstractModel.hh";
  }
  
  @Override
  public String getAbstractModelSource() {
    return "AbstractModel.cc";
  }
  
  @Override
  public String getAbstractModelObserverHeader() {
    return "AbstractModelObserver.hh";
  }
  
  @Override
  public String getAbstractModelObserverSource() {
    return "AbstractModelObserver.cc";
  }
  
  @Override
  public String getAbstractViewHeader() {
    return "AbstractView.hh";
  }
  
  @Override
  public String getAbstractControllerHeader() {
    return "AbstractController.hh";
  }
  
  @Override
  public CharSequence compileAbstractModelHeader() {
    StringConcatenation _builder = new StringConcatenation();
    String _copyright = this._copyrightHelpers.getCopyright();
    _builder.append(_copyright);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#ifndef ABSTRACTMODEL_H_");
    _builder.newLine();
    _builder.append("#define ABSTRACTMODEL_H_");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"");
    String _abstractModelObserverHeader = this.getAbstractModelObserverHeader();
    _builder.append(_abstractModelObserverHeader);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#include <mutex>");
    _builder.newLine();
    _builder.append("#include <set>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("namespace Smart {");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// forward declaration");
    _builder.newLine();
    _builder.append("class AbstractModelObserver;");
    _builder.newLine();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This class implements the abstract version of the Model part of the Model-View-Controller (MVC)");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* design pattern. Please derive from this class to implement your specific version of the ");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* model (which can be for example a wrapper around a device driver, or a data-base or any");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* other data source/sink).");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("class AbstractModel {");
    _builder.newLine();
    _builder.append("private:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("// observers registry");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("std::mutex registry_mutex;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("std::set<AbstractModelObserver*> registry;");
    _builder.newLine();
    _builder.newLine();
    _builder.append("protected:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("// this flag indicates that some values have changed");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("bool changed_values;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("// call this method (from within the internal thread) each time some values have changed");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("void notify();");
    _builder.newLine();
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("AbstractModel();");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual ~AbstractModel();");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// this method is used by AbstractModelObserver to get notified about model changes");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("void attach(AbstractModelObserver *observer);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("void detach(AbstractModelObserver *observer);");
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("} /* namespace Smart */");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#endif /* ABSTRACTMODEL_H_ */");
    _builder.newLine();
    return _builder;
  }
  
  @Override
  public CharSequence compileAbstractModelSource() {
    StringConcatenation _builder = new StringConcatenation();
    String _copyright = this._copyrightHelpers.getCopyright();
    _builder.append(_copyright);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#include \"");
    String _abstractModelHeader = this.getAbstractModelHeader();
    _builder.append(_abstractModelHeader);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("namespace Smart {");
    _builder.newLine();
    _builder.newLine();
    _builder.append("AbstractModel::AbstractModel()");
    _builder.newLine();
    _builder.append(":\tchanged_values(false)");
    _builder.newLine();
    _builder.append("{  }");
    _builder.newLine();
    _builder.newLine();
    _builder.append("AbstractModel::~AbstractModel()");
    _builder.newLine();
    _builder.append("{  }");
    _builder.newLine();
    _builder.newLine();
    _builder.append("void AbstractModel::attach(AbstractModelObserver *observer) {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("std::unique_lock<std::mutex> lock(registry_mutex);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("registry.insert(observer);");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("void AbstractModel::detach(AbstractModelObserver *observer) {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("std::unique_lock<std::mutex> lock(registry_mutex);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("registry.erase(observer);");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("void AbstractModel::notify() {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("std::unique_lock<std::mutex> lock(registry_mutex);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("auto it = registry.begin();");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("for(; it!=registry.end(); it++) {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("(*it)->update();");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("} /* namespace Smart */");
    _builder.newLine();
    return _builder;
  }
  
  @Override
  public CharSequence compileAbstractModelObserverHeader() {
    StringConcatenation _builder = new StringConcatenation();
    String _copyright = this._copyrightHelpers.getCopyright();
    _builder.append(_copyright);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#ifndef ABSTRACTMODELOBSERVER_H_");
    _builder.newLine();
    _builder.append("#define ABSTRACTMODELOBSERVER_H_");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"");
    String _abstractModelHeader = this.getAbstractModelHeader();
    _builder.append(_abstractModelHeader);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("namespace Smart {");
    _builder.newLine();
    _builder.newLine();
    _builder.append("// forward declaration");
    _builder.newLine();
    _builder.append("class AbstractModel;");
    _builder.newLine();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This class defines the abstract base-class for the ModelObserver part");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* of the Model-View-Controller (MVC) design pattern. From this class,");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* the View and the Controller classes are derived. The abstract method ");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* update() must implement the respective update strategy of the derived class.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("class AbstractModelObserver {");
    _builder.newLine();
    _builder.append("protected:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("AbstractModel *model;");
    _builder.newLine();
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/**");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* Default constructor attaches this class");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* to the given model (in case the model is not null).");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("AbstractModelObserver(AbstractModel *model=0);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/**");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* Default destructor detaches this class from the");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* internal model (in case the model is not null).");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual ~AbstractModelObserver();");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// implement this method in derived classes");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual void update() = 0;");
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("} /* namespace Smart */");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#endif /* ABSTRACTMODELOBSERVER_H_ */");
    _builder.newLine();
    return _builder;
  }
  
  @Override
  public CharSequence compileAbstractModelObserverSource() {
    StringConcatenation _builder = new StringConcatenation();
    String _copyright = this._copyrightHelpers.getCopyright();
    _builder.append(_copyright);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#include \"");
    String _abstractModelObserverHeader = this.getAbstractModelObserverHeader();
    _builder.append(_abstractModelObserverHeader);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("namespace Smart {");
    _builder.newLine();
    _builder.newLine();
    _builder.append("AbstractModelObserver::AbstractModelObserver(AbstractModel *model)");
    _builder.newLine();
    _builder.append(":\tmodel(model)");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("if(model != 0) {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("model->attach(this);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("AbstractModelObserver::~AbstractModelObserver()");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("if(model != 0) {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("model->detach(this);");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("} /* namespace Smart */");
    _builder.newLine();
    return _builder;
  }
  
  @Override
  public CharSequence compileAbstractViewHeader() {
    StringConcatenation _builder = new StringConcatenation();
    String _copyright = this._copyrightHelpers.getCopyright();
    _builder.append(_copyright);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#ifndef ABSTRACTVIEW_HH_");
    _builder.newLine();
    _builder.append("#define ABSTRACTVIEW_HH_");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"");
    String _abstractModelObserverHeader = this.getAbstractModelObserverHeader();
    _builder.append(_abstractModelObserverHeader);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("namespace Smart {");
    _builder.newLine();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This class implements the View part of the Model-View-Controller (MVC)");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* design pattern. Please override the method update() in derived classes");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* to realize your view update strategy. In addition, it is recommended to");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* override the method getModel() such that it returns your specific model class");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* rather than the AbstractModel. This is particularly useful for the");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* Controller class of the MVC pattern.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("class AbstractView : public AbstractModelObserver");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// default constructor");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("AbstractView(AbstractModel *model)");
    _builder.newLine();
    _builder.append("\t");
    _builder.append(":\tAbstractModelObserver(model)");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("{  }");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// default destructor");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual ~AbstractView()");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("{  }");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// implements AbstractModelObserver");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual void update() override {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("// default is a no-op");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// returns a pinter to the abstract model (overload this method in derived classes to return the specific model type)");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual AbstractModel* getModel() const {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("return model;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("} /* namespace Smart */");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#endif /* ABSTRACTVIEW_HH_ */");
    _builder.newLine();
    return _builder;
  }
  
  @Override
  public CharSequence compileAbstractControllerHeader() {
    StringConcatenation _builder = new StringConcatenation();
    String _copyright = this._copyrightHelpers.getCopyright();
    _builder.append(_copyright);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#ifndef ABSTRACTCONTROLLER_HH_");
    _builder.newLine();
    _builder.append("#define ABSTRACTCONTROLLER_HH_");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#include \"");
    String _abstractViewHeader = this.getAbstractViewHeader();
    _builder.append(_abstractViewHeader);
    _builder.append("\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("namespace Smart {");
    _builder.newLine();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This class implements the Controller part of the Model-View-Controller (MVC)");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* design pattern. The actual model can be internally accessed over the given view pointer.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("class AbstractController  : public AbstractModelObserver");
    _builder.newLine();
    _builder.append("{");
    _builder.newLine();
    _builder.append("public:");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// default constructor attaches itself as an observer to the model accessed through the given view");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("AbstractController(AbstractView *view)");
    _builder.newLine();
    _builder.append("\t");
    _builder.append(":\tAbstractModelObserver()");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("{");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("if(view != 0) {");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("view->getModel()->attach(this);");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("this->model = view->getModel();");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// default destructor");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual ~AbstractController()");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("{  }");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/// call this method at the end of the main function");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("virtual int run() = 0;");
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    _builder.append("} /* namespace Smart */");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#endif /* ABSTRACTCONTROLLER_HH_ */");
    _builder.newLine();
    return _builder;
  }
}
