/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.component.componentDefinition.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.Activity;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.AnswerPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinition;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinitionModelUtility;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinitionPackage;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentSubNode;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentSubNodeObserver;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.InputHandler;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.InputPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.InputPortLink;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.RequestHandler;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.RequestPortLink;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ComponentParameterPackage;
import org.eclipse.smartmdsd.ecore.component.componentParameter.ComponentParametersRef;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.CommunicationServiceUsageRealization;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.CoordinationExtensionPackage;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.CoordinationSlavePort;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.OperationModeBinding;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.PrivateOperationMode;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.PublicOperationMode;
import org.eclipse.smartmdsd.ecore.component.performanceExtension.ActivationConstraints;
import org.eclipse.smartmdsd.ecore.component.performanceExtension.DefaultPeriodicTimer;
import org.eclipse.smartmdsd.ecore.component.performanceExtension.DefaultTrigger;
import org.eclipse.smartmdsd.ecore.component.performanceExtension.PerformanceExtensionPackage;
import org.eclipse.smartmdsd.ecore.component.seronetExtension.SeronetExtensionPackage;
import org.eclipse.smartmdsd.ecore.component.seronetExtension.SupportedMiddleware;
import org.eclipse.smartmdsd.xtext.component.componentDefinition.validation.AbstractComponentDefinitionValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class ComponentDefinitionValidator extends AbstractComponentDefinitionValidator {
  protected static final String COMP_DEF_ISSUE_PREFIX = "org.xtext.component.componentDefinition.";
  
  public static final String DUPLICATE_ACTIVATION_CONSTRAINTS = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "DuplicateActivationConstraints");
  
  public static final String ACTIVATION_FREQUENCY_CONSTRAINT = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "ActivationFrequencyConstraint");
  
  public static final String DEFAULT_TIMED_TRIGGER_MIN_FREQUENCY = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "DefaultTimedTriggerMinFrequency");
  
  public static final String DEFAULT_TIMED_TRIGGER_MAX_FREQUENCY = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "DefaultTimedTriggerMaxFrequency");
  
  public static final String DUPLICATE_OPERATION_MODE_BINDING = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "DuplicateOperationModeBinding");
  
  public static final String ACTIVITY_OBSERVATION_CYLE = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "ActivityObservationCycle");
  
  public static final String NOT_USED_PRIVATE_OPERATION_MODE = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "NotUsedPrivateOperationMode");
  
  public static final String PRIVATE_OPERATION_MODE_NAME = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "PrivateOperationModeName");
  
  public static final String MULTIPLE_DEFAULT_OPERATION_MODES = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "MultipleDefaultOperationModes");
  
  public static final String DUPLICATE_PARAMETER_STRUCT = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "DuplicateParameterStruct");
  
  public static final String MATCHING_PARAMETER_STRUCT = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "MatchingParameterStruct");
  
  public static final String MULTIPLE_DEFAULT_MIDDLEWARES = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "MultipleDefaultMiddlewares");
  
  public static final String MULTIPLE_DEFAULT_TRIGGERS = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "MultipleDefaultTriggers");
  
  public static final String INPUT_HANDLER_LINK = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "InputHandlerLink");
  
  public static final String INCOMPLETE_COORDINATION_SLAVE_PORT = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "IncompleteCoordinationSlavePort");
  
  public static final String DUPLICATE_REALIZATIONS_COORDINATION_SLAVE_PORT = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "DuplicateRealizationsCoordinationSlavePort");
  
  public static final String MISSING_REQUEST_HANDLER = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "MissingRequestHandler");
  
  public static final String PASSIVE_REQUEST_HANDLER = (ComponentDefinitionValidator.COMP_DEF_ISSUE_PREFIX + "PassiveRequestHandler");
  
  @Check
  public void checkDuplicateActivationConstraints(final ActivationConstraints act) {
    final EObject parent = act.eContainer();
    if ((parent instanceof Activity)) {
      int _size = IterableExtensions.size(Iterables.<ActivationConstraints>filter(((Activity)parent).getExtensions(), ActivationConstraints.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple ActivationConstraints-elements found, but (at most) one per Activity is allowed.", 
          null, ComponentDefinitionValidator.DUPLICATE_ACTIVATION_CONSTRAINTS);
      }
    }
  }
  
  @Check
  public void checkMinActFreqSmallerMaxActFreq(final ActivationConstraints ac) {
    if (((ac.getMaxActFreq() > 0.0) && (ac.getMinActFreq() > ac.getMaxActFreq()))) {
      this.warning("Activation minimal-frequency should be smaller or equal to the maximal-frequency", 
        PerformanceExtensionPackage.Literals.ACTIVATION_CONSTRAINTS__MIN_ACT_FREQ, 
        ComponentDefinitionValidator.ACTIVATION_FREQUENCY_CONSTRAINT);
    }
  }
  
  @Check
  public void checkDefaultTimedTriggerFrequencyWithinActConstraints(final DefaultPeriodicTimer timer) {
    final EObject parent = timer.eContainer();
    if ((parent instanceof Activity)) {
      Iterable<ActivationConstraints> _filter = Iterables.<ActivationConstraints>filter(((Activity)parent).getExtensions(), ActivationConstraints.class);
      for (final ActivationConstraints act : _filter) {
        {
          double _minActFreq = act.getMinActFreq();
          boolean _greaterThan = (_minActFreq > 0.0);
          if (_greaterThan) {
            double _periodicActFreq = timer.getPeriodicActFreq();
            double _minActFreq_1 = act.getMinActFreq();
            boolean _lessThan = (_periodicActFreq < _minActFreq_1);
            if (_lessThan) {
              String _string = Double.valueOf(act.getMinActFreq()).toString();
              String _plus = ("Chosen periodic activation-frequency is smaller than minimum frequency " + _string);
              String _plus_1 = (_plus + " defined in ActivationConstraints");
              this.warning(_plus_1, 
                PerformanceExtensionPackage.Literals.DEFAULT_PERIODIC_TIMER__PERIODIC_ACT_FREQ, 
                ComponentDefinitionValidator.DEFAULT_TIMED_TRIGGER_MIN_FREQUENCY, 
                Double.valueOf(act.getMinActFreq()).toString());
            }
          }
          double _maxActFreq = act.getMaxActFreq();
          boolean _greaterThan_1 = (_maxActFreq > 0.0);
          if (_greaterThan_1) {
            double _periodicActFreq_1 = timer.getPeriodicActFreq();
            double _maxActFreq_1 = act.getMaxActFreq();
            boolean _greaterThan_2 = (_periodicActFreq_1 > _maxActFreq_1);
            if (_greaterThan_2) {
              String _string_1 = Double.valueOf(act.getMaxActFreq()).toString();
              String _plus_2 = ("Chosen periodic activation-frequency is higher than maximum frequency " + _string_1);
              String _plus_3 = (_plus_2 + " defined in ActivationConstraints");
              this.warning(_plus_3, 
                PerformanceExtensionPackage.Literals.DEFAULT_PERIODIC_TIMER__PERIODIC_ACT_FREQ, 
                ComponentDefinitionValidator.DEFAULT_TIMED_TRIGGER_MAX_FREQUENCY, 
                Double.valueOf(act.getMaxActFreq()).toString());
            }
          }
        }
      }
    }
  }
  
  @Check
  public void checkDuplicateOperationModeBinding(final OperationModeBinding state) {
    final EObject parent = state.eContainer();
    if ((parent instanceof Activity)) {
      int _size = IterableExtensions.size(Iterables.<OperationModeBinding>filter(((Activity)parent).getExtensions(), OperationModeBinding.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple OperationModeBinding-elements found, but (at most) one per Activity is allowed.", 
          null, ComponentDefinitionValidator.DUPLICATE_OPERATION_MODE_BINDING);
      }
    }
  }
  
  @Check
  public void checkNoActivityObserverCycle(final ComponentSubNodeObserver observer) {
    final EObject parent = observer.eContainer();
    final ComponentSubNode subject = observer.getSubject();
    if ((subject instanceof Activity)) {
      boolean _contains = ComponentDefinitionModelUtility.getActivityObservationHierarchy(((Activity)subject)).contains(parent);
      if (_contains) {
        String _name = ((Activity)subject).getName();
        String _plus = ("Cycle in ActivityObservation of observed Activity " + _name);
        String _plus_1 = (_plus + ".");
        this.error(_plus_1, 
          ComponentDefinitionPackage.Literals.COMPONENT_SUB_NODE_OBSERVER__SUBJECT, 
          ComponentDefinitionValidator.ACTIVITY_OBSERVATION_CYLE);
      }
    }
  }
  
  @Check
  public void checkPrivateOperationModeReferencedFromPublicOperationMode(final PrivateOperationMode mode) {
    final EObject parent = mode.eContainer();
    if ((parent instanceof CoordinationSlavePort)) {
      final Function1<PublicOperationMode, Boolean> _function = (PublicOperationMode it) -> {
        final Function1<PrivateOperationMode, Boolean> _function_1 = (PrivateOperationMode it_1) -> {
          return Boolean.valueOf(Objects.equal(it_1, mode));
        };
        return Boolean.valueOf(IterableExtensions.<PrivateOperationMode>exists(it.getActivates(), _function_1));
      };
      boolean _exists = IterableExtensions.<PublicOperationMode>exists(Iterables.<PublicOperationMode>filter(((CoordinationSlavePort)parent).getElements(), PublicOperationMode.class), _function);
      boolean _not = (!_exists);
      if (_not) {
        String _name = mode.getName();
        String _plus = ("PrivateOperationMode " + _name);
        String _plus_1 = (_plus + " is not used by any PublicOperationMode.");
        this.warning(_plus_1, 
          CoordinationExtensionPackage.Literals.PRIVATE_OPERATION_MODE__NAME, 
          ComponentDefinitionValidator.NOT_USED_PRIVATE_OPERATION_MODE, 
          mode.getName());
      }
    }
  }
  
  @Check
  public void checkPrivateOperationModeNameBeginsWithSmallLetter(final PrivateOperationMode mode) {
    boolean _isLowerCase = Character.isLowerCase(mode.getName().charAt(0));
    boolean _not = (!_isLowerCase);
    if (_not) {
      this.warning("PrivateOperationMode name should begin with a small letter.", 
        CoordinationExtensionPackage.Literals.PRIVATE_OPERATION_MODE__NAME, 
        ComponentDefinitionValidator.PRIVATE_OPERATION_MODE_NAME, 
        mode.getName());
    }
  }
  
  @Check
  public void checkSingleDefaultOperationMode(final PublicOperationMode mode) {
    final EObject parent = mode.eContainer();
    if ((parent instanceof CoordinationSlavePort)) {
      final Function1<PublicOperationMode, Boolean> _function = (PublicOperationMode it) -> {
        boolean _isIsDefaultInit = it.isIsDefaultInit();
        return Boolean.valueOf((_isIsDefaultInit == true));
      };
      int _size = IterableExtensions.size(IterableExtensions.<PublicOperationMode>filter(Iterables.<PublicOperationMode>filter(((CoordinationSlavePort)parent).getElements(), PublicOperationMode.class), _function));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.warning("Multiple default PublicOperationModes are detected, but at most one is allowed.", 
          CoordinationExtensionPackage.Literals.PUBLIC_OPERATION_MODE__IS_DEFAULT_INIT, 
          ComponentDefinitionValidator.MULTIPLE_DEFAULT_OPERATION_MODES);
      }
    }
  }
  
  @Check
  public void checkSingleParamStruct(final ComponentParametersRef param) {
    final EObject parent = param.eContainer();
    if ((parent instanceof ComponentDefinition)) {
      int _size = IterableExtensions.size(Iterables.<ComponentParametersRef>filter(((ComponentDefinition)parent).getElements(), ComponentParametersRef.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple ParameterStructs defined, but at most one is allowed.", 
          ComponentParameterPackage.Literals.COMPONENT_PARAMETERS_REF__PARAMETER, 
          ComponentDefinitionValidator.DUPLICATE_PARAMETER_STRUCT);
      }
    }
  }
  
  @Check
  public void checkMatchingParamStruct(final ComponentParametersRef param) {
    final EObject component = param.eContainer();
    if ((component instanceof ComponentDefinition)) {
      boolean _equals = param.getParameter().getComponent().equals(component);
      boolean _not = (!_equals);
      if (_not) {
        this.warning("ParameterStruct is not using this component.", 
          ComponentParameterPackage.Literals.COMPONENT_PARAMETERS_REF__PARAMETER, 
          ComponentDefinitionValidator.MATCHING_PARAMETER_STRUCT);
      }
    }
  }
  
  @Check
  public void checkSingleDefaultMiddleware(final SupportedMiddleware mw) {
    boolean _isDefault = mw.isDefault();
    if (_isDefault) {
      final EObject parent = mw.eContainer();
      if ((parent instanceof ComponentPort)) {
        Iterable<SupportedMiddleware> _filter = Iterables.<SupportedMiddleware>filter(((ComponentPort)parent).getExtensions(), SupportedMiddleware.class);
        for (final SupportedMiddleware other : _filter) {
          if (((!Objects.equal(other, mw)) && other.isDefault())) {
            this.error("Multiple default middlewares are specified", 
              SeronetExtensionPackage.Literals.SUPPORTED_MIDDLEWARE__DEFAULT, 
              ComponentDefinitionValidator.MULTIPLE_DEFAULT_MIDDLEWARES, 
              mw.eClass().getName());
          }
        }
      }
    }
  }
  
  @Check
  public void checkSingleDefaultTrigger(final DefaultTrigger trigger) {
    final EObject parent = trigger.eContainer();
    if ((parent instanceof Activity)) {
      int _size = IterableExtensions.size(Iterables.<DefaultTrigger>filter(((Activity)parent).getExtensions(), DefaultTrigger.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple default triggers are specified", 
          null, 
          ComponentDefinitionValidator.MULTIPLE_DEFAULT_TRIGGERS);
      }
    }
  }
  
  @Check
  public void checkInputHandlerLink(final InputPortLink link) {
    final EObject parent = link.eContainer();
    if ((parent instanceof InputHandler)) {
      InputPort _inputPort = link.getInputPort();
      InputPort _inputPort_1 = ((InputHandler)parent).getInputPort();
      boolean _equals = Objects.equal(_inputPort, _inputPort_1);
      if (_equals) {
        this.error("InputLink cannot be the same as the InputHandler\'s input-trigger", 
          ComponentDefinitionPackage.Literals.INPUT_PORT_LINK__INPUT_PORT, 
          ComponentDefinitionValidator.INPUT_HANDLER_LINK);
      }
    }
  }
  
  @Check
  public void checkForMissingOptionalCoordinationSlaveRealization(final CoordinationSlavePort slavePort) {
    int _size = slavePort.getService().getServices().size();
    int _size_1 = IterableExtensions.size(Iterables.<CommunicationServiceUsageRealization>filter(slavePort.getElements(), CommunicationServiceUsageRealization.class));
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      this.warning("CoordinationSlavePort needs to realize all CommunicationServiceUsages.", 
        CoordinationExtensionPackage.Literals.COORDINATION_SLAVE_PORT__ELEMENTS, 
        ComponentDefinitionValidator.INCOMPLETE_COORDINATION_SLAVE_PORT);
    }
  }
  
  @Check
  public void checkDuplicateOptionalCoordinationSlaveRealization(final CommunicationServiceUsageRealization cServiceRealization) {
    final EObject parent = cServiceRealization.eContainer();
    if ((parent instanceof CoordinationSlavePort)) {
      final Function1<CommunicationServiceUsageRealization, Boolean> _function = (CommunicationServiceUsageRealization it) -> {
        return Boolean.valueOf(it.getServiceUsage().equals(cServiceRealization.getServiceUsage()));
      };
      int _size = IterableExtensions.size(IterableExtensions.<CommunicationServiceUsageRealization>filter(Iterables.<CommunicationServiceUsageRealization>filter(((CoordinationSlavePort)parent).getElements(), CommunicationServiceUsageRealization.class), _function));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Duplicate realization of a CommunicationServiceUsage found, but exactly one realization is required.", 
          CoordinationExtensionPackage.Literals.COMMUNICATION_SERVICE_USAGE_REALIZATION__SERVICE_USAGE, 
          ComponentDefinitionValidator.DUPLICATE_REALIZATIONS_COORDINATION_SLAVE_PORT);
      }
    }
  }
  
  @Check
  public void checkExistingRequestHandler(final AnswerPort port) {
    final EObject parent = port.eContainer();
    if ((parent instanceof ComponentDefinition)) {
      final Function1<RequestHandler, Boolean> _function = (RequestHandler it) -> {
        AnswerPort _answerPort = it.getAnswerPort();
        return Boolean.valueOf(Objects.equal(_answerPort, port));
      };
      boolean _exists = IterableExtensions.<RequestHandler>exists(Iterables.<RequestHandler>filter(((ComponentDefinition)parent).getElements(), RequestHandler.class), _function);
      boolean _not = (!_exists);
      if (_not) {
        String _name = port.getName();
        String _plus = ("No RequestHandler has been defined for the AnswerPort " + _name);
        this.warning(_plus, 
          null, ComponentDefinitionValidator.MISSING_REQUEST_HANDLER);
      }
    }
  }
  
  @Check
  public void checkRequestHandlerHasNoRequestPortLinks(final RequestHandler handler) {
    if (((handler.isActiveQueue() == false) && (IterableExtensions.size(Iterables.<RequestPortLink>filter(handler.getLinks(), RequestPortLink.class)) > 0))) {
      this.warning(
        "A passive request-handler actively initiates requests which can lead to deadlocks.", 
        null, ComponentDefinitionValidator.PASSIVE_REQUEST_HANDLER);
    }
  }
}
