/**
 * Copyright (c) 2019 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtend.datasheet.generator.component;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Injector;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.AbstractDatasheetElement;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.DatasheetProperty;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.DefaultDatasheetProperties;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.GenericDatasheetModel;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.MandatoryDatasheetElement;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.MandatoryDatasheetElementNames;
import org.eclipse.smartmdsd.ecore.component.componentDatasheet.ComponentDatasheet;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.AnswerPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinitionModelUtility;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.InputPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.OutputPort;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.RequestPort;
import org.eclipse.smartmdsd.ecore.service.domainModelsDatasheet.DomainModelsDatasheet;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.CommunicationServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ForkingServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.JoiningServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.OneWayCommunicationService;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ServiceDefRepository;
import org.eclipse.smartmdsd.xtext.service.domainModelsDatasheet.ui.internal.DomainModelsDatasheetActivator;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.documentation.IEObjectDocumentationProvider;
import org.eclipse.xtext.generator.AbstractGenerator;
import org.eclipse.xtext.generator.IFileSystemAccess2;
import org.eclipse.xtext.generator.IGeneratorContext;
import org.eclipse.xtext.resource.FileExtensionProvider;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class ComponentDatasheetGeneratorImpl extends AbstractGenerator {
  @Inject
  private IEObjectDocumentationProvider doc;
  
  public String getRdfDatasheetFilename(final ComponentDatasheet ds) {
    String _name = ds.getComponent().getName();
    return (_name + ".ttl");
  }
  
  @Override
  public void doGenerate(final Resource input, final IFileSystemAccess2 fsa, final IGeneratorContext context) {
    Iterable<ComponentDatasheet> _filter = Iterables.<ComponentDatasheet>filter(IteratorExtensions.<EObject>toIterable(input.getAllContents()), ComponentDatasheet.class);
    for (final ComponentDatasheet datasheet : _filter) {
      fsa.generateFile(this.getRdfDatasheetFilename(datasheet), this.compileRdfDatasheet(datasheet));
    }
  }
  
  public DomainModelsDatasheet getDomainModelsDatasheet(final ServiceDefRepository repo) {
    final Injector injector = DomainModelsDatasheetActivator.getInstance().getInjector(DomainModelsDatasheetActivator.ORG_ECLIPSE_SMARTMDSD_XTEXT_SERVICE_DOMAINMODELSDATASHEET_DOMAINMODELSDATASHEET);
    final XtextResourceSet resourceSet = injector.<XtextResourceSet>getInstance(XtextResourceSet.class);
    resourceSet.addLoadOption(XtextResource.OPTION_RESOLVE_ALL, Boolean.TRUE);
    final URI baseUri = repo.eResource().getURI().trimFileExtension();
    final FileExtensionProvider fileExtProvider = injector.<FileExtensionProvider>getInstance(FileExtensionProvider.class);
    final URI xtextResourceUri = baseUri.appendFileExtension(fileExtProvider.getPrimaryFileExtension());
    final boolean resourceExists = resourceSet.getURIConverter().exists(xtextResourceUri, null);
    if ((resourceExists == true)) {
      final Resource resource = resourceSet.getResource(xtextResourceUri, true);
      boolean _isLoaded = resource.isLoaded();
      boolean _equals = (_isLoaded == true);
      if (_equals) {
        final EObject root_obj = resource.getContents().get(0);
        if ((root_obj instanceof DomainModelsDatasheet)) {
          return ((DomainModelsDatasheet)root_obj);
        }
      }
    } else {
      InputOutput.<String>println(("non-existing resource: " + xtextResourceUri));
    }
    return null;
  }
  
  public ServiceDefRepository getRepo(final CommunicationServiceDefinition svc) {
    EObject _eContainer = svc.eContainer();
    return ((ServiceDefRepository) _eContainer);
  }
  
  public String getBaseURI(final GenericDatasheetModel ds) {
    final Function1<MandatoryDatasheetElement, Boolean> _function = (MandatoryDatasheetElement it) -> {
      return Boolean.valueOf(it.getName().equals(MandatoryDatasheetElementNames.BASE_URI));
    };
    final MandatoryDatasheetElement base_uri = IterableExtensions.<MandatoryDatasheetElement>findFirst(Iterables.<MandatoryDatasheetElement>filter(ds.getElements(), MandatoryDatasheetElement.class), _function);
    if ((base_uri != null)) {
      return base_uri.getValue();
    }
    return "";
  }
  
  public String getShortDescription(final GenericDatasheetModel ds) {
    final Function1<MandatoryDatasheetElement, Boolean> _function = (MandatoryDatasheetElement it) -> {
      return Boolean.valueOf(it.getName().equals(MandatoryDatasheetElementNames.SHORT_DESCRIPTION));
    };
    final MandatoryDatasheetElement base_uri = IterableExtensions.<MandatoryDatasheetElement>findFirst(Iterables.<MandatoryDatasheetElement>filter(ds.getElements(), MandatoryDatasheetElement.class), _function);
    if ((base_uri != null)) {
      return base_uri.getValue();
    }
    return "";
  }
  
  public CharSequence compile(final AbstractDatasheetElement element) {
    CharSequence _xifexpression = null;
    if ((element instanceof DatasheetProperty)) {
      CharSequence _switchResult = null;
      String _name = ((DatasheetProperty)element).getName();
      boolean _matched = false;
      String _literal = DefaultDatasheetProperties.SPDX_LICENSE.getLiteral();
      if (Objects.equal(_name, _literal)) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("# License will contain SPDX License Identifier");
        _builder.newLine();
        _builder.append("# see https://spdx.org/licenses/");
        _builder.newLine();
        _builder.append("t1:license <http://spdx.org/licenses/");
        String _value = ((DatasheetProperty)element).getValue();
        _builder.append(_value);
        _builder.append(".html>;");
        _builder.newLineIfNotEmpty();
        _switchResult = _builder;
      }
      if (!_matched) {
        String _literal_1 = DefaultDatasheetProperties.TECHNOLOGY_READINESS_LEVEL.getLiteral();
        if (Objects.equal(_name, _literal_1)) {
          _matched=true;
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("# Technology Readiness Scale, e.g.");
          _builder_1.newLine();
          _builder_1.append("# http://www.innovationseeds.eu/Virtual_Library/Knowledge/TLR_Scale.kl");
          _builder_1.newLine();
          _builder_1.append("# Level 1 to level 9.");
          _builder_1.newLine();
          _builder_1.append("# plus trl:undefined for undefined TRL");
          _builder_1.newLine();
          _builder_1.append("t1:trl t1:TRL_Level");
          String _value_1 = ((DatasheetProperty)element).getValue();
          int _length = ((DatasheetProperty)element).getValue().length();
          int _minus = (_length - 1);
          String _substring = _value_1.substring(_minus);
          _builder_1.append(_substring);
          _builder_1.append(";");
          _builder_1.newLineIfNotEmpty();
          _switchResult = _builder_1;
        }
      }
      if (!_matched) {
        String _literal_2 = DefaultDatasheetProperties.HOMEPAGE.getLiteral();
        if (Objects.equal(_name, _literal_2)) {
          _matched=true;
          StringConcatenation _builder_2 = new StringConcatenation();
          _builder_2.append("# The webpage with additional information about this component");
          _builder_2.newLine();
          _builder_2.append("t1:homepage \"");
          String _value_2 = ((DatasheetProperty)element).getValue();
          _builder_2.append(_value_2);
          _builder_2.append("\"^^xsd:anyURI;");
          _builder_2.newLineIfNotEmpty();
          _switchResult = _builder_2;
        }
      }
      if (!_matched) {
        StringConcatenation _builder_3 = new StringConcatenation();
        _builder_3.append("# DatasheetProperty ");
        String _name_1 = ((DatasheetProperty)element).getName();
        _builder_3.append(_name_1);
        _builder_3.newLineIfNotEmpty();
        _builder_3.append("t1:");
        String _firstLower = StringExtensions.toFirstLower(((DatasheetProperty)element).getName());
        _builder_3.append(_firstLower);
        _builder_3.append("Description \"");
        String _value_3 = ((DatasheetProperty)element).getValue();
        _builder_3.append(_value_3);
        _builder_3.append("\";");
        _builder_3.newLineIfNotEmpty();
        _switchResult = _builder_3;
      }
      _xifexpression = _switchResult;
    }
    return _xifexpression;
  }
  
  public CharSequence compileRdfDatasheet(final ComponentDatasheet ds) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# baseURI: ");
    String _baseURI = this.getBaseURI(ds);
    _builder.append(_baseURI);
    _builder.append("/ComponentsAndSystems#");
    String _name = ds.getComponent().getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    _builder.append("# imports: http://robmosys.eu/rdf/composition-structures");
    _builder.newLine();
    _builder.append("# imports: http://seronet-projekt.de/models/t2#Navigation");
    _builder.newLine();
    _builder.newLine();
    {
      final Function1<DomainModelsDatasheet, String> _function = (DomainModelsDatasheet c) -> {
        return c.getName().toString();
      };
      List<DomainModelsDatasheet> _sortBy = IterableExtensions.<DomainModelsDatasheet, String>sortBy(this.getAllDomainModelDatasheets(ds), _function);
      for(final DomainModelsDatasheet dm_ds : _sortBy) {
        _builder.append("# imports: ");
        String _baseURI_1 = this.getBaseURI(dm_ds);
        _builder.append(_baseURI_1);
        _builder.append("/DomainModels/");
        String _name_1 = dm_ds.getName();
        _builder.append(_name_1);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("#######################################");
    _builder.newLine();
    _builder.append("# Technical Data Sheet");
    _builder.newLine();
    _builder.append("# for a SeRoNet Software Component as ");
    _builder.newLine();
    _builder.append("# exported by the SmartMDSD Toolchain");
    _builder.newLine();
    _builder.append("#");
    _builder.newLine();
    _builder.append("# erste Zeile des .ttl muss enthalten: \"# ComponentID: <id uri>\"");
    _builder.newLine();
    _builder.append("# This file will be placed in:");
    _builder.newLine();
    _builder.append("# <componentName>/datasheet-gen/<componentName>.ttl");
    _builder.newLine();
    _builder.append("#######################################");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# all generated files use these prefixes");
    _builder.newLine();
    _builder.append("@prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .");
    _builder.newLine();
    _builder.append("@prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .");
    _builder.newLine();
    _builder.append("@prefix t1: <http://seronet-projekt.de/models/t1#> .");
    _builder.newLine();
    _builder.append("@prefix spdx: <http://spdx.org/rdf/terms#> .");
    _builder.newLine();
    _builder.append("@prefix spdx-lic: <http://spdx.org/licenses/> .");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@prefix owl: <http://www.w3.org/2002/07/owl#> .");
    _builder.newLine();
    _builder.append("@prefix sh: <http://www.w3.org/ns/shacl#> .");
    _builder.newLine();
    _builder.append("@prefix xsd: <http://www.w3.org/2001/XMLSchema#> .");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# TODO past “Durchstich”: exactly identify which elements belong to SeRoNet and which belong to RobMoSys");
    _builder.newLine();
    _builder.append("@prefix robmosys: <http://robmosys.eu/rdf/composition-structures#> .");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# these namespaces can be different in generated files");
    _builder.newLine();
    _builder.append("@prefix t3: <");
    String _baseURI_2 = this.getBaseURI(ds);
    _builder.append(_baseURI_2);
    _builder.append("/ComponentsAndSystems#> .");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      final Function1<DomainModelsDatasheet, String> _function_1 = (DomainModelsDatasheet c) -> {
        return c.getName().toString();
      };
      List<DomainModelsDatasheet> _sortBy_1 = IterableExtensions.<DomainModelsDatasheet, String>sortBy(this.getAllDomainModelDatasheets(ds), _function_1);
      for(final DomainModelsDatasheet dm_ds_1 : _sortBy_1) {
        _builder.append("@prefix ");
        String _name_2 = dm_ds_1.getName();
        _builder.append(_name_2);
        _builder.append(": <");
        String _baseURI_3 = this.getBaseURI(dm_ds_1);
        _builder.append(_baseURI_3);
        _builder.append("/DomainModels/");
        String _name_3 = dm_ds_1.getName();
        _builder.append(_name_3);
        _builder.append("#> .");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("##### Software Component ");
    String _name_4 = ds.getComponent().getName();
    _builder.append(_name_4);
    _builder.newLineIfNotEmpty();
    _builder.append("# see https://robmosys.eu/wiki/modeling:metamodels:component");
    _builder.newLine();
    _builder.append("t3:");
    String _name_5 = ds.getComponent().getName();
    _builder.append(_name_5);
    _builder.append(" a robmosys:ComponentDefinition;");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("a owl:Ontology ;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("owl:imports <http://robmosys.eu/rdf/composition-structures> ;");
    _builder.newLine();
    {
      final Function1<DatasheetProperty, Boolean> _function_2 = (DatasheetProperty it) -> {
        return Boolean.valueOf(it.getName().equals(DefaultDatasheetProperties.SPDX_LICENSE.getLiteral()));
      };
      Iterable<DatasheetProperty> _filter = IterableExtensions.<DatasheetProperty>filter(Iterables.<DatasheetProperty>filter(ds.getElements(), DatasheetProperty.class), _function_2);
      for(final DatasheetProperty license : _filter) {
        _builder.append("\t");
        _builder.append("owl:imports <http://spdx.org/licenses/");
        String _value = license.getValue();
        _builder.append(_value, "\t");
        _builder.append(">;");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      final Function1<DomainModelsDatasheet, String> _function_3 = (DomainModelsDatasheet c) -> {
        return c.getName().toString();
      };
      List<DomainModelsDatasheet> _sortBy_2 = IterableExtensions.<DomainModelsDatasheet, String>sortBy(this.getAllDomainModelDatasheets(ds), _function_3);
      for(final DomainModelsDatasheet dm_ds_2 : _sortBy_2) {
        _builder.append("\t");
        _builder.append("owl:imports <");
        String _baseURI_4 = this.getBaseURI(dm_ds_2);
        _builder.append(_baseURI_4, "\t");
        _builder.append("/DomainModels/");
        String _name_6 = dm_ds_2.getName();
        _builder.append(_name_6, "\t");
        _builder.append("> ;");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Human readable name of this component");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Equals with the project name");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rdfs:label \"");
    String _name_7 = ds.getComponent().getName();
    _builder.append(_name_7, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Abstract. 1-sentence, very brief, description about this component.");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("t1:shortDescription \"");
    String _shortDescription = this.getShortDescription(ds);
    _builder.append(_shortDescription, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      if (((ds.getComponent() != null) && (ds.getComponent().getDocumentation() != null))) {
        _builder.append("\t");
        _builder.append("# A long description of this component");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("t1:description \"\"\"");
        _builder.newLine();
        _builder.append("\t");
        String _multilineHtmlDocumentation = ds.getComponent().getMultilineHtmlDocumentation();
        _builder.append(_multilineHtmlDocumentation, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\"\"\";");
        _builder.newLine();
      }
    }
    _builder.append("\t");
    _builder.newLine();
    {
      EList<AbstractDatasheetElement> _elements = ds.getElements();
      for(final AbstractDatasheetElement element : _elements) {
        _builder.append("\t");
        CharSequence _compile = this.compile(element);
        _builder.append(_compile, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Services this component provides or requires");
    _builder.newLine();
    {
      Iterable<ComponentPort> _filter_1 = Iterables.<ComponentPort>filter(ds.getComponent().getElements(), ComponentPort.class);
      for(final ComponentPort port : _filter_1) {
        _builder.append("\t");
        _builder.append("# ComponentPort ");
        String _name_8 = port.getName();
        _builder.append(_name_8, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("t1:");
        String _serviceType = this.getServiceType(port);
        _builder.append(_serviceType, "\t");
        _builder.append(" t3:");
        String _name_9 = ds.getComponent().getName();
        _builder.append(_name_9, "\t");
        _builder.append("_");
        String _name_10 = port.getName();
        _builder.append(_name_10, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append(".");
    _builder.newLine();
    _builder.newLine();
    {
      Iterable<ComponentPort> _filter_2 = Iterables.<ComponentPort>filter(ds.getComponent().getElements(), ComponentPort.class);
      for(final ComponentPort port_1 : _filter_2) {
        CharSequence _compileComponentPort = this.compileComponentPort(ds, port_1);
        _builder.append(_compileComponentPort);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  public String getServiceType(final ComponentPort port) {
    if ((port instanceof OutputPort)) {
      OneWayCommunicationService _service = ((OutputPort)port).getService();
      if ((_service instanceof ForkingServiceDefinition)) {
        return "providedService";
      }
    } else {
      if ((port instanceof InputPort)) {
        final OneWayCommunicationService svc = ((InputPort)port).getService();
        if ((svc instanceof JoiningServiceDefinition)) {
          return "providedService";
        }
      } else {
        if ((port instanceof AnswerPort)) {
          return "providedService";
        }
      }
    }
    return "requiredService";
  }
  
  public CharSequence compileComponentPort(final ComponentDatasheet ds, final ComponentPort port) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# a service definition usage. Here: provided service");
    _builder.newLine();
    _builder.append("# see https://robmosys.eu/wiki/modeling:metamodels:component");
    _builder.newLine();
    _builder.append("t3:");
    String _name = ds.getComponent().getName();
    _builder.append(_name);
    _builder.append("_");
    String _name_1 = port.getName();
    _builder.append(_name_1);
    _builder.append(" ");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("rdf:type ");
    String _name_2 = this.getRepo(ComponentDefinitionModelUtility.getService(port)).getName();
    _builder.append(_name_2, "\t");
    _builder.append(":");
    String _name_3 = ComponentDefinitionModelUtility.getService(port).getName();
    _builder.append(_name_3, "\t");
    _builder.append(" ;");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("# name of this service definition as from the model");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rdfs:label \"");
    String _name_4 = port.getName();
    _builder.append(_name_4, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      String _documentation = this.doc.getDocumentation(port);
      boolean _tripleNotEquals = (_documentation != null);
      if (_tripleNotEquals) {
        _builder.append("\t");
        _builder.append("# verbose description about this service");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("t1:description \"");
        String _documentation_1 = this.doc.getDocumentation(port);
        _builder.append(_documentation_1, "\t");
        _builder.append("\";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append(".");
    _builder.newLine();
    _builder.newLine();
    return _builder;
  }
  
  public CommunicationServiceDefinition getServiceDefinition(final ComponentPort port) {
    CommunicationServiceDefinition _switchResult = null;
    boolean _matched = false;
    if (port instanceof OutputPort) {
      _matched=true;
      _switchResult = ((OutputPort)port).getService();
    }
    if (!_matched) {
      if (port instanceof InputPort) {
        _matched=true;
        _switchResult = ((InputPort)port).getService();
      }
    }
    if (!_matched) {
      if (port instanceof AnswerPort) {
        _matched=true;
        _switchResult = ((AnswerPort)port).getService();
      }
    }
    if (!_matched) {
      if (port instanceof RequestPort) {
        _matched=true;
        _switchResult = ((RequestPort)port).getService();
      }
    }
    return _switchResult;
  }
  
  public Collection<DomainModelsDatasheet> getAllDomainModelDatasheets(final ComponentDatasheet ds) {
    final HashSet<DomainModelsDatasheet> dm_datasheets = new HashSet<DomainModelsDatasheet>();
    Iterable<ComponentPort> _filter = Iterables.<ComponentPort>filter(ds.getComponent().getElements(), ComponentPort.class);
    for (final ComponentPort port : _filter) {
      {
        final DomainModelsDatasheet dm_ds = this.getDomainModelsDatasheet(this.getRepo(ComponentDefinitionModelUtility.getService(port)));
        if ((dm_ds != null)) {
          final Function1<DomainModelsDatasheet, Boolean> _function = (DomainModelsDatasheet it) -> {
            String _name = it.getName();
            String _name_1 = dm_ds.getName();
            return Boolean.valueOf(Objects.equal(_name, _name_1));
          };
          boolean _exists = IterableExtensions.<DomainModelsDatasheet>exists(dm_datasheets, _function);
          boolean _not = (!_exists);
          if (_not) {
            dm_datasheets.add(dm_ds);
          }
        }
      }
    }
    return dm_datasheets;
  }
}
