/**
 * Copyright (c) 2018 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Matthias Lutz, Dennis Stampfer
 */
package org.eclipse.smartmdsd.xtend.smartsoft.generator.system;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.Activity;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinitionModelUtility;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentPort;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.CoordinationSlavePort;
import org.eclipse.smartmdsd.ecore.component.coordinationExtension.PublicOperationMode;
import org.eclipse.smartmdsd.ecore.component.seronetExtension.OpcUaDeviceClient;
import org.eclipse.smartmdsd.ecore.component.seronetExtension.OpcUaReadServer;
import org.eclipse.smartmdsd.ecore.service.roboticMiddleware.ACE_SmartSoft;
import org.eclipse.smartmdsd.ecore.service.roboticMiddleware.RoboticMiddleware;
import org.eclipse.smartmdsd.ecore.system.compArchSeronetExtension.OpcUaDeviceClientInstance;
import org.eclipse.smartmdsd.ecore.system.compArchSeronetExtension.OpcUaReadServerInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ActivityConfigurationMapping;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentArchitectureModelUtility;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.Connection;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ProvidedService;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.RequiredService;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ServiceInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.SystemComponentArchitecture;
import org.eclipse.smartmdsd.xtend.smartsoft.generator.CopyrightHelpers;
import org.eclipse.smartmdsd.xtend.smartsoft.generator.ExtendedOutputConfigurationProvider;
import org.eclipse.smartmdsd.xtend.smartsoft.generator.system.ActivityConfig;
import org.eclipse.smartmdsd.xtend.smartsoft.generator.system.BehaviorSystem;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.generator.AbstractGenerator;
import org.eclipse.xtext.generator.IFileSystemAccess2;
import org.eclipse.xtext.generator.IGeneratorContext;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

@SuppressWarnings("all")
public class SystemGenerator2Impl extends AbstractGenerator {
  @Inject
  @Extension
  private CopyrightHelpers _copyrightHelpers;
  
  @Inject
  @Extension
  private ActivityConfig _activityConfig;
  
  @Inject
  @Extension
  private BehaviorSystem _behaviorSystem;
  
  @Override
  public void doGenerate(final Resource resource, final IFileSystemAccess2 fsa, final IGeneratorContext context) {
    Iterable<SystemComponentArchitecture> _filter = Iterables.<SystemComponentArchitecture>filter(IteratorExtensions.<EObject>toIterable(resource.getAllContents()), SystemComponentArchitecture.class);
    for (final SystemComponentArchitecture model : _filter) {
      {
        fsa.generateFile("CMakeLists.txt", ExtendedOutputConfigurationProvider.SMARTSOFT_OUTPUT, this.compileCMakeLists(model));
        fsa.generateFile("BuildExternalComponents.cmake", ExtendedOutputConfigurationProvider.SRC_GEN_SYS_CONFIG, this.compileBuildExternalComponents(model));
        fsa.generateFile("CoordinationModuleConnections.json", ExtendedOutputConfigurationProvider.SRC_GEN_SYS_CONFIG, this._behaviorSystem.compileModuleConnections(model));
        fsa.generateFile("behaviorProjectFiles.sh", ExtendedOutputConfigurationProvider.SRC_GEN_SYS_CONFIG, this._behaviorSystem.compileBehaviorProjectFiles(model));
        EList<ComponentInstance> _components = model.getComponents();
        for (final ComponentInstance component : _components) {
          String _name = component.getName();
          String _plus = (_name + ".ini");
          fsa.generateFile(_plus, ExtendedOutputConfigurationProvider.SRC_GEN_SYS_CONFIG, this.compileIniFile(component));
        }
      }
    }
  }
  
  public CharSequence compileCMakeLists(final SystemComponentArchitecture system) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("CMAKE_MINIMUM_REQUIRED(VERSION 3.5)");
    _builder.newLine();
    _builder.newLine();
    _builder.append("INCLUDE(src-gen/system/BuildExternalComponents.cmake)");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence compileBuildExternalComponents(final SystemComponentArchitecture system) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("CMAKE_MINIMUM_REQUIRED(VERSION 3.5)");
    _builder.newLine();
    _builder.append("PROJECT(");
    String _name = system.getName();
    _builder.append(_name);
    _builder.append(")");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("# this is a pseudo-target just to trigger CMake builder");
    _builder.newLine();
    _builder.append("ADD_CUSTOM_TARGET(${PROJECT_NAME} ALL)");
    _builder.newLine();
    return _builder;
  }
  
  public String getSmartSoftFolder(final ComponentInstance componentInstance) {
    final IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(componentInstance.getComponent().getName());
    final IFolder smartSoftFolder = proj.getFolder("smartsoft");
    boolean _exists = smartSoftFolder.exists();
    if (_exists) {
      return smartSoftFolder.getLocation().toOSString();
    }
    return "";
  }
  
  public CharSequence compileIniFile(final ComponentInstance compInstance) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("[smartsoft]");
    _builder.newLine();
    String _copyrightHash = this._copyrightHelpers.getCopyrightHash();
    _builder.append(_copyrightHash);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("##########################");
    _builder.newLine();
    _builder.append("# component parameters");
    _builder.newLine();
    _builder.newLine();
    _builder.append("[component]");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# the name of the component for the naming service");
    _builder.newLine();
    _builder.append("name ");
    String _name = compInstance.getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    _builder.append("# the initial MainState of the component");
    _builder.newLine();
    _builder.append("initialMainState ");
    String _initialMainState = this.getInitialMainState(compInstance);
    _builder.append(_initialMainState);
    _builder.newLineIfNotEmpty();
    _builder.append("#defaultScheduler FIFO");
    _builder.newLine();
    _builder.append("#useLogger true");
    _builder.newLine();
    _builder.newLine();
    _builder.append("##########################");
    _builder.newLine();
    _builder.append("# server port parameters");
    _builder.newLine();
    {
      Iterable<ComponentPort> _allServerPorts = ComponentArchitectureModelUtility.getAllServerPorts(compInstance);
      for(final ComponentPort server : _allServerPorts) {
        _builder.newLine();
        _builder.append("[");
        String _name_1 = server.getName();
        _builder.append(_name_1);
        _builder.append("]");
        _builder.newLineIfNotEmpty();
        _builder.append("serviceName ");
        String _name_2 = server.getName();
        _builder.append(_name_2);
        _builder.newLineIfNotEmpty();
        _builder.append("roboticMiddleware ");
        String _selectedMiddlewareString = this.getSelectedMiddlewareString(compInstance, server);
        _builder.append(_selectedMiddlewareString);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("##########################");
    _builder.newLine();
    _builder.append("# client port parameters");
    _builder.newLine();
    {
      Iterable<ComponentPort> _allClientPorts = ComponentArchitectureModelUtility.getAllClientPorts(compInstance);
      for(final ComponentPort client : _allClientPorts) {
        _builder.newLine();
        _builder.append("[");
        String _name_3 = client.getName();
        _builder.append(_name_3);
        _builder.append("]");
        _builder.newLineIfNotEmpty();
        _builder.append("wiringName ");
        String _name_4 = client.getName();
        _builder.append(_name_4);
        _builder.newLineIfNotEmpty();
        {
          boolean _hasConnection = this.hasConnection(compInstance, client);
          if (_hasConnection) {
            _builder.append("serverName ");
            String _name_5 = this.getConnectedServer(compInstance, client).getName();
            _builder.append(_name_5);
            _builder.newLineIfNotEmpty();
            _builder.append("serviceName ");
            String _name_6 = this.getConnectedService(compInstance, client).getName();
            _builder.append(_name_6);
            _builder.newLineIfNotEmpty();
            _builder.append("initialConnect true");
            _builder.newLine();
          } else {
            _builder.append("serverName unknown");
            _builder.newLine();
            _builder.append("serviceName unknown");
            _builder.newLine();
            _builder.append("initialConnect false");
            _builder.newLine();
          }
        }
        _builder.append("roboticMiddleware ");
        String _selectedMiddlewareString_1 = this.getSelectedMiddlewareString(compInstance, client);
        _builder.append(_selectedMiddlewareString_1);
        _builder.newLineIfNotEmpty();
        _builder.append("interval 1");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      Iterable<OpcUaDeviceClient> _filter = Iterables.<OpcUaDeviceClient>filter(compInstance.getComponent().getElements(), OpcUaDeviceClient.class);
      for(final OpcUaDeviceClient opcDeviceClient : _filter) {
        _builder.append("[");
        String _name_7 = opcDeviceClient.getName();
        _builder.append(_name_7);
        _builder.append("]");
        _builder.newLineIfNotEmpty();
        {
          final Function1<OpcUaDeviceClientInstance, Boolean> _function = (OpcUaDeviceClientInstance it) -> {
            OpcUaDeviceClient _deviceClient = it.getDeviceClient();
            return Boolean.valueOf(Objects.equal(_deviceClient, opcDeviceClient));
          };
          boolean _exists = IterableExtensions.<OpcUaDeviceClientInstance>exists(Iterables.<OpcUaDeviceClientInstance>filter(compInstance.getExtensions(), OpcUaDeviceClientInstance.class), _function);
          if (_exists) {
            _builder.append("deviceURI ");
            final Function1<OpcUaDeviceClientInstance, Boolean> _function_1 = (OpcUaDeviceClientInstance it) -> {
              OpcUaDeviceClient _deviceClient = it.getDeviceClient();
              return Boolean.valueOf(Objects.equal(_deviceClient, opcDeviceClient));
            };
            OpcUaDeviceClientInstance _findFirst = IterableExtensions.<OpcUaDeviceClientInstance>findFirst(Iterables.<OpcUaDeviceClientInstance>filter(compInstance.getExtensions(), OpcUaDeviceClientInstance.class), _function_1);
            String _deviceURI = null;
            if (_findFirst!=null) {
              _deviceURI=_findFirst.getDeviceURI();
            }
            _builder.append(_deviceURI);
            _builder.newLineIfNotEmpty();
          } else {
            _builder.append("deviceURI ");
            String _deviceURI_1 = opcDeviceClient.getDeviceURI();
            _builder.append(_deviceURI_1);
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("opcuaXmlFile ");
        String _opcuaXmlFile = opcDeviceClient.getOpcuaXmlFile();
        _builder.append(_opcuaXmlFile);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    {
      Iterable<OpcUaReadServer> _filter_1 = Iterables.<OpcUaReadServer>filter(compInstance.getComponent().getElements(), OpcUaReadServer.class);
      for(final OpcUaReadServer opcStatusServer : _filter_1) {
        _builder.append("[");
        String _name_8 = opcStatusServer.getName();
        _builder.append(_name_8);
        _builder.append("]");
        _builder.newLineIfNotEmpty();
        {
          final Function1<OpcUaReadServerInstance, Boolean> _function_2 = (OpcUaReadServerInstance it) -> {
            OpcUaReadServer _readServer = it.getReadServer();
            return Boolean.valueOf(Objects.equal(_readServer, opcStatusServer));
          };
          boolean _exists_1 = IterableExtensions.<OpcUaReadServerInstance>exists(Iterables.<OpcUaReadServerInstance>filter(compInstance.getExtensions(), OpcUaReadServerInstance.class), _function_2);
          if (_exists_1) {
            _builder.append("portNumber ");
            final Function1<OpcUaReadServerInstance, Boolean> _function_3 = (OpcUaReadServerInstance it) -> {
              OpcUaReadServer _readServer = it.getReadServer();
              return Boolean.valueOf(Objects.equal(_readServer, opcStatusServer));
            };
            int _portNumber = IterableExtensions.<OpcUaReadServerInstance>findFirst(Iterables.<OpcUaReadServerInstance>filter(compInstance.getExtensions(), OpcUaReadServerInstance.class), _function_3).getPortNumber();
            _builder.append(_portNumber);
            _builder.newLineIfNotEmpty();
          } else {
            _builder.append("portNumber ");
            int _portNumber_1 = opcStatusServer.getPortNumber();
            _builder.append(_portNumber_1);
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    _builder.newLine();
    _builder.append("##########################");
    _builder.newLine();
    _builder.append("# activity parameters");
    _builder.newLine();
    {
      Iterable<Activity> _activities = ComponentDefinitionModelUtility.getActivities(compInstance.getComponent());
      for(final Activity activity : _activities) {
        _builder.newLine();
        _builder.append("[");
        String _name_9 = activity.getName();
        _builder.append(_name_9);
        _builder.append("]");
        _builder.newLineIfNotEmpty();
        {
          final Function1<ActivityConfigurationMapping, Boolean> _function_4 = (ActivityConfigurationMapping it) -> {
            Activity _activity = it.getActivity();
            return Boolean.valueOf(Objects.equal(_activity, activity));
          };
          boolean _exists_2 = IterableExtensions.<ActivityConfigurationMapping>exists(Iterables.<ActivityConfigurationMapping>filter(compInstance.getExtensions(), ActivityConfigurationMapping.class), _function_4);
          if (_exists_2) {
            final Function1<ActivityConfigurationMapping, Boolean> _function_5 = (ActivityConfigurationMapping it) -> {
              Activity _activity = it.getActivity();
              return Boolean.valueOf(Objects.equal(_activity, activity));
            };
            CharSequence _compileActivityNode = this._activityConfig.compileActivityNode(IterableExtensions.<ActivityConfigurationMapping>findFirst(Iterables.<ActivityConfigurationMapping>filter(compInstance.getExtensions(), ActivityConfigurationMapping.class), _function_5));
            _builder.append(_compileActivityNode);
            _builder.newLineIfNotEmpty();
          } else {
            CharSequence _compileDefaultActivity = this._activityConfig.compileDefaultActivity(activity);
            _builder.append(_compileDefaultActivity);
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    return _builder;
  }
  
  private String getInitialMainState(final ComponentInstance componentInstance) {
    Iterable<CoordinationSlavePort> _filter = Iterables.<CoordinationSlavePort>filter(componentInstance.getComponent().getElements(), CoordinationSlavePort.class);
    for (final CoordinationSlavePort slave : _filter) {
      final Function1<PublicOperationMode, Boolean> _function = (PublicOperationMode it) -> {
        boolean _isIsDefaultInit = it.isIsDefaultInit();
        return Boolean.valueOf((_isIsDefaultInit == true));
      };
      Iterable<PublicOperationMode> _filter_1 = IterableExtensions.<PublicOperationMode>filter(Iterables.<PublicOperationMode>filter(slave.getElements(), PublicOperationMode.class), _function);
      for (final PublicOperationMode mode : _filter_1) {
        return mode.getName();
      }
    }
    return "Neutral";
  }
  
  private boolean hasConnection(final ComponentInstance component, final ComponentPort service) {
    final Function1<ServiceInstance, Boolean> _function = (ServiceInstance it) -> {
      ComponentPort _port = it.getPort();
      return Boolean.valueOf(Objects.equal(_port, service));
    };
    final ServiceInstance port = IterableExtensions.<ServiceInstance>findFirst(component.getPorts(), _function);
    if ((port instanceof RequiredService)) {
      final EObject parent = component.eContainer();
      if ((parent instanceof SystemComponentArchitecture)) {
        final Function1<Connection, Boolean> _function_1 = (Connection it) -> {
          RequiredService _from = it.getFrom();
          return Boolean.valueOf(Objects.equal(_from, port));
        };
        return IterableExtensions.<Connection>exists(((SystemComponentArchitecture)parent).getConnections(), _function_1);
      }
    }
    return false;
  }
  
  private ProvidedService getConnectedService(final ComponentInstance component, final ComponentPort service) {
    final Function1<ServiceInstance, Boolean> _function = (ServiceInstance it) -> {
      ComponentPort _port = it.getPort();
      return Boolean.valueOf(Objects.equal(_port, service));
    };
    final ServiceInstance port = IterableExtensions.<ServiceInstance>findFirst(component.getPorts(), _function);
    if ((port instanceof RequiredService)) {
      final EObject parent = component.eContainer();
      if ((parent instanceof SystemComponentArchitecture)) {
        final Function1<Connection, Boolean> _function_1 = (Connection it) -> {
          RequiredService _from = it.getFrom();
          return Boolean.valueOf(Objects.equal(_from, port));
        };
        final Connection connection = IterableExtensions.<Connection>findFirst(((SystemComponentArchitecture)parent).getConnections(), _function_1);
        if ((connection != null)) {
          return connection.getTo();
        }
      }
    }
    return null;
  }
  
  private ComponentInstance getConnectedServer(final ComponentInstance component, final ComponentPort service) {
    final ProvidedService connectedSvc = this.getConnectedService(component, service);
    if ((connectedSvc != null)) {
      final EObject parent = connectedSvc.eContainer();
      if ((parent instanceof ComponentInstance)) {
        return ((ComponentInstance)parent);
      }
    }
    return null;
  }
  
  public String getSelectedMiddlewareString(final ComponentInstance component, final ComponentPort service) {
    final EObject parent = component.eContainer();
    if ((parent instanceof SystemComponentArchitecture)) {
      final Function1<ServiceInstance, Boolean> _function = (ServiceInstance it) -> {
        ComponentPort _port = it.getPort();
        return Boolean.valueOf(Objects.equal(_port, service));
      };
      final ServiceInstance port = IterableExtensions.<ServiceInstance>findFirst(component.getPorts(), _function);
      if ((port instanceof RequiredService)) {
        final Function1<Connection, Boolean> _function_1 = (Connection it) -> {
          RequiredService _from = it.getFrom();
          return Boolean.valueOf(Objects.equal(_from, port));
        };
        final Connection connection = IterableExtensions.<Connection>findFirst(((SystemComponentArchitecture)parent).getConnections(), _function_1);
        if ((connection != null)) {
          RoboticMiddleware _middlewareSelection = connection.getMiddlewareSelection();
          boolean _tripleNotEquals = (_middlewareSelection != null);
          if (_tripleNotEquals) {
            return connection.getMiddlewareSelection().eClass().getName();
          }
        }
      } else {
        if ((port instanceof ProvidedService)) {
          final Function1<Connection, Boolean> _function_2 = (Connection it) -> {
            ProvidedService _to = it.getTo();
            return Boolean.valueOf(Objects.equal(_to, port));
          };
          Iterable<Connection> _filter = IterableExtensions.<Connection>filter(((SystemComponentArchitecture)parent).getConnections(), _function_2);
          for (final Connection connection_1 : _filter) {
            RoboticMiddleware _middlewareSelection_1 = connection_1.getMiddlewareSelection();
            boolean _tripleNotEquals_1 = (_middlewareSelection_1 != null);
            if (_tripleNotEquals_1) {
              return connection_1.getMiddlewareSelection().eClass().getName();
            }
          }
        }
      }
    }
    return ACE_SmartSoft.class.getSimpleName();
  }
}
