/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.base.basicAttributes.validation;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AbstractAttributeType;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AbstractValue;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.ArrayValue;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AttributeDefinition;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.AttributeRefinement;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.BasicAttributesPackage;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.BasicAttributesTypeConformance;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.EnumerationElement;
import org.eclipse.smartmdsd.xtext.base.basicAttributes.validation.AbstractBasicAttributesValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Conversions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class BasicAttributesValidator extends AbstractBasicAttributesValidator {
  @Inject
  private BasicAttributesTypeConformance conf;
  
  protected static final String ISSUE_PREFIX = "org.xtext.service.communicationObject.";
  
  public static final String INCOMPATIBLE_TYPES = (BasicAttributesValidator.ISSUE_PREFIX + "IncompatibleTypes");
  
  public static final String INCOMPATIBLE_CARDINALITY = (BasicAttributesValidator.ISSUE_PREFIX + "IncompatibleCardinality");
  
  public static final String SMALL_ATTR_NAME = (BasicAttributesValidator.ISSUE_PREFIX + "SmallAttributeName");
  
  public static final String CPP_KEYWORD_ATTR_NAME = (BasicAttributesValidator.ISSUE_PREFIX + "CppKeywordAttributeName");
  
  public static final String RESERVED_ENUM_NAME = (BasicAttributesValidator.ISSUE_PREFIX + "ReservedEnumName");
  
  public Boolean isCompatible(final AbstractValue av, final AbstractAttributeType type) {
    return Boolean.valueOf(this.conf.isCompatible(av, type));
  }
  
  @Check
  public void checkTypeConformance(final AbstractValue av) {
    EObject parent = av.eContainer();
    if ((parent instanceof ArrayValue)) {
      parent = ((ArrayValue)parent).eContainer();
    }
    if ((parent instanceof AttributeDefinition)) {
      Boolean _isCompatible = this.isCompatible(av, ((AttributeDefinition)parent).getType());
      boolean _not = (!(_isCompatible).booleanValue());
      if (_not) {
        String _cardinalityName = this.conf.getCardinalityName(((AttributeDefinition)parent).getType());
        String _plus = ("Incompatible types. Expected " + _cardinalityName);
        String _plus_1 = (_plus + 
          ", but assigned is ");
        String _valueTypeName = this.conf.getValueTypeName(av);
        String _plus_2 = (_plus_1 + _valueTypeName);
        String _plus_3 = (_plus_2 + ".");
        this.error(_plus_3, 
          null, BasicAttributesValidator.INCOMPATIBLE_TYPES);
      }
    } else {
      if ((parent instanceof AttributeRefinement)) {
        Boolean _isCompatible_1 = this.isCompatible(av, ((AttributeRefinement)parent).getAttribute().getType());
        boolean _not_1 = (!(_isCompatible_1).booleanValue());
        if (_not_1) {
          String _cardinalityName_1 = this.conf.getCardinalityName(((AttributeRefinement)parent).getAttribute().getType());
          String _plus_4 = ("Incompatible types. Expected " + _cardinalityName_1);
          String _plus_5 = (_plus_4 + 
            ", but assigned is ");
          String _valueTypeName_1 = this.conf.getValueTypeName(av);
          String _plus_6 = (_plus_5 + _valueTypeName_1);
          String _plus_7 = (_plus_6 + ".");
          this.error(_plus_7, 
            null, BasicAttributesValidator.INCOMPATIBLE_TYPES);
        }
      }
    }
  }
  
  @Check
  public void checkCardinality(final AbstractValue av) {
    EObject parent = av.eContainer();
    if ((parent instanceof ArrayValue)) {
      parent = ((ArrayValue)parent).eContainer();
    }
    if ((av instanceof ArrayValue)) {
      AbstractAttributeType type = null;
      if ((parent instanceof AttributeDefinition)) {
        type = ((AttributeDefinition)parent).getType();
      } else {
        if ((parent instanceof AttributeRefinement)) {
          type = ((AttributeRefinement)parent).getAttribute().getType();
        }
      }
      if (((type != null) && (type.getArray() != null))) {
        final int numberValues = ((Object[])Conversions.unwrapArray(((ArrayValue)av).getValues(), Object.class)).length;
        final String length = type.getArray().getLength();
        if (((length != null) && (!Objects.equal(length, "*")))) {
          final Integer arrayLength = Integer.valueOf(length);
          if ((numberValues > (arrayLength).intValue())) {
            this.error(
              (((("Array length mismatch. Assigned array-value of size " + Integer.valueOf(numberValues)) + 
                " to a static array-type of size ") + length) + "."), 
              null, BasicAttributesValidator.INCOMPATIBLE_CARDINALITY);
          }
        }
      }
    }
  }
  
  @Check
  public void checkAttributeStartsWithSmallLetter(final AttributeDefinition attr) {
    boolean _isLowerCase = Character.isLowerCase(attr.getName().charAt(0));
    boolean _not = (!_isLowerCase);
    if (_not) {
      this.warning("Element name should start with a small letter!", 
        BasicAttributesPackage.Literals.ATTRIBUTE_DEFINITION__NAME, 
        BasicAttributesValidator.SMALL_ATTR_NAME);
    }
  }
  
  @Check
  public void checkCppReservedKeywords(final AttributeDefinition elem) {
    boolean _contains = BasicAttributesTypeConformance.cppKeywords.contains(elem.getName());
    if (_contains) {
      this.error("Element name must not be a C++ reserved keyword.", 
        BasicAttributesPackage.Literals.ATTRIBUTE_DEFINITION__NAME, 
        BasicAttributesValidator.CPP_KEYWORD_ATTR_NAME);
    }
  }
  
  @Check
  public void checkEnumNameNotDefaultValue(final EnumerationElement en) {
    boolean _equals = en.getName().equals("ENUM_VALUE_UNDEFINED");
    if (_equals) {
      this.error("This EnumerationElement is not allowed (it is reserved for internal usage).", 
        BasicAttributesPackage.Literals.ENUMERATION_ELEMENT__NAME, 
        BasicAttributesValidator.RESERVED_ENUM_NAME);
    }
  }
}
