/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.indexer;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IContainer;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This is a helper class to streamline access to the global Xtext indexer.
 * The implementation is inspired from the smalljava example available at:
 * https://github.com/LorenzoBettini/packtpub-xtext-book-2nd-examples
 */
@SuppressWarnings("all")
public class XtextResourceIndex {
  @Inject
  private ResourceDescriptionsProvider rdp;
  
  @Inject
  private IContainer.Manager cm;
  
  /**
   * Get IResourceDescription for the given EObject from the Xtext index.
   */
  public IResourceDescription getResourceDescription(final EObject o) {
    final IResourceDescriptions index = this.rdp.getResourceDescriptions(o.eResource());
    final IResourceDescription description = index.getResourceDescription(o.eResource().getURI());
    return description;
  }
  
  /**
   * Get a list of IEObjectDescription for all the exported
   * EObject descriptions in the Xtext index
   */
  public Iterable<IEObjectDescription> getExportedEObjectDescriptions(final EObject o) {
    return this.getResourceDescription(o).getExportedObjects();
  }
  
  /**
   * Get all exported IEObjectDescription from the Xtext index
   * of a certain type.
   */
  public Iterable<IEObjectDescription> getExportedEObjectDescriptionsByType(final EObject o, final EClass type) {
    return this.getResourceDescription(o).getExportedObjectsByType(type);
  }
  
  /**
   * Get visible IContainer resources for a given EObject
   */
  public List<IContainer> getVisibleContainers(final EObject o) {
    List<IContainer> _xblockexpression = null;
    {
      final IResourceDescriptions index = this.rdp.getResourceDescriptions(o.eResource());
      final IResourceDescription rd = index.getResourceDescription(o.eResource().getURI());
      _xblockexpression = this.cm.getVisibleContainers(rd, index);
    }
    return _xblockexpression;
  }
  
  /**
   * Get IEObjectDescriptions from all visible IContainers
   * for a given EClass type.
   */
  public Iterable<IEObjectDescription> getVisibleEObjectDescriptions(final EObject o, final EClass type) {
    final Function1<IContainer, Iterable<IEObjectDescription>> _function = (IContainer container) -> {
      return container.getExportedObjectsByType(type);
    };
    return Iterables.<IEObjectDescription>concat(ListExtensions.<IContainer, Iterable<IEObjectDescription>>map(this.getVisibleContainers(o), _function));
  }
  
  /**
   * Get IEObjectDescriptions from all visible IContainers
   * for a given EClass type, excluding those from the local resource.
   */
  public Map<QualifiedName, IEObjectDescription> getVisibleExternalDescriptions(final EObject o, final EClass type) {
    final Iterable<IEObjectDescription> allVisibleClasses = this.getVisibleEObjectDescriptions(o, type);
    final Iterable<IEObjectDescription> allExportedClasses = this.getExportedEObjectDescriptionsByType(o, type);
    final Set<IEObjectDescription> difference = IterableExtensions.<IEObjectDescription>toSet(allVisibleClasses);
    difference.removeAll(IterableExtensions.<IEObjectDescription>toSet(allExportedClasses));
    final Function1<IEObjectDescription, QualifiedName> _function = (IEObjectDescription it) -> {
      return it.getQualifiedName();
    };
    return IterableExtensions.<QualifiedName, IEObjectDescription>toMap(difference, _function);
  }
}
