/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.service.parameterDefinition.validation;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.base.basicAttributes.BasicAttributesTypeConformance;
import org.eclipse.smartmdsd.ecore.service.parameterDefinition.AbstractParameter;
import org.eclipse.smartmdsd.ecore.service.parameterDefinition.ParameterDefinition;
import org.eclipse.smartmdsd.ecore.service.parameterDefinition.ParameterDefinitionPackage;
import org.eclipse.smartmdsd.ecore.service.parameterDefinition.ParameterSetDefinition;
import org.eclipse.smartmdsd.ecore.service.parameterDefinition.TriggerDefinition;
import org.eclipse.smartmdsd.xtext.service.parameterDefinition.ParameterDefinitionDefaultLib;
import org.eclipse.smartmdsd.xtext.service.parameterDefinition.ParameterDefinitionXtextUtils;
import org.eclipse.smartmdsd.xtext.service.parameterDefinition.validation.AbstractParameterDefinitionValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class ParameterDefinitionValidator extends AbstractParameterDefinitionValidator {
  @Inject
  @Extension
  private ParameterDefinitionXtextUtils _parameterDefinitionXtextUtils;
  
  protected static final String ISSUE_PREFIX = "org.xtext.service.parameterDefinition.";
  
  public static final String RESERVED_PARAMETER_SET_NAME = (ParameterDefinitionValidator.ISSUE_PREFIX + "ReservedParameterSetName");
  
  public static final String PARAM_SET_HIERARCHY = (ParameterDefinitionValidator.ISSUE_PREFIX + "ParemeterSetHierarchy");
  
  public static final String PARAMETER_DUPLICATE = (ParameterDefinitionValidator.ISSUE_PREFIX + "ParameterDuplicate");
  
  public static final String TRIGGER_DUPLICATE = (ParameterDefinitionValidator.ISSUE_PREFIX + "TriggerDuplicate");
  
  public static final String CPP_KEYWORD_PARAM_SET_NAME = (ParameterDefinitionValidator.ISSUE_PREFIX + "CppKeywordParamSetName");
  
  public static final String CPP_KEYWORD_PARAM_NAME = (ParameterDefinitionValidator.ISSUE_PREFIX + "CppKeywordParamName");
  
  @Check
  public void checkReservedParameterSetNames(final ParameterSetDefinition s) {
    String _name = s.getName();
    String fqn = ((ParameterDefinitionDefaultLib.DEFAULT_PARAM_PACKAGE + ".") + _name);
    boolean _equals = Objects.equal(fqn, ParameterDefinitionDefaultLib.DEFAULT_PARAMETR_SET);
    if (_equals) {
      String _name_1 = s.getName();
      String _plus = ("\'" + _name_1);
      String _plus_1 = (_plus + "\' is a reserved library name");
      this.warning(_plus_1, 
        ParameterDefinitionPackage.Literals.PARAMETER_SET_DEFINITION__NAME, 
        ParameterDefinitionValidator.RESERVED_PARAMETER_SET_NAME);
    }
  }
  
  @Check
  public void checkParameterSetHierarchy(final ParameterSetDefinition s) {
    EList<ParameterSetDefinition> _extends = s.getExtends();
    for (final ParameterSetDefinition next : _extends) {
      if ((next == s)) {
        String _name = s.getName();
        String _plus = ("Cycle in extensions, \'" + _name);
        String _plus_1 = (_plus + "\' cannot extend itself");
        this.error(_plus_1, 
          ParameterDefinitionPackage.Literals.PARAMETER_SET_DEFINITION__EXTENDS, 
          s.getExtends().indexOf(next), 
          ParameterDefinitionValidator.PARAM_SET_HIERARCHY);
      } else {
        boolean _contains = this._parameterDefinitionXtextUtils.getParameterSetHierarchy(next).contains(s);
        if (_contains) {
          String _name_1 = next.getName();
          String _plus_2 = ("Cycle in extensions hierarchy of ParameterSet \'" + _name_1);
          String _plus_3 = (_plus_2 + "\'");
          this.error(_plus_3, 
            ParameterDefinitionPackage.Literals.PARAMETER_SET_DEFINITION__EXTENDS, 
            s.getExtends().indexOf(next), 
            ParameterDefinitionValidator.PARAM_SET_HIERARCHY);
        }
      }
    }
  }
  
  @Check
  public void checkDulpcateInheritedParameters(final ParameterDefinition param) {
    EObject _eContainer = param.eContainer();
    final ParameterSetDefinition parent = ((ParameterSetDefinition) _eContainer);
    final Iterable<ParameterDefinition> visibleParentParameters = this._parameterDefinitionXtextUtils.getParameterHierarchy(parent);
    final Function1<ParameterDefinition, Boolean> _function = (ParameterDefinition it) -> {
      String _name = it.getName();
      String _name_1 = param.getName();
      return Boolean.valueOf(Objects.equal(_name, _name_1));
    };
    boolean _exists = IterableExtensions.<ParameterDefinition>exists(visibleParentParameters, _function);
    if (_exists) {
      this.warning("This Parameter is a duplicate for one of the inherited Parameters", 
        ParameterDefinitionPackage.Literals.ABSTRACT_PARAMETER__NAME, 
        ParameterDefinitionValidator.PARAMETER_DUPLICATE);
    }
  }
  
  @Check
  public void checkDulpcateInheritedTriggers(final TriggerDefinition trigger) {
    EObject _eContainer = trigger.eContainer();
    final ParameterSetDefinition parent = ((ParameterSetDefinition) _eContainer);
    final Iterable<TriggerDefinition> visibleParentTriggers = this._parameterDefinitionXtextUtils.getTriggerHierarchy(parent);
    final Function1<TriggerDefinition, Boolean> _function = (TriggerDefinition it) -> {
      String _name = it.getName();
      String _name_1 = trigger.getName();
      return Boolean.valueOf(Objects.equal(_name, _name_1));
    };
    boolean _exists = IterableExtensions.<TriggerDefinition>exists(visibleParentTriggers, _function);
    if (_exists) {
      this.warning("This Trigger is a duplicate for one of the inherited Triggers", 
        ParameterDefinitionPackage.Literals.ABSTRACT_PARAMETER__NAME, 
        ParameterDefinitionValidator.PARAMETER_DUPLICATE);
    }
  }
  
  @Check
  public void checkCppReservedKeywords(final ParameterSetDefinition elem) {
    boolean _contains = BasicAttributesTypeConformance.cppKeywords.contains(elem.getName());
    if (_contains) {
      this.error("Element name must not be a C++ reserved keyword.", 
        ParameterDefinitionPackage.Literals.PARAMETER_SET_DEFINITION__NAME, 
        ParameterDefinitionValidator.CPP_KEYWORD_PARAM_SET_NAME);
    }
  }
  
  @Check
  public void checkCppReservedKeywords(final AbstractParameter elem) {
    boolean _contains = BasicAttributesTypeConformance.cppKeywords.contains(elem.getName());
    if (_contains) {
      this.error("Element name must not be a C++ reserved keyword.", 
        ParameterDefinitionPackage.Literals.ABSTRACT_PARAMETER__NAME, 
        ParameterDefinitionValidator.CPP_KEYWORD_PARAM_NAME);
    }
  }
}
