/**
 * Copyright (c) 2018 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.service.serviceDefinition.ui.quickfix;

import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.smartmdsd.ecore.service.coordinationPattern.CoordinationPatternFactory;
import org.eclipse.smartmdsd.ecore.service.coordinationPattern.ParameterPattern;
import org.eclipse.smartmdsd.ecore.service.coordinationPattern.StatePattern;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.CoordinationServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ServiceDefRepository;
import org.eclipse.smartmdsd.xtext.base.stateMachine.StateMachineDefaultLifecycleLib;
import org.eclipse.smartmdsd.xtext.service.parameterDefinition.ParameterDefinitionDefaultLib;
import org.eclipse.smartmdsd.xtext.service.serviceDefinition.validation.ServiceDefinitionValidator;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.xtext.ui.editor.XtextEditor;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.model.edit.IModification;
import org.eclipse.xtext.ui.editor.model.edit.IModificationContext;
import org.eclipse.xtext.ui.editor.model.edit.ISemanticModification;
import org.eclipse.xtext.ui.editor.quickfix.DefaultQuickfixProvider;
import org.eclipse.xtext.ui.editor.quickfix.Fix;
import org.eclipse.xtext.ui.editor.quickfix.IssueResolutionAcceptor;
import org.eclipse.xtext.validation.Issue;

/**
 * Custom quickfixes.
 * 
 * See https://www.eclipse.org/Xtext/documentation/310_eclipse_support.html#quick-fixes
 */
@SuppressWarnings("all")
public class ServiceDefinitionQuickfixProvider extends DefaultQuickfixProvider {
  @Inject
  private ParameterDefinitionDefaultLib paramLib;
  
  @Inject
  private StateMachineDefaultLifecycleLib stateLib;
  
  @Fix(ServiceDefinitionValidator.MISSING_PARAMETER_PATTERN)
  public void missingParameterPattern(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      final CoordinationServiceDefinition service = ((CoordinationServiceDefinition) element);
      final ParameterPattern parameter = CoordinationPatternFactory.eINSTANCE.createParameterPattern();
      parameter.setParameterSet(this.paramLib.getDefaultParameterSetObject(service));
      service.setParameterPattern(parameter);
    };
    acceptor.accept(issue, 
      "Add default ParameterPattern definition", 
      "Add default ParameterPattern definition that uses the EmptyDefaultParameterSet", 
      "", _function);
    final IEditorPart editor = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActiveEditor();
    if ((editor instanceof XtextEditor)) {
      ISourceViewer _internalSourceViewer = ((XtextEditor)editor).getInternalSourceViewer();
      ((SourceViewer) _internalSourceViewer).doOperation(ISourceViewer.FORMAT);
    }
  }
  
  @Fix(ServiceDefinitionValidator.MISSING_STATE_PATTERN)
  public void missingStatePattern(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      final CoordinationServiceDefinition service = ((CoordinationServiceDefinition) element);
      service.setStatePattern(CoordinationPatternFactory.eINSTANCE.createStatePattern());
      StatePattern _statePattern = service.getStatePattern();
      _statePattern.setLifecycle(this.stateLib.getDefaultComponentLifecycleObject(service));
    };
    acceptor.accept(issue, 
      "Add default StatePattern definition", 
      "Add default StatePattern definition that uses the default ComponentLifecycle", 
      "", _function);
    final IEditorPart editor = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActiveEditor();
    if ((editor instanceof XtextEditor)) {
      ISourceViewer _internalSourceViewer = ((XtextEditor)editor).getInternalSourceViewer();
      ((SourceViewer) _internalSourceViewer).doOperation(ISourceViewer.FORMAT);
    }
  }
  
  @Fix(ServiceDefinitionValidator.INVALID_REPO_NAME)
  public void repoName(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      ((ServiceDefRepository) element).setName(issue.getData()[0]);
    };
    acceptor.accept(issue, 
      "Fix repository name to match project name", 
      "Repository name will be set to project name", 
      "", _function);
  }
  
  @Fix(ServiceDefinitionValidator.CAPITAL_SERVICE_NAME)
  public void capitalizeName(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final IModification _function = (IModificationContext context) -> {
      final IXtextDocument xtextDocument = context.getXtextDocument();
      final String firstLetter = xtextDocument.get((issue.getOffset()).intValue(), 1);
      xtextDocument.replace((issue.getOffset()).intValue(), 1, firstLetter.toUpperCase());
    };
    acceptor.accept(issue, "Capitalize name", "Capitalize the name.", "", _function);
  }
}
