/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.componentArchitecture.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentPort;
import org.eclipse.smartmdsd.ecore.component.seronetExtension.SupportedMiddleware;
import org.eclipse.smartmdsd.ecore.service.roboticMiddleware.RoboticMiddleware;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentArchitectureModelUtility;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentArchitecturePackage;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.Connection;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ProvidedService;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.RequiredService;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ServiceInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.SystemComponentArchitecture;
import org.eclipse.smartmdsd.ecore.system.systemParameter.ComponentParameterInstance;
import org.eclipse.smartmdsd.ecore.system.systemParameter.ParameterStructInstance;
import org.eclipse.smartmdsd.ecore.system.systemParameter.SystemParamModel;
import org.eclipse.smartmdsd.ecore.system.systemParameter.SystemParameterPackage;
import org.eclipse.smartmdsd.xtext.system.componentArchitecture.validation.AbstractComponentArchitectureValidator;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class ComponentArchitectureValidator extends AbstractComponentArchitectureValidator {
  @Inject
  private IQualifiedNameProvider fqn_provider;
  
  protected static final String COMP_ARCH_ISSUE_PREFIX = "org.xtext.system.componentArchitecture.";
  
  public static final String INVALID_COMP_NAME = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "InvalidName");
  
  public static final String INCOMPATIBLE_SERVICE = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "IncompatibleService");
  
  public static final String MISSING_PORTS = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "MissingPorts");
  
  public static final String NOT_CONNECTED_SERVICE = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "NotConenctedService");
  
  public static final String MIDDLWARE_SELECTION_NOT_SUPPORTED = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "MiddlewareSelectionNotSupported");
  
  public static final String CONFLICTING_MIDDLWARE_SELECTION = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "ConflictingMiddlewareSelection");
  
  public static final String DUPLICATE_PARAMETER_STRUCT = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "DuplicateParameterStruct");
  
  public static final String INVALID_PARAMETER_STRUCT = (ComponentArchitectureValidator.COMP_ARCH_ISSUE_PREFIX + "InvalidParameterStruct");
  
  @Check
  public void checkCompNameStartsWithCapital(final ComponentInstance comp) {
    boolean _isUpperCase = Character.isUpperCase(comp.getName().charAt(0));
    boolean _not = (!_isUpperCase);
    if (_not) {
      this.warning("Name should start with a capital", 
        ComponentArchitecturePackage.Literals.COMPONENT_INSTANCE__NAME, 
        ComponentArchitectureValidator.INVALID_COMP_NAME);
    }
  }
  
  @Check
  public void checkServiceCompatibility(final Connection connection) {
    boolean _servicePortsCompatible = ComponentArchitectureModelUtility.servicePortsCompatible(connection.getFrom(), connection.getTo());
    boolean _equals = (_servicePortsCompatible == false);
    if (_equals) {
      this.error("Connected to an incompatible service provider.", 
        ComponentArchitecturePackage.Literals.CONNECTION__TO, 
        ComponentArchitectureValidator.INCOMPATIBLE_SERVICE);
    }
  }
  
  @Check
  public void checkAllRequiredPortsInstantiated(final ComponentInstance component) {
    final Iterable<ComponentPort> services = ComponentArchitectureModelUtility.getNonOptionalClientPorts(component);
    final Function1<ComponentPort, Boolean> _function = (ComponentPort svc) -> {
      final Function1<ServiceInstance, Boolean> _function_1 = (ServiceInstance it) -> {
        ComponentPort _port = it.getPort();
        return Boolean.valueOf(Objects.equal(_port, svc));
      };
      return Boolean.valueOf(IterableExtensions.<ServiceInstance>exists(component.getPorts(), _function_1));
    };
    boolean _forall = IterableExtensions.<ComponentPort>forall(services, _function);
    boolean _not = (!_forall);
    if (_not) {
      this.warning("Some non-optional component-ports have not been instantiated!", 
        ComponentArchitecturePackage.Literals.COMPONENT_INSTANCE__NAME, 
        ComponentArchitectureValidator.MISSING_PORTS);
    }
  }
  
  @Check
  public void checkRequiredPortConnected(final RequiredService svc) {
    final EObject parent = svc.eContainer().eContainer();
    if ((parent instanceof SystemComponentArchitecture)) {
      final Function1<Connection, Boolean> _function = (Connection it) -> {
        RequiredService _from = it.getFrom();
        return Boolean.valueOf(Objects.equal(_from, svc));
      };
      boolean _exists = IterableExtensions.<Connection>exists(((SystemComponentArchitecture)parent).getConnections(), _function);
      boolean _not = (!_exists);
      if (_not) {
        this.warning("RequiredService has not yet been connected to any ProvidedService!", 
          ComponentArchitecturePackage.Literals.SERVICE_INSTANCE__PORT, 
          ComponentArchitectureValidator.NOT_CONNECTED_SERVICE);
      }
    }
  }
  
  @Check
  public void checkSupportedMiddlewareSelection(final Connection connection) {
    RoboticMiddleware _middlewareSelection = connection.getMiddlewareSelection();
    boolean _tripleNotEquals = (_middlewareSelection != null);
    if (_tripleNotEquals) {
      final Function1<SupportedMiddleware, Boolean> _function = (SupportedMiddleware it) -> {
        EClass _eClass = it.getMiddleware().eClass();
        EClass _eClass_1 = connection.getMiddlewareSelection().eClass();
        return Boolean.valueOf(Objects.equal(_eClass, _eClass_1));
      };
      boolean _exists = IterableExtensions.<SupportedMiddleware>exists(Iterables.<SupportedMiddleware>filter(connection.getFrom().getPort().getExtensions(), SupportedMiddleware.class), _function);
      boolean _not = (!_exists);
      if (_not) {
        String _name = connection.getMiddlewareSelection().eClass().getName();
        String _plus = ("Selected middleware " + _name);
        String _plus_1 = (_plus + " is not supported by the required service ");
        String _name_1 = connection.getFrom().getName();
        String _plus_2 = (_plus_1 + _name_1);
        this.error(_plus_2, 
          ComponentArchitecturePackage.Literals.CONNECTION__MIDDLEWARE_SELECTION, 
          ComponentArchitectureValidator.MIDDLWARE_SELECTION_NOT_SUPPORTED);
      }
      final Function1<SupportedMiddleware, Boolean> _function_1 = (SupportedMiddleware it) -> {
        EClass _eClass = it.getMiddleware().eClass();
        EClass _eClass_1 = connection.getMiddlewareSelection().eClass();
        return Boolean.valueOf(Objects.equal(_eClass, _eClass_1));
      };
      boolean _exists_1 = IterableExtensions.<SupportedMiddleware>exists(Iterables.<SupportedMiddleware>filter(connection.getTo().getPort().getExtensions(), SupportedMiddleware.class), _function_1);
      boolean _not_1 = (!_exists_1);
      if (_not_1) {
        String _name_2 = connection.getMiddlewareSelection().eClass().getName();
        String _plus_3 = ("Selected middleware " + _name_2);
        String _plus_4 = (_plus_3 + " is not supported by the provided service ");
        String _name_3 = connection.getTo().getName();
        String _plus_5 = (_plus_4 + _name_3);
        this.error(_plus_5, 
          ComponentArchitecturePackage.Literals.CONNECTION__MIDDLEWARE_SELECTION, 
          ComponentArchitectureValidator.MIDDLWARE_SELECTION_NOT_SUPPORTED);
      }
    }
  }
  
  @Check
  public void checkConflictingMiddlewareSelection(final Connection connection) {
    RoboticMiddleware _middlewareSelection = connection.getMiddlewareSelection();
    boolean _tripleNotEquals = (_middlewareSelection != null);
    if (_tripleNotEquals) {
      final EObject parent = connection.eContainer();
      if ((parent instanceof SystemComponentArchitecture)) {
        EList<Connection> _connections = ((SystemComponentArchitecture)parent).getConnections();
        for (final Connection other : _connections) {
          boolean _notEquals = (!Objects.equal(connection, other));
          if (_notEquals) {
            ProvidedService _to = connection.getTo();
            ProvidedService _to_1 = other.getTo();
            boolean _equals = Objects.equal(_to, _to_1);
            if (_equals) {
              RoboticMiddleware _middlewareSelection_1 = other.getMiddlewareSelection();
              boolean _tripleNotEquals_1 = (_middlewareSelection_1 != null);
              if (_tripleNotEquals_1) {
                EClass _eClass = connection.getMiddlewareSelection().eClass();
                EClass _eClass_1 = other.getMiddlewareSelection().eClass();
                boolean _notEquals_1 = (!Objects.equal(_eClass, _eClass_1));
                if (_notEquals_1) {
                  String _name = connection.getMiddlewareSelection().eClass().getName();
                  String _plus = ("Selected middleware " + _name);
                  String _plus_1 = (_plus + " conflicts with another selection for the same service ");
                  String _name_1 = connection.getTo().getName();
                  String _plus_2 = (_plus_1 + _name_1);
                  this.error(_plus_2, 
                    ComponentArchitecturePackage.Literals.CONNECTION__MIDDLEWARE_SELECTION, 
                    ComponentArchitectureValidator.CONFLICTING_MIDDLWARE_SELECTION);
                }
              }
            }
          }
        }
      }
    }
  }
  
  @Check
  public void checkSingleParamStruct(final ParameterStructInstance param) {
    final EObject parent = param.eContainer();
    if ((parent instanceof ComponentInstance)) {
      int _size = IterableExtensions.size(Iterables.<ParameterStructInstance>filter(((ComponentInstance)parent).getExtensions(), ParameterStructInstance.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple ParameterStructs defined, but at most one is allowed.", 
          SystemParameterPackage.Literals.PARAMETER_STRUCT_INSTANCE__PARAMETER, 
          ComponentArchitectureValidator.DUPLICATE_PARAMETER_STRUCT);
      }
    }
  }
  
  @Check
  public void checkCompatibleParamStruct(final ParameterStructInstance paramInstance) {
    final EObject compInstance = paramInstance.eContainer();
    if ((compInstance instanceof ComponentInstance)) {
      final EObject paramModel = paramInstance.getParameter().eContainer();
      String fqn = "";
      if ((paramModel instanceof SystemParamModel)) {
        EList<ComponentParameterInstance> _components = ((SystemParamModel)paramModel).getComponents();
        for (final ComponentParameterInstance param : _components) {
          boolean _equals = param.getComponentInstance().equals(compInstance);
          if (_equals) {
            fqn = this.fqn_provider.getFullyQualifiedName(param).toString();
          }
        }
      }
      boolean _equals_1 = paramInstance.getParameter().getComponentInstance().equals(compInstance);
      boolean _not = (!_equals_1);
      if (_not) {
        this.error("Invalid ComponentParameterInstance.", 
          SystemParameterPackage.Literals.PARAMETER_STRUCT_INSTANCE__PARAMETER, 
          ComponentArchitectureValidator.INVALID_PARAMETER_STRUCT, fqn);
      }
    }
  }
}
