/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.deployment.validation;

import com.google.common.collect.Iterables;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.system.deployment.DeploymentModel;
import org.eclipse.smartmdsd.ecore.system.deployment.NamingService;
import org.eclipse.smartmdsd.ecore.system.deployment.NetworkInterfaceSelection;
import org.eclipse.smartmdsd.ecore.system.deployment.TargetPlatformReference;
import org.eclipse.smartmdsd.ecore.system.deployment.UploadDirectory;
import org.eclipse.smartmdsd.xtext.system.deployment.validation.AbstractDeploymentValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class DeploymentValidator extends AbstractDeploymentValidator {
  protected static final String DEPLOYMENT_ISSUE_PREFIX = "org.xtext.system.deployment.";
  
  public static final String MULTIPLE_NAMING_SERVICES = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MultipleNamingServices");
  
  public static final String MISSING_NETWORK_INTERFACE = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MissingNetworkInterface");
  
  public static final String MISSING_UPLOAD_DIRECTORY = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MissingUploadDirectory");
  
  @Check
  public void checkSingleNamingService(final NamingService ns) {
    final EObject parent = ns.eContainer();
    if ((parent instanceof DeploymentModel)) {
      int _size = IterableExtensions.size(Iterables.<NamingService>filter(((DeploymentModel)parent).getElements(), NamingService.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple NamingService-elements found, but (at most) one is allowed.", 
          null, DeploymentValidator.MULTIPLE_NAMING_SERVICES);
      }
    }
  }
  
  @Check
  public void checkMissingElements(final TargetPlatformReference ref) {
    NetworkInterfaceSelection _host = ref.getHost();
    boolean _tripleEquals = (_host == null);
    if (_tripleEquals) {
      this.warning("Missing NetworkInterfaceInstance.", 
        null, DeploymentValidator.MISSING_NETWORK_INTERFACE);
    }
    UploadDirectory _directory = ref.getDirectory();
    boolean _tripleEquals_1 = (_directory == null);
    if (_tripleEquals_1) {
      this.warning("Missing UploadDirectory.", 
        null, DeploymentValidator.MISSING_UPLOAD_DIRECTORY);
    }
  }
}
