/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.deployment.ui.quickfix;

import com.google.common.collect.Iterables;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.system.deployment.DeploymentFactory;
import org.eclipse.smartmdsd.ecore.system.deployment.DeploymentModel;
import org.eclipse.smartmdsd.ecore.system.deployment.NetworkInterfaceSelection;
import org.eclipse.smartmdsd.ecore.system.deployment.TargetPlatformReference;
import org.eclipse.smartmdsd.ecore.system.deployment.UploadDirectory;
import org.eclipse.smartmdsd.ecore.system.targetPlatform.NetworkInterface;
import org.eclipse.smartmdsd.xtext.system.deployment.validation.DeploymentValidator;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.model.edit.IModificationContext;
import org.eclipse.xtext.ui.editor.model.edit.ISemanticModification;
import org.eclipse.xtext.ui.editor.quickfix.DefaultQuickfixProvider;
import org.eclipse.xtext.ui.editor.quickfix.Fix;
import org.eclipse.xtext.ui.editor.quickfix.IssueResolutionAcceptor;
import org.eclipse.xtext.util.concurrent.IUnitOfWork;
import org.eclipse.xtext.validation.Issue;

/**
 * Custom quickfixes.
 * 
 * See https://www.eclipse.org/Xtext/documentation/310_eclipse_support.html#quick-fixes
 */
@SuppressWarnings("all")
public class DeploymentQuickfixProvider extends DefaultQuickfixProvider {
  @Fix(DeploymentValidator.MULTIPLE_NAMING_SERVICES)
  public void fixDuplicateOperationModeBinding(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      EObject _eContainer = element.eContainer();
      final DeploymentModel model = ((DeploymentModel) _eContainer);
      model.getElements().remove(element);
    };
    acceptor.accept(issue, 
      "Remove duplicate NamingService", 
      "Remove duplicate NamingService.", "", _function);
  }
  
  @Fix(DeploymentValidator.MISSING_UPLOAD_DIRECTORY)
  public void fixMissingUploadDirectoy(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      final TargetPlatformReference platform = ((TargetPlatformReference) element);
      platform.setDirectory(DeploymentFactory.eINSTANCE.createUploadDirectory());
      UploadDirectory _directory = platform.getDirectory();
      _directory.setPath("/tmp");
    };
    acceptor.accept(issue, 
      "Create a default UploadDirectory", 
      "Create a default UploadDirectory", "", _function);
  }
  
  @Fix(DeploymentValidator.MISSING_NETWORK_INTERFACE)
  public void fixMissingNetworkInterface(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final EObject object = this.getEObject(issue);
    if ((object instanceof TargetPlatformReference)) {
      Iterable<NetworkInterface> _filter = Iterables.<NetworkInterface>filter(((TargetPlatformReference)object).getPlatform().getElements(), NetworkInterface.class);
      for (final NetworkInterface network : _filter) {
        String _name = network.getName();
        String _plus = ("Use the NetworkInterface " + _name);
        String _name_1 = network.getName();
        String _plus_1 = ("Use the NetworkInterface " + _name_1);
        final ISemanticModification _function = (EObject element, IModificationContext context) -> {
          final TargetPlatformReference platform = ((TargetPlatformReference) element);
          platform.setHost(DeploymentFactory.eINSTANCE.createNetworkInterfaceSelection());
          NetworkInterfaceSelection _host = platform.getHost();
          _host.setNetwork(network);
        };
        acceptor.accept(issue, _plus, _plus_1, "", _function);
      }
    }
  }
  
  private EObject getEObject(final Issue issue) {
    final IModificationContext modificationContext = this.getModificationContextFactory().createModificationContext(issue);
    final IXtextDocument xtextDocument = modificationContext.getXtextDocument();
    final EObject object = xtextDocument.<EObject>readOnly(new IUnitOfWork<EObject, XtextResource>() {
      @Override
      public EObject exec(final XtextResource state) throws Exception {
        return state.getEObject(issue.getUriToProblem().fragment());
      }
    });
    return object;
  }
}
