/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.deployment.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentInstance;
import org.eclipse.smartmdsd.ecore.system.deployment.AbstractDeploymentElement;
import org.eclipse.smartmdsd.ecore.system.deployment.ComponentArtefact;
import org.eclipse.smartmdsd.ecore.system.deployment.DeploymentModel;
import org.eclipse.smartmdsd.ecore.system.deployment.DeploymentPackage;
import org.eclipse.smartmdsd.ecore.system.deployment.NamingService;
import org.eclipse.smartmdsd.ecore.system.deployment.NetworkInterfaceSelection;
import org.eclipse.smartmdsd.ecore.system.deployment.TargetPlatformReference;
import org.eclipse.smartmdsd.ecore.system.deployment.UploadDirectory;
import org.eclipse.smartmdsd.xtext.system.deployment.validation.AbstractDeploymentValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class DeploymentValidator extends AbstractDeploymentValidator {
  protected static final String DEPLOYMENT_ISSUE_PREFIX = "org.xtext.system.deployment.";
  
  public static final String MULTIPLE_NAMING_SERVICES = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MultipleNamingServices");
  
  public static final String MISSING_NETWORK_INTERFACE = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MissingNetworkInterface");
  
  public static final String MISSING_UPLOAD_DIRECTORY = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MissingUploadDirectory");
  
  public static final String NO_TARGET_PLATFORMS_DEFINED = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "NoTargetPlatformsDefined");
  
  public static final String MISSING_COMPONENT_ARTEFACT = (DeploymentValidator.DEPLOYMENT_ISSUE_PREFIX + "MissingComponentArtefact");
  
  @Check
  public void checkSingleNamingService(final NamingService ns) {
    final EObject parent = ns.eContainer();
    if ((parent instanceof DeploymentModel)) {
      int _size = IterableExtensions.size(Iterables.<NamingService>filter(((DeploymentModel)parent).getElements(), NamingService.class));
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.error("Multiple NamingService-elements found, but (at most) one is allowed.", 
          null, DeploymentValidator.MULTIPLE_NAMING_SERVICES);
      }
    }
  }
  
  @Check
  public void checkMissingElements(final TargetPlatformReference ref) {
    NetworkInterfaceSelection _host = ref.getHost();
    boolean _tripleEquals = (_host == null);
    if (_tripleEquals) {
      this.warning("Missing NetworkInterfaceInstance.", 
        null, DeploymentValidator.MISSING_NETWORK_INTERFACE);
    }
    UploadDirectory _directory = ref.getDirectory();
    boolean _tripleEquals_1 = (_directory == null);
    if (_tripleEquals_1) {
      this.warning("Missing UploadDirectory.", 
        null, DeploymentValidator.MISSING_UPLOAD_DIRECTORY);
    }
  }
  
  @Check
  public void hasTargetPlatforms(final DeploymentModel model) {
    final Function1<AbstractDeploymentElement, Boolean> _function = (AbstractDeploymentElement it) -> {
      return Boolean.valueOf((it instanceof TargetPlatformReference));
    };
    boolean _exists = IterableExtensions.<AbstractDeploymentElement>exists(model.getElements(), _function);
    boolean _not = (!_exists);
    if (_not) {
      this.warning("At least one TargetPlatformReference has to be defined.", DeploymentPackage.Literals.DEPLOYMENT_MODEL__NAME, DeploymentValidator.NO_TARGET_PLATFORMS_DEFINED);
    }
  }
  
  @Check
  public void allComponentArtefactsAvailable(final DeploymentModel model) {
    EList<ComponentInstance> _components = model.getComponentArch().getComponents();
    for (final ComponentInstance component : _components) {
      final Function1<ComponentArtefact, Boolean> _function = (ComponentArtefact it) -> {
        ComponentInstance _component = it.getComponent();
        return Boolean.valueOf(Objects.equal(_component, component));
      };
      boolean _exists = IterableExtensions.<ComponentArtefact>exists(Iterables.<ComponentArtefact>filter(model.getElements(), ComponentArtefact.class), _function);
      boolean _not = (!_exists);
      if (_not) {
        String _name = component.getName();
        String _plus = ("ComponentInstance " + _name);
        String _plus_1 = (_plus + " is not used in the deployment model");
        this.warning(_plus_1, 
          DeploymentPackage.Literals.DEPLOYMENT_MODEL__NAME, 
          DeploymentValidator.MISSING_COMPONENT_ARTEFACT, 
          component.getName());
      }
    }
  }
}
