/**
 * Copyright (c) 2018 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.service.serviceDefinition.validation;

import com.google.common.base.Objects;
import org.eclipse.emf.common.util.URI;
import org.eclipse.smartmdsd.ecore.service.coordinationPattern.ParameterPattern;
import org.eclipse.smartmdsd.ecore.service.coordinationPattern.StatePattern;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.AbstractServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.CoordinationServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ServiceDefRepository;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ServiceDefinitionPackage;
import org.eclipse.smartmdsd.xtext.service.serviceDefinition.validation.AbstractServiceDefinitionValidator;
import org.eclipse.xtext.validation.Check;

/**
 * This class contains custom validation rules.
 * 
 * See https://www.eclipse.org/Xtext/documentation/303_runtime_concepts.html#validation
 */
@SuppressWarnings("all")
public class ServiceDefinitionValidator extends AbstractServiceDefinitionValidator {
  protected static final String ISSUE_PREFIX = "org.xtext.service.serviceDefinition.";
  
  public static final String MISSING_PARAMETER_PATTERN = (ServiceDefinitionValidator.ISSUE_PREFIX + "MissingParameterPattern");
  
  public static final String MISSING_STATE_PATTERN = (ServiceDefinitionValidator.ISSUE_PREFIX + "MissingStatePattern");
  
  public static final String INVALID_REPO_NAME = (ServiceDefinitionValidator.ISSUE_PREFIX + "InvalidRepoName");
  
  public static final String CAPITAL_SERVICE_NAME = (ServiceDefinitionValidator.ISSUE_PREFIX + "CapitalServiceName");
  
  @Check
  public void checkMissingCoordinationPatterns(final CoordinationServiceDefinition service) {
    ParameterPattern _parameterPattern = service.getParameterPattern();
    boolean _tripleEquals = (_parameterPattern == null);
    if (_tripleEquals) {
      this.error("Missing ParameterPattern definition", 
        ServiceDefinitionPackage.Literals.ABSTRACT_SERVICE_DEFINITION__NAME, 
        ServiceDefinitionValidator.MISSING_PARAMETER_PATTERN);
    }
    StatePattern _statePattern = service.getStatePattern();
    boolean _tripleEquals_1 = (_statePattern == null);
    if (_tripleEquals_1) {
      this.error("Missing StatePattern definition", 
        ServiceDefinitionPackage.Literals.ABSTRACT_SERVICE_DEFINITION__NAME, 
        ServiceDefinitionValidator.MISSING_STATE_PATTERN);
    }
  }
  
  @Check
  public void checkRepositoryNameIsProjectName(final ServiceDefRepository repo) {
    final URI uri = repo.eResource().getURI();
    boolean _isPlatform = uri.isPlatform();
    if (_isPlatform) {
      final String projectName = uri.segment(1);
      String _name = repo.getName();
      boolean _notEquals = (!Objects.equal(_name, projectName));
      if (_notEquals) {
        this.warning("Repository name should match project name.", 
          ServiceDefinitionPackage.Literals.SERVICE_DEF_REPOSITORY__NAME, 
          ServiceDefinitionValidator.INVALID_REPO_NAME, projectName);
      }
    }
  }
  
  @Check
  public void checkServicedDefNameStartsWithCapital(final AbstractServiceDefinition service) {
    boolean _isUpperCase = Character.isUpperCase(service.getName().charAt(0));
    boolean _not = (!_isUpperCase);
    if (_not) {
      this.warning("ServiceDefinition name should start with a capital!", 
        ServiceDefinitionPackage.Literals.ABSTRACT_SERVICE_DEFINITION__NAME, 
        ServiceDefinitionValidator.CAPITAL_SERVICE_NAME);
    }
  }
}
