/**
 * Copyright (c) 2019 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtend.datasheet.generator.domainModels;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Injector;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.MandatoryDatasheetElement;
import org.eclipse.smartmdsd.ecore.base.genericDatasheet.MandatoryDatasheetElementNames;
import org.eclipse.smartmdsd.ecore.service.communicationObject.AbstractCommElement;
import org.eclipse.smartmdsd.ecore.service.communicationObject.CommObjectModel;
import org.eclipse.smartmdsd.ecore.service.communicationObject.CommObjectsRepository;
import org.eclipse.smartmdsd.ecore.service.communicationObject.CommunicationObject;
import org.eclipse.smartmdsd.ecore.service.communicationPattern.CommunicationPattern;
import org.eclipse.smartmdsd.ecore.service.communicationPattern.EventPattern;
import org.eclipse.smartmdsd.ecore.service.communicationPattern.PushPattern;
import org.eclipse.smartmdsd.ecore.service.communicationPattern.QueryPattern;
import org.eclipse.smartmdsd.ecore.service.communicationPattern.SendPattern;
import org.eclipse.smartmdsd.ecore.service.domainModelsDatasheet.DomainModelsDatasheet;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.AbstractServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.CommunicationServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ForkingServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.JoiningServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.RequestAnswerServiceDefinition;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ServiceDefModel;
import org.eclipse.smartmdsd.ecore.service.serviceDefinition.ServiceDefRepository;
import org.eclipse.smartmdsd.xtext.service.communicationObject.ui.internal.CommunicationObjectActivator;
import org.eclipse.smartmdsd.xtext.service.serviceDefinition.ui.internal.ServiceDefinitionActivator;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.documentation.IEObjectDocumentationProvider;
import org.eclipse.xtext.generator.AbstractGenerator;
import org.eclipse.xtext.generator.IFileSystemAccess2;
import org.eclipse.xtext.generator.IGeneratorContext;
import org.eclipse.xtext.resource.FileExtensionProvider;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

@SuppressWarnings("all")
public class DomainModelsDatasheetGeneratorImpl extends AbstractGenerator {
  @Inject
  private IEObjectDocumentationProvider doc;
  
  public String getRdfDatasheetFilename(final DomainModelsDatasheet ds) {
    String _name = ds.getName();
    return (_name + ".ttl");
  }
  
  @Override
  public void doGenerate(final Resource input, final IFileSystemAccess2 fsa, final IGeneratorContext context) {
    Iterable<DomainModelsDatasheet> _filter = Iterables.<DomainModelsDatasheet>filter(IteratorExtensions.<EObject>toIterable(input.getAllContents()), DomainModelsDatasheet.class);
    for (final DomainModelsDatasheet datasheet : _filter) {
      fsa.generateFile(this.getRdfDatasheetFilename(datasheet), this.compileRdfDatasheet(datasheet));
    }
  }
  
  public ServiceDefRepository getServiceDefRepository(final DomainModelsDatasheet ds) {
    final Injector injector = ServiceDefinitionActivator.getInstance().getInjector(ServiceDefinitionActivator.ORG_ECLIPSE_SMARTMDSD_XTEXT_SERVICE_SERVICEDEFINITION_SERVICEDEFINITION);
    final XtextResourceSet resourceSet = injector.<XtextResourceSet>getInstance(XtextResourceSet.class);
    resourceSet.addLoadOption(XtextResource.OPTION_RESOLVE_ALL, Boolean.TRUE);
    final URI baseUri = ds.eResource().getURI().trimFileExtension();
    final FileExtensionProvider fileExtProvider = injector.<FileExtensionProvider>getInstance(FileExtensionProvider.class);
    final URI xtextResourceUri = baseUri.appendFileExtension(fileExtProvider.getPrimaryFileExtension());
    final boolean resourceExists = resourceSet.getURIConverter().exists(xtextResourceUri, null);
    if ((resourceExists == true)) {
      final Resource resource = resourceSet.getResource(xtextResourceUri, true);
      boolean _isLoaded = resource.isLoaded();
      boolean _equals = (_isLoaded == true);
      if (_equals) {
        final EObject root_obj = resource.getContents().get(0);
        if ((root_obj instanceof ServiceDefModel)) {
          return ((ServiceDefModel)root_obj).getRepository();
        }
      }
    } else {
      InputOutput.<String>println(("non-existing resource: " + xtextResourceUri));
    }
    return null;
  }
  
  public CommObjectsRepository getCommObjectsRepository(final DomainModelsDatasheet ds) {
    final Injector injector = CommunicationObjectActivator.getInstance().getInjector(CommunicationObjectActivator.ORG_ECLIPSE_SMARTMDSD_XTEXT_SERVICE_COMMUNICATIONOBJECT_COMMUNICATIONOBJECT);
    final XtextResourceSet resourceSet = injector.<XtextResourceSet>getInstance(XtextResourceSet.class);
    resourceSet.addLoadOption(XtextResource.OPTION_RESOLVE_ALL, Boolean.TRUE);
    final URI baseUri = ds.eResource().getURI().trimFileExtension();
    final FileExtensionProvider fileExtProvider = injector.<FileExtensionProvider>getInstance(FileExtensionProvider.class);
    final URI xtextResourceUri = baseUri.appendFileExtension(fileExtProvider.getPrimaryFileExtension());
    final boolean resourceExists = resourceSet.getURIConverter().exists(xtextResourceUri, null);
    if ((resourceExists == true)) {
      final Resource resource = resourceSet.getResource(xtextResourceUri, true);
      boolean _isLoaded = resource.isLoaded();
      boolean _equals = (_isLoaded == true);
      if (_equals) {
        final EObject root_obj = resource.getContents().get(0);
        if ((root_obj instanceof CommObjectModel)) {
          return ((CommObjectModel)root_obj).getRepository();
        }
      }
    } else {
      InputOutput.<String>println(("non-existing resource: " + xtextResourceUri));
    }
    return null;
  }
  
  public String getBaseURI(final DomainModelsDatasheet ds) {
    final Function1<MandatoryDatasheetElement, Boolean> _function = (MandatoryDatasheetElement it) -> {
      return Boolean.valueOf(it.getName().equals(MandatoryDatasheetElementNames.BASE_URI));
    };
    final MandatoryDatasheetElement baseuri = IterableExtensions.<MandatoryDatasheetElement>findFirst(Iterables.<MandatoryDatasheetElement>filter(ds.getElements(), MandatoryDatasheetElement.class), _function);
    if ((baseuri != null)) {
      return baseuri.getValue();
    }
    return "";
  }
  
  public String getBShortDescription(final DomainModelsDatasheet ds) {
    final Function1<MandatoryDatasheetElement, Boolean> _function = (MandatoryDatasheetElement it) -> {
      return Boolean.valueOf(it.getName().equals(MandatoryDatasheetElementNames.SHORT_DESCRIPTION));
    };
    final MandatoryDatasheetElement property = IterableExtensions.<MandatoryDatasheetElement>findFirst(Iterables.<MandatoryDatasheetElement>filter(ds.getElements(), MandatoryDatasheetElement.class), _function);
    if ((property != null)) {
      return property.getValue();
    }
    return "";
  }
  
  public CharSequence compileCommObj(final CommunicationObject commObj, final DomainModelsDatasheet ds) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(":");
    String _name = commObj.getName();
    _builder.append(_name);
    _builder.append(" a robmosys:CommunicationObject;");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("rdfs:label \"");
    String _name_1 = commObj.getName();
    _builder.append(_name_1, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("t1:shortDescription \"CommunicationObject ");
    String _name_2 = commObj.getName();
    _builder.append(_name_2, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    {
      String _documentation = this.doc.getDocumentation(commObj);
      boolean _tripleNotEquals = (_documentation != null);
      if (_tripleNotEquals) {
        _builder.append("\t");
        _builder.append("t1:description \"\"\"");
        String _documentation_1 = this.doc.getDocumentation(commObj);
        _builder.append(_documentation_1, "\t");
        _builder.append("\"\"\";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("t1:memberOfRepository <");
    String _baseURI = this.getBaseURI(ds);
    _builder.append(_baseURI, "\t");
    _builder.append("/DomainModels/");
    String _name_3 = ds.getName();
    _builder.append(_name_3, "\t");
    _builder.append(">;");
    _builder.newLineIfNotEmpty();
    _builder.append(".");
    _builder.newLine();
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence compileServiceDefinition(final CommunicationServiceDefinition service, final DomainModelsDatasheet ds) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(":");
    String _name = service.getName();
    _builder.append(_name);
    _builder.append(" a robmosys:ServiceDefinition;");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("rdf:type rdfs:Class;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rdf:type sh:NodeShape;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rdfs:subClassOf robmosys:AbstractService;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rdfs:label \"");
    String _name_1 = service.getName();
    _builder.append(_name_1, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("t1:shortDescription \"ServiceDefinition ");
    String _name_2 = service.getName();
    _builder.append(_name_2, "\t");
    _builder.append("\";");
    _builder.newLineIfNotEmpty();
    {
      String _documentation = this.doc.getDocumentation(service);
      boolean _tripleNotEquals = (_documentation != null);
      if (_tripleNotEquals) {
        _builder.append("\t");
        _builder.append("t1:description \"\"\"");
        String _documentation_1 = this.doc.getDocumentation(service);
        _builder.append(_documentation_1, "\t");
        _builder.append("\"\"\";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("t1:memberOfRepository <");
    String _baseURI = this.getBaseURI(ds);
    _builder.append(_baseURI, "\t");
    _builder.append("/DomainModels/");
    String _name_3 = ds.getName();
    _builder.append(_name_3, "\t");
    _builder.append(">;");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("t1:commPatternUsage [");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("a robmosys:");
    String _name_4 = this.getPattern(service).eClass().getName();
    _builder.append(_name_4, "\t\t");
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("# FIXME: check the namespace: it will be wrong when elements from other repos are used.");
    _builder.newLine();
    _builder.append("\t\t");
    CharSequence _objects = this.getObjects(this.getPattern(service));
    _builder.append(_objects, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("];");
    _builder.newLine();
    _builder.append(".");
    _builder.newLine();
    _builder.newLine();
    return _builder;
  }
  
  public ServiceDefRepository getRepo(final CommunicationServiceDefinition svc) {
    EObject _eContainer = svc.eContainer();
    return ((ServiceDefRepository) _eContainer);
  }
  
  public CommunicationPattern getPattern(final CommunicationServiceDefinition svc) {
    CommunicationPattern _switchResult = null;
    boolean _matched = false;
    if (svc instanceof ForkingServiceDefinition) {
      _matched=true;
      _switchResult = ((ForkingServiceDefinition)svc).getPattern();
    }
    if (!_matched) {
      if (svc instanceof JoiningServiceDefinition) {
        _matched=true;
        _switchResult = ((JoiningServiceDefinition)svc).getPattern();
      }
    }
    if (!_matched) {
      if (svc instanceof RequestAnswerServiceDefinition) {
        _matched=true;
        _switchResult = ((RequestAnswerServiceDefinition)svc).getPattern();
      }
    }
    return _switchResult;
  }
  
  public CharSequence getObjects(final CommunicationPattern pattern) {
    CharSequence _switchResult = null;
    boolean _matched = false;
    if (pattern instanceof PushPattern) {
      _matched=true;
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("t1:dataType :");
      String _name = ((PushPattern)pattern).getDataType().getName();
      _builder.append(_name);
      _builder.append(";");
      _switchResult = _builder;
    }
    if (!_matched) {
      if (pattern instanceof EventPattern) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("t1:eventType :");
        String _name = ((EventPattern)pattern).getEventType().getName();
        _builder.append(_name);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("t1:activationType :");
        String _name_1 = ((EventPattern)pattern).getActivationType().getName();
        _builder.append(_name_1);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("t1:eventStateType :");
        String _name_2 = ((EventPattern)pattern).getEventStateType().getName();
        _builder.append(_name_2);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _switchResult = _builder;
      }
    }
    if (!_matched) {
      if (pattern instanceof SendPattern) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("t1:dataType :");
        String _name = ((SendPattern)pattern).getDataType().getName();
        _builder.append(_name);
        _builder.append(";");
        _switchResult = _builder;
      }
    }
    if (!_matched) {
      if (pattern instanceof QueryPattern) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("t1:requestType :");
        String _name = ((QueryPattern)pattern).getRequestType().getName();
        _builder.append(_name);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("t1:answerType :");
        String _name_1 = ((QueryPattern)pattern).getAnswerType().getName();
        _builder.append(_name_1);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _switchResult = _builder;
      }
    }
    return _switchResult;
  }
  
  public CharSequence compileRdfDatasheet(final DomainModelsDatasheet ds) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("# baseURI: ");
    String _baseURI = this.getBaseURI(ds);
    _builder.append(_baseURI);
    _builder.append("/DomainModels/");
    String _name = ds.getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    _builder.append("# imports: http://robmosys.eu/rdf/composition-structures");
    _builder.newLine();
    _builder.append("# imports: http://seronet-projekt.de/models/t2#Navigation");
    _builder.newLine();
    _builder.newLine();
    _builder.append("#######################################");
    _builder.newLine();
    _builder.append("# Technical Data Sheet");
    _builder.newLine();
    _builder.append("# for a SeRoNet DomainModels Repository as ");
    _builder.newLine();
    _builder.append("# exported by the SmartMDSD Toolchain");
    _builder.newLine();
    _builder.append("#");
    _builder.newLine();
    _builder.append("# <domainModelsName>/datasheet-gen/<domainModelsName>.ttl");
    _builder.newLine();
    _builder.append("#######################################");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# all generated files use these prefixes");
    _builder.newLine();
    _builder.append("@prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .");
    _builder.newLine();
    _builder.append("@prefix rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .");
    _builder.newLine();
    _builder.append("@prefix t1: <http://seronet-projekt.de/models/t1#> .");
    _builder.newLine();
    _builder.append("@prefix spdx: <http://spdx.org/rdf/terms#> .");
    _builder.newLine();
    _builder.append("@prefix spdx-lic: <http://spdx.org/licenses/> .");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@prefix owl: <http://www.w3.org/2002/07/owl#> .");
    _builder.newLine();
    _builder.append("@prefix sh: <http://www.w3.org/ns/shacl#> .");
    _builder.newLine();
    _builder.append("@prefix xsd: <http://www.w3.org/2001/XMLSchema#> .");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# TODO past “Durchstich”: exactly identify which elements belong to SeRoNet and which belong to RobMoSys");
    _builder.newLine();
    _builder.append("@prefix robmosys: <http://robmosys.eu/rdf/composition-structures#> .");
    _builder.newLine();
    _builder.newLine();
    _builder.append("# these namespaces can be different in generated files");
    _builder.newLine();
    _builder.append("@prefix : <");
    String _baseURI_1 = this.getBaseURI(ds);
    _builder.append(_baseURI_1);
    _builder.append("/DomainModels/");
    String _name_1 = ds.getName();
    _builder.append(_name_1);
    _builder.append("#> .");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("##### Domain Model Repository");
    _builder.newLine();
    _builder.append("# ... they resemble eclipse projects in the tooling");
    _builder.newLine();
    _builder.append("# Note: this will later not be provided by the software component");
    _builder.newLine();
    _builder.append("<");
    String _baseURI_2 = this.getBaseURI(ds);
    _builder.append(_baseURI_2);
    _builder.append("/DomainModels/");
    String _name_2 = ds.getName();
    _builder.append(_name_2);
    _builder.append(">");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("a robmosys:DomainModelRepository;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("rdf:type owl:Ontology ;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("owl:imports <http://robmosys.eu/rdf/composition-structures> ;");
    _builder.newLine();
    {
      String _documentation = this.doc.getDocumentation(ds);
      boolean _tripleNotEquals = (_documentation != null);
      if (_tripleNotEquals) {
        _builder.append("\t");
        _builder.append("t1:description \"");
        String _documentation_1 = this.doc.getDocumentation(ds);
        _builder.append(_documentation_1, "\t");
        _builder.append("\";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append(".");
    _builder.newLine();
    _builder.newLine();
    _builder.append("##### Communication Objects");
    _builder.newLine();
    _builder.append("# see https://robmosys.eu/wiki/modeling:metamodels:commobject ");
    _builder.newLine();
    _builder.append("# Note: this will later not be provided by the software component");
    _builder.newLine();
    {
      CommObjectsRepository _commObjectsRepository = this.getCommObjectsRepository(ds);
      EList<AbstractCommElement> _elements = null;
      if (_commObjectsRepository!=null) {
        _elements=_commObjectsRepository.getElements();
      }
      final Function1<CommunicationObject, String> _function = (CommunicationObject it) -> {
        return it.getName();
      };
      List<CommunicationObject> _sortBy = IterableExtensions.<CommunicationObject, String>sortBy(Iterables.<CommunicationObject>filter(_elements, CommunicationObject.class), _function);
      for(final CommunicationObject commObj : _sortBy) {
        CharSequence _compileCommObj = this.compileCommObj(commObj, ds);
        _builder.append(_compileCommObj);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("##### Service Definitions");
    _builder.newLine();
    _builder.append("# see https://robmosys.eu/wiki/modeling:metamodels:service ");
    _builder.newLine();
    _builder.append("# see https://robmosys.eu/wiki/modeling:metamodels:commpattern ");
    _builder.newLine();
    _builder.append("# Note: this will later not be provided by the software component");
    _builder.newLine();
    {
      ServiceDefRepository _serviceDefRepository = this.getServiceDefRepository(ds);
      EList<AbstractServiceDefinition> _services = null;
      if (_serviceDefRepository!=null) {
        _services=_serviceDefRepository.getServices();
      }
      final Function1<CommunicationServiceDefinition, String> _function_1 = (CommunicationServiceDefinition it) -> {
        return it.getName();
      };
      List<CommunicationServiceDefinition> _sortBy_1 = IterableExtensions.<CommunicationServiceDefinition, String>sortBy(Iterables.<CommunicationServiceDefinition>filter(_services, CommunicationServiceDefinition.class), _function_1);
      for(final CommunicationServiceDefinition service : _sortBy_1) {
        CharSequence _compileServiceDefinition = this.compileServiceDefinition(service, ds);
        _builder.append(_compileServiceDefinition);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
}
