/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.componentArchitecture.ui.quickfix;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentPort;
import org.eclipse.smartmdsd.ecore.component.seronetExtension.SupportedMiddleware;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentArchitectureFactory;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentArchitectureModelUtility;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentInstance;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.Connection;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.RequiredService;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ServiceInstance;
import org.eclipse.smartmdsd.xtext.service.roboticMiddleware.ui.quickfix.RoboticMiddlewareQuickfixProvider;
import org.eclipse.smartmdsd.xtext.system.componentArchitecture.validation.ComponentArchitectureValidator;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.model.edit.IModification;
import org.eclipse.xtext.ui.editor.model.edit.IModificationContext;
import org.eclipse.xtext.ui.editor.model.edit.ISemanticModification;
import org.eclipse.xtext.ui.editor.quickfix.Fix;
import org.eclipse.xtext.ui.editor.quickfix.IssueResolutionAcceptor;
import org.eclipse.xtext.ui.resource.IResourceSetProvider;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Custom quickfixes.
 * 
 * See https://www.eclipse.org/Xtext/documentation/310_eclipse_support.html#quick-fixes
 */
@SuppressWarnings("all")
public class ComponentArchitectureQuickfixProvider extends RoboticMiddlewareQuickfixProvider {
  @Inject
  private IResourceSetProvider resourceSetProvider;
  
  @Fix(ComponentArchitectureValidator.INVALID_COMP_NAME)
  public void capitalizeName(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final IModification _function = (IModificationContext context) -> {
      final IXtextDocument xtextDocument = context.getXtextDocument();
      final String firstLetter = xtextDocument.get((issue.getOffset()).intValue(), 1);
      xtextDocument.replace((issue.getOffset()).intValue(), 1, firstLetter.toUpperCase());
    };
    acceptor.accept(issue, "Capitalize name", "Capitalize the name.", "upcase.png", _function);
  }
  
  @Fix(ComponentArchitectureValidator.MISSING_PORTS)
  public void addAllMissingPorts(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      final ComponentInstance instance = ((ComponentInstance) element);
      final Iterable<ComponentPort> services = ComponentArchitectureModelUtility.getAllClientPorts(instance);
      for (final ComponentPort svc : services) {
        final Function1<ServiceInstance, Boolean> _function_1 = (ServiceInstance it) -> {
          ComponentPort _port = it.getPort();
          return Boolean.valueOf(Objects.equal(_port, svc));
        };
        boolean _exists = IterableExtensions.<ServiceInstance>exists(instance.getPorts(), _function_1);
        boolean _not = (!_exists);
        if (_not) {
          final RequiredService svc_use = ComponentArchitectureFactory.eINSTANCE.createRequiredService();
          svc_use.setPort(svc);
          instance.getPorts().add(svc_use);
        }
      }
    };
    acceptor.accept(issue, 
      "Add all missing ports", 
      "Add all missing ports", "", _function);
  }
  
  @Fix(ComponentArchitectureValidator.MISSING_PORTS)
  public void addNonOptionalMissingPorts(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      final ComponentInstance instance = ((ComponentInstance) element);
      final Iterable<ComponentPort> services = ComponentArchitectureModelUtility.getNonOptionalClientPorts(instance);
      for (final ComponentPort svc : services) {
        final Function1<ServiceInstance, Boolean> _function_1 = (ServiceInstance it) -> {
          ComponentPort _port = it.getPort();
          return Boolean.valueOf(Objects.equal(_port, svc));
        };
        boolean _exists = IterableExtensions.<ServiceInstance>exists(instance.getPorts(), _function_1);
        boolean _not = (!_exists);
        if (_not) {
          final RequiredService svc_use = ComponentArchitectureFactory.eINSTANCE.createRequiredService();
          svc_use.setPort(svc);
          instance.getPorts().add(svc_use);
        }
      }
    };
    acceptor.accept(issue, 
      "Add non-optional ports only", 
      "Add non-optional ports only", "", _function);
  }
  
  @Fix(ComponentArchitectureValidator.MIDDLWARE_SELECTION_NOT_SUPPORTED)
  public void addChangeMiddlewareSelection(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ResourceSet rsp = this.resourceSetProvider.get(null);
    final EObject object = rsp.getEObject(issue.getUriToProblem(), true);
    if ((object instanceof Connection)) {
      Iterable<SupportedMiddleware> _filter = Iterables.<SupportedMiddleware>filter(((Connection)object).getFrom().getPort().getExtensions(), SupportedMiddleware.class);
      for (final SupportedMiddleware mw : _filter) {
        final Function1<SupportedMiddleware, Boolean> _function = (SupportedMiddleware it) -> {
          String _name = it.getMiddleware().eClass().getName();
          String _name_1 = mw.getMiddleware().eClass().getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        };
        boolean _exists = IterableExtensions.<SupportedMiddleware>exists(Iterables.<SupportedMiddleware>filter(((Connection)object).getTo().getPort().getExtensions(), SupportedMiddleware.class), _function);
        if (_exists) {
          String _name = mw.getMiddleware().eClass().getName();
          final String text = ("Change to " + _name);
          final ISemanticModification _function_1 = (EObject element, IModificationContext context) -> {
            final Connection connection = ((Connection) element);
            connection.setMiddlewareSelection(mw.getMiddleware());
          };
          acceptor.accept(issue, text, text, "", _function_1);
        }
      }
    }
  }
  
  @Fix(ComponentArchitectureValidator.DUPLICATE_PARAMETER_STRUCT)
  public void fixDuplicateParameterStruct(final Issue issue, final IssueResolutionAcceptor acceptor) {
    final ISemanticModification _function = (EObject element, IModificationContext context) -> {
      EObject _eContainer = element.eContainer();
      final ComponentInstance component = ((ComponentInstance) _eContainer);
      component.getExtensions().remove(element);
    };
    acceptor.accept(issue, 
      "Remove duplicate ParameterStruct", 
      "Remove duplicate ParameterStruct.", "", _function);
  }
  
  @Fix(ComponentArchitectureValidator.INVALID_PARAMETER_STRUCT)
  public void fixInvalidParameterStruct(final Issue issue, final IssueResolutionAcceptor acceptor) {
    String _get = issue.getData()[0];
    String _plus = ("Change to " + _get);
    String _get_1 = issue.getData()[0];
    String _plus_1 = ("Change to " + _get_1);
    final IModification _function = (IModificationContext context) -> {
      final IXtextDocument xtextDocument = context.getXtextDocument();
      xtextDocument.replace((issue.getOffset()).intValue(), (issue.getLength()).intValue(), issue.getData()[0]);
    };
    acceptor.accept(issue, _plus, _plus_1, "", _function);
  }
}
