/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.Bucket;
import org.eclipse.smila.jobmanager.BucketDefinition;
import org.eclipse.smila.jobmanager.IllegalJobStateException;
import org.eclipse.smila.jobmanager.JobDefinition;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.JobManagerException;
import org.eclipse.smila.jobmanager.JobRunDefinitions;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.JobRunMode;
import org.eclipse.smila.jobmanager.JobState;
import org.eclipse.smila.jobmanager.WorkerDefinition;
import org.eclipse.smila.jobmanager.WorkerDefinition.Mode;
import org.eclipse.smila.jobmanager.WorkerDefinition.OutputMode;
import org.eclipse.smila.jobmanager.WorkflowDefinition;
import org.eclipse.smila.jobmanager.events.JobListener;
import org.eclipse.smila.jobmanager.events.PrepareToFinishEvent;
import org.eclipse.smila.jobmanager.persistence.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.jobmanager.persistence.DefinitionStorage;
import org.eclipse.smila.jobmanager.persistence.JobRunListener;
import org.eclipse.smila.jobmanager.persistence.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.persistence.RunStorageException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.jobmanager.util.IdGenerator;
import org.eclipse.smila.objectstore.NoSuchStoreException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.utils.collections.MultiValueMap;
import org.osgi.service.component.ComponentContext;

/**
 * Standard implementation of JobManager service.
 */
public class JobManagerImpl implements JobManager, JobRunListener {

  /** simple date format used for creating readable job id. */
  private static final int JOB_RUN_ID_SUFFIX_MAX = 1000;

  /** random element to make job ids unique. */
  private static final Random RANDOM = new Random(System.nanoTime());

  /** private log. */
  private final Log _log = LogFactory.getLog(getClass());

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /** service reference to definition storage. */
  private DefinitionStorage _defStorage;

  /** service reference to definition storage. */
  private RunStorage _runStorage;

  /** service reference to object store. */
  private ObjectStoreService _objectStore;

  /** service reference to task generator provider used to select a task generator. */
  private TaskGeneratorProvider _taskGeneratorProvider;

  /** service reference to task generator provider used to select a task generator. */
  private TaskManager _taskManager;

  /** service reference to cluster config service to read job property. */
  private ClusterConfigService _clusterConfigService;

  /** the map with the job runs. job name as key, job run as value */
  private final Map<String, JobRun> _jobRuns = new HashMap<String, JobRun>();

  // TODO ConcurrentHashMap ?

  /** Listener for job events. */
  private final CopyOnWriteArrayList<JobListener> _jobListeners = new CopyOnWriteArrayList<JobListener>();

  /**
   * initialize JobManager internally on service start.
   */
  protected void activate(final ComponentContext context) {
    try {
      _defPersistence = new DefinitionPersistence(_defStorage, this, _objectStore);
      final Collection<String> workers = _defPersistence.getWorkers();
      for (final String worker : workers) {
        _taskManager.addTaskQueue(worker);
      }
      if (_log.isDebugEnabled()) {
        _log.debug("JobManager activate() successful");
      }
    } catch (final Throwable e) {
      final String msg = "Error while activating '" + JobManagerConstants.CONFIGURATION_BUNDLE + "'.";
      _log.error(msg, e);
      throw new RuntimeException(msg, e);
    }
  }

  /**
   * clean up JobManager on service shutdown.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * @return the defPersistence
   */
  protected DefinitionPersistence getDefPersistence() {
    return _defPersistence;
  }

  /**
   * @return the defStorage
   */
  protected DefinitionStorage getDefStorage() {
    return _defStorage;
  }

  /**
   * @return the runStorage
   */
  protected RunStorage getRunStorage() {
    return _runStorage;
  }

  /**
   * @return the objectStore
   */
  protected ObjectStoreService getObjectStore() {
    return _objectStore;
  }

  /**
   * @return the taskGeneratorProvider
   */
  protected TaskGeneratorProvider getTaskGeneratorProvider() {
    return _taskGeneratorProvider;
  }

  /**
   * @return the taskManager
   */
  protected TaskManager getTaskManager() {
    return _taskManager;
  }

  /**
   * @return the JobRuns
   */
  protected Map<String, JobRun> getJobRuns() {
    return _jobRuns;
  }

  /**
   * @return the jobListeners
   */
  protected CopyOnWriteArrayList<JobListener> getJobListeners() {
    return _jobListeners;
  }

  // ----- Jobs ------
  /**
   * Returns a job from the persistence definition.
   * 
   * @param name
   *          the name of the job
   * @return the job with the given name
   * @throws JobManagerException
   *           the job does not exist or an error occured while accessing job definitions.
   */
  private JobDefinition getJob(final String name) throws JobManagerException {
    final JobDefinition job = _defPersistence.getJob(name);
    if (job == null) {
      throw new ConfigNotFoundException("Job '" + name + "' not found.");
    }
    return job;
  }

  /**
   * Returns a workflow from the persistence definition.
   * 
   * @param name
   *          the name of the workflow
   * @return the workflow with the given name
   * @throws JobManagerException
   *           the workflow does not exist or an error occurred while accessing workflow definitions.
   */
  private WorkflowDefinition getWorkflow(final String name) throws JobManagerException {
    final WorkflowDefinition workflow = _defPersistence.getWorkflow(name);
    if (workflow == null) {
      throw new ConfigNotFoundException("Workflow '" + name + "' not found.");
    }
    return workflow;
  }

  /**
   * Returns a worker from the persistence definition.
   * 
   * @param name
   *          the name of the worker
   * @return the worker with the given name
   * @throws ConfigNotFoundException
   *           the worker does not exist.
   */
  private WorkerDefinition getWorker(final String name) throws ConfigNotFoundException {
    final WorkerDefinition worker = _defPersistence.getWorker(name);
    if (worker == null) {
      throw new ConfigNotFoundException("Worker '" + name + "' not found.");
    }
    return worker;
  }

  /** {@inheritDoc} */
  @Override
  public String startJob(final String jobName) throws JobManagerException {
    return startJob(jobName, JobRunMode.STANDARD);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String startJob(final String jobName, final JobRunMode jobRunMode) throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("start called for job '" + jobName + "', jobRunMode '" + jobRunMode + "'");
    }
    String jobId = null;
    JobState currentState = null;
    try {
      if (!_defPersistence.hasJob(jobName)) {
        throw new IllegalArgumentException("No definition for job '" + jobName + "' exists.");
      }
      // check if another job run of this job is already (or still) active.
      currentState = _runStorage.getJobState(jobName);
      if (currentState != null) {
        throw new IllegalStateException("Job '" + jobName + "' is in state '" + currentState + "'.");
      }
      final JobRunDefinitions jobRunDefs = getJobRunDefinitions(jobName);
      jobId = createJobId();
      _runStorage.startJobRun(jobName, jobId, jobRunMode, jobRunDefs);
      _runStorage.registerJobRunListener(this, jobName); // to avoid memory leaks in cluster
      // create job run and store it in map
      final JobRun jobRun = ensureJobRun(jobName, jobId);
      checkBuckets(jobRun);
      addJobTriggers(jobRun);
      if (_runStorage.setJobState(jobName, jobId, JobState.PREPARING, JobState.RUNNING)) {
        if (JobRunMode.RUNONCE == jobRunMode) {
          executeRunOnceJob(jobRun);
        }
        if (_log.isInfoEnabled()) {
          _log.info("started job run '" + jobId + "' for job '" + jobName + "'");
        }
        return jobId;
      } else {
        throw new JobManagerException("Error while starting job '" + jobName + "' with jobId '" + jobId + "'. ");
      }
    } catch (final Exception e) {
      final String messagePrefix = "Could not start job '" + jobName + "': ";
      if (jobId != null) {
        currentState = _runStorage.getJobState(jobName);
        if (currentState == null) {
          // job run could have been removed in between, e.g. by a cancel, so have a look in persistent storage
          final AnyMap jobData = getJobRunData(jobName, jobId, false);
          if (jobData != null && jobData.getStringValue(JobManagerConstants.DATA_JOB_STATE) != null) {
            currentState = JobState.valueOf(jobData.getStringValue(JobManagerConstants.DATA_JOB_STATE));
          }
        }
      }
      if (currentState == JobState.CANCELED || currentState == JobState.CANCELLING) {
        throw new JobManagerException(messagePrefix + "Job run was canceled while starting.", e);
      }
      cleanupFailedJobRun(jobName, jobId);
      throw new JobManagerException(messagePrefix + e.getMessage(), e);
    }
  }

  /**
   * @return the currently persisted job/workflow/bucket definitions for the given job.
   */
  private JobRunDefinitions getJobRunDefinitions(final String jobName) throws JobManagerException {
    final JobRunDefinitions runDefs = new JobRunDefinitions();
    final JobDefinition jobDef = getJob(jobName);
    runDefs.setJobDefinition(jobDef);
    final WorkflowDefinition workflowDef = getWorkflow(jobDef.getWorkflow());
    runDefs.setWorkflowDefinition(workflowDef);
    final Collection<String> workflowBuckets = workflowDef.getReferencedBuckets();
    final Collection<String> persistentBuckets = _defPersistence.getBuckets();
    persistentBuckets.retainAll(workflowBuckets); // intersection
    for (final String bucket : persistentBuckets) {
      final BucketDefinition bucketDef = _defPersistence.getBucket(bucket);
      runDefs.addBucketDefinition(bucketDef);
    }
    return runDefs;
  }

  /**
   * prepares and automatically finishes a run once job.
   */
  private void executeRunOnceJob(final JobRun jobRun) throws JobManagerException, TaskmanagerException {
    // prepare run
    final String workflowRunId = _runStorage.startWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId());

    // in run once mode, tasks are initially read from the start action's input bucket
    final List<Task> followUpTasks = jobRun.getInitialRunOnceTasks(workflowRunId);
    if (!followUpTasks.isEmpty()) {
      _runStorage.startTasks(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId, followUpTasks);
    }
    // in run once mode, jobs are automatically finished
    finishJob(jobRun.getJobName(), jobRun.getJobRunId());
    // if no tasks are generated we also have to finish the workflow run and complete the job run
    if (followUpTasks.isEmpty()) {
      _runStorage.finishWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
      _runStorage.deleteWorkflowRun(jobRun.getJobName(), jobRun.getJobRunId(), workflowRunId);
      checkAndHandleJobRunCompleted(jobRun.getJobName(), jobRun.getJobRunId());
    }
    // last step, to avoid that we have to roll it back if an error occurs in the Jobmanager before.
    _taskManager.addTasks(followUpTasks);
  }

  /**
   * register job to be triggered by its input buckets.
   * 
   * @param jobRun
   *          job run
   * @throws RunStorageException
   *           error
   */
  private void addJobTriggers(final JobRun jobRun) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    for (final Bucket triggerBucket : jobRun.getTriggerBuckets()) {
      _runStorage.addJobTrigger(triggerBucket.getBucketId(), jobName);
    }
  }

  /**
   * Clean up a failed job run. Set state to FAILED and persist the run data.
   * 
   * @param jobName
   *          job name
   * @param jobRunId
   *          job run id
   */
  private void cleanupFailedJobRun(final String jobName, final String jobRunId) {
    if (jobRunId != null) {
      try {
        if (_jobRuns.containsKey(jobName)) {
          // we have to make sure that no other job run was started in parallel
          final String currentJobRunId = _runStorage.getJobRunId(jobName);
          if (jobRunId.equals(currentJobRunId)) {
            removeJobTriggers(_jobRuns.get(jobName));
          }
        }
        completeJobRun(jobName, jobRunId, JobState.FAILED);
      } catch (final Exception ex) {
        _log.warn(
          "Error while cleaning up failed job '" + jobName + "' with run '" + jobRunId + "' after prepare.", ex);
      }
    }
  }

  /**
   * Checks if all persistent buckets of the given job refer to an existing store. Prepares the store for all transient
   * buckets of thie given job.
   * 
   * @param jobRun
   *          the job run data
   * @throws ObjectStoreException
   *           error checking or creating store in {@link ObjectStoreService}
   * 
   */
  private void checkBuckets(final JobRun jobRun) throws ObjectStoreException {
    for (final Bucket bucket : jobRun.getBuckets()) {
      if (bucket.isTransient()) {
        ObjectStoreRetryUtil.retryEnsureStore(_objectStore, bucket.getStoreName());
      } else if (!ObjectStoreRetryUtil.retryExistsStore(_objectStore, bucket.getStoreName())) {
        throw new NoSuchStoreException("Store '" + bucket.getStoreName() + "' of persistent bucket '"
          + bucket.getBucketId() + "' does not exist.");
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getJobRunData(final String jobName, final String jobId) throws JobManagerException {
    return getJobRunData(jobName, jobId, false);
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getJobRunData(final String jobName, final String jobId, final boolean returnDetails)
    throws JobManagerException {
    try {
      final String jobRunId = _runStorage.getJobRunId(jobName);
      if (jobId.equals(jobRunId)) {
        return _runStorage.getJobRunData(jobName, returnDetails);
      }
    } catch (final RunStorageException ex) {
      _log.warn("Error determining live job run data for job '" + jobName + "'. Trying to read from history.", ex);
    }
    if (_defStorage.containsJobRun(jobName, jobId)) {
      final AnyMap jobRunData = _defStorage.getJobRunData(jobName, jobId);
      if (!returnDetails && jobRunData != null) {
        jobRunData.remove(JobManagerConstants.DATA_JOB_RUN_WORKFLOW_DEF);
        jobRunData.remove(JobManagerConstants.DATA_JOB_RUN_BUCKET_DEFS);
      }
      return jobRunData;
    }
    throw new ConfigNotFoundException("No job run data available for run '" + jobId + "' of job '" + jobName + "'.");
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap getWorkflowRunData(final String jobName, final String jobId, final String workflowRunId)
    throws JobManagerException {
    final String jobRunId = _runStorage.getJobRunId(jobName);
    if (jobId.equals(jobRunId)) {
      return _runStorage.getWorkflowRunData(jobName, workflowRunId);
    }
    throw new ConfigNotFoundException("No workflow run data available for run '" + jobId + "' of job '" + jobName
      + "'.");
  }

  /** {@inheritDoc} */
  @Override
  public JobRunInfo getJobRunInfo(final String jobName) throws JobManagerException {
    return _runStorage.getJobRunInfo(jobName);
  }

  /** {@inheritDoc} */
  @Override
  public Collection<String> getCompletedJobRunIds(final String jobName) throws JobManagerException {
    final Collection<String> completedJobRuns = _defStorage.getJobRunIds(jobName);
    final String currentjobRunRunId = _runStorage.getJobRunId(jobName);
    if (currentjobRunRunId != null) {
      // there is a small timespan where job run is written to persistent storage but not yet deleted from RunStorage
      completedJobRuns.remove(currentjobRunRunId);
    }
    return completedJobRuns;
  }

  /** {@inheritDoc} */
  @Override
  public void deleteJobRunData(final String jobName, final String jobId) throws JobManagerException {
    final String currentJobRunId = _runStorage.getJobRunId(jobName);
    if (jobId.equals(currentJobRunId)) {
      throw new JobManagerException("Job run data of active job run cannot be deleted.");
    }
    _defStorage.deleteJobRunData(jobName, jobId);
  }

  /**
   * 
   * Checks if a task may be generated for the job run with the given jobRunId.
   * 
   * @param jobName
   *          The name of the job to check.
   * @param jobRunId
   *          The job id to check.
   * @param isInitialTask
   *          Is this task an initial task (true) or follow-up task(false).
   * @return true if tasks may be created, false if not.
   * @throws RunStorageException
   *           exception while accessing JobRun data.
   */
  private boolean checkJobStateForTaskCreation(final String jobName, final String jobRunId,
    final boolean isInitialTask) throws RunStorageException {
    final JobState jobState = _runStorage.getJobState(jobName, jobRunId);
    if (jobState == JobState.RUNNING) {
      return true;
    }
    if (!isInitialTask && jobState == JobState.FINISHING) {
      return true;
    }
    return false;
  }

  /**
   * @return The readable job id
   */
  private synchronized String createJobId() {
    final Date date = new Date();
    return String.format("%1$tY%1$tm%1$td-%1$tH%1$tM%1$tS%1$tL%2$03d", date, RANDOM.nextInt(JOB_RUN_ID_SUFFIX_MAX));
  }

  /** {@inheritDoc} */
  @Override
  public void finishJob(final String jobName, final String jobRunId) throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("finish called for job '" + jobName + "', run '" + jobRunId + "'");
    }
    final JobState state = _runStorage.getJobState(jobName, jobRunId);
    if (state == JobState.RUNNING) {
      // notify listeners...
      for (final JobListener jobListener : _jobListeners) {
        try {
          jobListener.processJobEvent(new PrepareToFinishEvent(jobName, jobRunId));
        } catch (final Throwable t) {
          _log.error("JobListener failed to process PrepareToFinishEvent for job '" + jobName + "', rob run id '"
            + jobRunId + "'.", t);
        }
      }
      if (_runStorage.finishJobRun(jobName, jobRunId)) {
        removeJobTriggers(ensureJobRun(jobName, jobRunId));
        checkAndHandleJobRunCompleted(jobName, jobRunId);
      }
    } else if (state == null) {
      // not found -> check if we find jobRunId in objectstore persistence -> Job was finished, failed or canceled
      // before
      checkNonFoundJobRunWasPersisted(jobName, jobRunId);
    } else {
      // found in RunStorage but not RUNNING
      throw new IllegalJobStateException("Job run '" + jobRunId + "' of job '" + jobName
        + "' couldn't be finished, because it's in state '" + state + "'.");
    }
  }

  /** {@inheritDoc} */
  @Override
  public void cancelJob(final String jobName, final String jobRunId) throws JobManagerException {
    final int maxTries = 3;
    int tries = 0;
    boolean success = false;
    JobState state = null;
    // loop to handle meantime job run state changes
    // (loop can only happen for state changes PREPRARING -> RUNNING -> FINISHING)
    do {
      tries++;
      state = _runStorage.getJobState(jobName, jobRunId);
      if (state == null) {
        // not found -> check if we find jobRunId in objectstore persistence -> Job was finished, failed or canceled
        // before
        checkNonFoundJobRunWasPersisted(jobName, jobRunId);
      } else if (state == JobState.CANCELLING || state == JobState.CANCELED) {
        throw new IllegalJobStateException("Job run '" + jobRunId + "' of job '" + jobName
          + "' couldn't be canceled, because someone else canceled it before.");
      } else if (!(state == JobState.RUNNING || state == JobState.PREPARING || state == JobState.FINISHING)) {
        throw new IllegalJobStateException("Job run '" + jobRunId + "' of job '" + jobName
          + "' couldn't be canceled, because it's in state '" + state + "'.");
      }
      // set job run state on CANCELLING (only successful if former state is still valid)
      success = _runStorage.setJobState(jobName, jobRunId, state, JobState.CANCELLING);
    } while (!success && tries <= maxTries);

    try {
      // remove tasks in taskmanager
      final AnyMap taskFilter = DataFactory.DEFAULT.createAnyMap();
      taskFilter.put(Task.PROPERTY_JOB_NAME, jobName);
      taskFilter.put(Task.PROPERTY_JOB_RUN_ID, jobRunId);
      _taskManager.removeTasks(taskFilter);

      // cancel workflow runs and tasks in run storage
      final List<String> canceledWorkflowRunIds = _runStorage.cancelJobRun(jobName, jobRunId);
      // remove transient data for canceled workflow runs
      for (final String canceledWorkflowRun : canceledWorkflowRunIds) {
        deleteTransientBulks(jobName, jobRunId, canceledWorkflowRun);
      }

      // persist job run data in objectstore, delete job run data in ZK, set job run state on CANCELED
      completeJobRun(jobName, jobRunId, JobState.CANCELED);

    } catch (final TaskmanagerException e) {
      throw new JobManagerException("Error while canceling job run '" + jobRunId + "' of job '" + jobName
        + "': TaskManager couldn't remove canceled tasks", e);
    } catch (final Exception e) {
      throw new JobManagerException("Error while canceling job run '" + jobRunId + "' of job '" + jobName + "'", e);
    }
  }

  /**
   * Check if given (not found) job run can be found in persistent storage to throw an appropriate exception.
   */
  private void checkNonFoundJobRunWasPersisted(final String jobName, final String jobRunId)
    throws JobManagerException {
    if (!_defPersistence.hasJob(jobName)) {
      throw new ConfigNotFoundException("Couldn't find job definition '" + jobName + "'.");
    } else if (!_defStorage.containsJobRun(jobName, jobRunId)) {
      throw new ConfigNotFoundException("Couldn't find job run '" + jobRunId + "' for job '" + jobName + "'.");
    } else {
      throw new IllegalJobStateException("Requested job run '" + jobRunId + "' of job '" + jobName
        + "' was already closed, can be found in job run history.", true);
    }
  }

  /**
   * register job to be triggered by its input buckets.
   * 
   * @param jobRun
   *          job run
   * @throws RunStorageException
   *           error
   */
  private void removeJobTriggers(final JobRun jobRun) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    for (final Bucket triggerBucket : jobRun.getTriggerBuckets()) {
      try {
        _runStorage.removeJobTrigger(triggerBucket.getBucketId(), jobName);
      } catch (final Exception ex) {
        _log.info("Could not remove job '" + jobName + "' from trigger bucket '" + triggerBucket.getBucketId()
          + "'. This is not critical.", ex);
      }
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @throws JobManagerException
   *           error because there was a jobmanager exception or a non-start-action-worker tried to get an initial task.
   */
  @Override
  public Task getInitialTask(final String workerName, final String jobName) throws JobManagerException {
    String jobRunId = null;
    String workflowRunId = null;
    Task task = null;
    jobRunId = _runStorage.getJobRunId(jobName);
    if (jobRunId == null || !checkJobStateForTaskCreation(jobName, jobRunId, true)) {
      throw new IllegalJobStateException("Job with name '" + jobName + "' is not running or is already finishing.");
    }
    final JobRun jobRun = ensureJobRun(jobName, jobRunId);
    try {
      workflowRunId = _runStorage.startWorkflowRun(jobName, jobRunId);
      task = jobRun.getInitialTask(workerName, workflowRunId);
      // initial tasks are not recoverable, because no other worker would ever retry them
      task.getProperties().put(Task.PROPERTY_RECOVERABLE, Boolean.toString(Boolean.FALSE));
      _runStorage.startTask(jobName, jobRunId, workflowRunId, workerName, task.getTaskId());
      _taskManager.addInProgressTask(task);
      return task;
    } catch (final Exception e) {
      cleanupFailedWorkflowRun(jobName, jobRunId, workflowRunId, workerName, task);
      throw new JobManagerException("Getting initial task failed for worker '" + workerName + "' for job '"
        + jobName + "' due to error.", e);
    }
  }

  /**
   * cleanup possibly created workflow run data after failure during start. Exceptions during clean up are logged.
   * 
   * @param jobName
   *          job name
   * @param jobRunId
   *          job run id
   * @param workflowRunId
   *          workflow run id. may be null, then nothing has to be done.
   * @param task
   *          inital task (maybe "null")
   */
  private void cleanupFailedWorkflowRun(final String jobName, final String jobRunId, final String workerName,
    final String workflowRunId, final Task task) {
    if (jobRunId != null && workflowRunId != null) {
      try {
        handleFatalError(jobName, jobRunId, workflowRunId, workerName, task, false);
      } catch (final Exception ex) {
        _log.error("Error during cleanup of failed new workflow run '" + jobRunId + "' for job '" + jobName + "'.",
          ex);
      }
    }
  }

  /**
   * Check if the job run data are still valid creates a new job run if not.
   * 
   * @param jobName
   *          The job name
   * @param jobRunId
   *          The job run id
   * @return The job run
   * @throws JobManagerException
   *           The job manager exception
   */
  private synchronized JobRun ensureJobRun(final String jobName, final String jobRunId) throws JobManagerException {
    JobRun jobRun = _jobRuns.get(jobName);
    if (jobRun == null || !jobRunId.equals(jobRun.getJobRunId())) {
      try {
        jobRun = new JobRun(jobRunId, jobName, _runStorage, _defPersistence, _taskGeneratorProvider);
      } catch (final Exception e) {
        throw new JobManagerException("Error during creation of job run: ", e);
      }
      _jobRuns.put(jobName, jobRun);
    }
    return jobRun;
  }

  /**
   * create follow up tasks.
   * 
   * @param currentTask
   *          the currently processed task
   * @param jobRun
   *          job run
   * @param workflowRunId
   *          The workflow run id
   * @return a list of follow up tasks
   * @throws JobManagerException
   *           error
   */
  private List<Task> createFollowUpTasks(final Task currentTask, final JobRun jobRun, final String workflowRunId)
    throws JobManagerException {

    // check if all non-optional output buckets have been created, and get all output bulks
    final MultiValueMap<String, BulkInfo> bucketNameToCreatedBulksMap =
      checkOutputBulksForCreation(jobRun, currentTask);

    // check for new tasks for another job
    final Collection<JobRun> triggeredJobs = new ArrayList<JobRun>();
    for (final Entry<String, List<BulkInfo>> bucketToBulksEntry : bucketNameToCreatedBulksMap.entrySet()) {
      final String bucketName = bucketToBulksEntry.getKey();
      if (jobRun.getBucket(bucketName).isPersistent()) {
        triggeredJobs.addAll(getJobsTriggeredByBucket(jobRun, bucketName));
      } else {
        storeTransientBulks(jobRun, workflowRunId, bucketToBulksEntry.getValue());
      }
    }

    final AnyMap taskParamsToCopy = getTaskParametersToCopy(currentTask);
    final List<Task> followUpTasks =
      jobRun.getFollowupTasks(workflowRunId, bucketNameToCreatedBulksMap, taskParamsToCopy);
    for (final JobRun triggeredJob : triggeredJobs) {
      try {
        final Collection<Task> triggeredTasks = triggeredJob.getTriggeredInitialTasks(bucketNameToCreatedBulksMap);
        if (!triggeredTasks.isEmpty()) {
          final String triggeredJobWorkflowRunId =
            _runStorage.startWorkflowRun(triggeredJob.getJobName(), triggeredJob.getJobRunId());
          for (final Task task : triggeredTasks) {
            task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, triggeredJobWorkflowRunId);
          }
          followUpTasks.addAll(triggeredTasks);
        }
      } catch (final Exception e) {
        _log.error(
          "Tried to create tasks for potentially triggered jobs, but got an error. Job '"
            + triggeredJob.getJobName() + "' will not be triggered.", e);
      }
    }
    return followUpTasks;
  }

  /** TODO remove this hack to copy task parameters in follow up tasks. */
  private AnyMap getTaskParametersToCopy(final Task currentTask) {
    final AnyMap taskParams = currentTask.getParameters();
    final AnyMap paramsToCopy = DataFactory.DEFAULT.createAnyMap();
    for (final Entry<String, Any> param : taskParams.entrySet()) {
      if (param.getKey().startsWith(JobManagerConstants.SYSTEM_PARAMETER_PREFIX)
        && !param.getKey().startsWith(JobManagerConstants.TEMP_PREFIX)) {
        paramsToCopy.put(param.getKey(), param.getValue());
      }
    }
    return paramsToCopy;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Task> finishTask(final Task currentTask) throws JobManagerException {
    final ResultDescription resultDescription = currentTask.getResultDescription();
    final String workflowRunId = currentTask.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);
    final String jobRunId = currentTask.getProperties().get(Task.PROPERTY_JOB_RUN_ID);
    final String jobName = currentTask.getProperties().get(Task.PROPERTY_JOB_NAME);
    final String workerName = getOriginalWorkerName(currentTask);
    if (!_runStorage.hasTask(jobName, workflowRunId, currentTask.getTaskId())) {
      throw new IllegalJobStateException("Task '" + currentTask.getTaskId() + "' for job '" + jobName
        + "' and run '" + jobRunId + "' is unknown, maybe already finished or workflow run was canceled.");
    }
    try {
      switch (resultDescription.getStatus()) {
        case SUCCESSFUL:
          if (_log.isTraceEnabled()) {
            _log.trace("Successfully handles task '" + currentTask.getTaskId() + "' of worker '" + workerName
              + "' in job run '" + jobRunId + "' of job '" + jobName + "'");
          }
          return handleSuccessfulTask(jobName, jobRunId, workflowRunId, workerName, currentTask,
            resultDescription.getCounters());
        case OBSOLETE:
          if (_log.isTraceEnabled()) {
            _log.trace("Obsolete task '" + currentTask.getTaskId() + "' of worker '" + workerName
              + "' in job run '" + jobRunId + "' of job '" + jobName + "'");
          }
          handleObsoleteTask(jobName, jobRunId, workflowRunId, workerName, currentTask);
          return Collections.emptyList();
        case RECOVERABLE_ERROR:
          // recoverable error - try to repeat the task processing
          _log.warn("A recoverable error '" + resultDescription.getErrorCode() + "'('"
            + resultDescription.getErrorMessage() + "') occurred in processing of task '" + currentTask.getTaskId()
            + "' for worker '" + workerName + "'");
          return handleRecoverableTaskError(jobName, jobRunId, workflowRunId, workerName, currentTask,
            resultDescription);
        default:
        case FATAL_ERROR:
          // fatal error - don't try to repeat the task processing
          _log.error("A fatal error '" + resultDescription.getErrorCode() + "'('"
            + resultDescription.getErrorMessage() + "') occurred in processing of task " + currentTask.getTaskId()
            + " of worker " + workerName + ". Workflow run '" + workflowRunId
            + "' will be marked as failed, its tasks will be canceled.");
          handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
          return new ArrayList<Task>();
      }
    } catch (final JobManagerException jme) {
      if (resultDescription.getStatus() != TaskCompletionStatus.FATAL_ERROR && !jme.isRecoverable()) {
        try {
          handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
        } catch (final JobManagerException e) {
          _log.error("Exception while handling fatal error during nonrecoverable finishing exception.", e);
        }
      }
      throw jme;
    }
  }

  /** @return value of task property {@link Task#PROPERTY_ORIGINAL_WORKER}. */
  private String getOriginalWorkerName(final Task currentTask) {
    return currentTask.getProperties().get(Task.PROPERTY_ORIGINAL_WORKER);
  }

  /**
   * @return the follow up tasks of the task that is successfully finished.
   */
  protected List<Task> handleSuccessfulTask(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final Map<String, Number> workerCounter) throws JobManagerException {

    final List<Task> followUpTasks = new ArrayList<Task>();
    try {
      final JobRun jobRun = ensureJobRun(jobName, jobRunId);
      final WorkerDefinition worker = jobRun.getWorkerDefinition(workerName);

      if (!checkJobStateForTaskCreation(jobName, jobRunId, false)) {
        throw new IllegalJobStateException("Could not finish Task. Job with name'" + jobName
          + "' is not running or is already completed.");
      }
      // job is RUNNING or FINISHING: continue.
      // create follow up tasks
      followUpTasks.addAll(createFollowUpTasks(currentTask, jobRun, workflowRunId));
      for (final Task newTask : followUpTasks) {
        final String newTaskJobName = newTask.getProperties().get(Task.PROPERTY_JOB_NAME);
        final String newTaskJobRunId = newTask.getProperties().get(Task.PROPERTY_JOB_RUN_ID);
        final String newTaskWfRunId = newTask.getProperties().get(Task.PROPERTY_WORKFLOW_RUN_ID);

        // TODO find a clean solution for handling the following statements in a more "atomic" way.
        // If RunStorage task entry creation is done first (as is), failures can happen if process/node is crashing
        // right between the two statements: Task entry in RunStorage will exist there until job run is canceled.
        // If instead Taskmanager task creation would be done first, this may cause bug #3566.
        _runStorage.startTask(newTaskJobName, newTaskJobRunId, newTaskWfRunId, newTask.getWorkerName(),
          newTask.getTaskId());
        _taskManager.addTask(newTask);
      }
      notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.SUCCESSFUL);
      // delete AFTER the new tasks have been added.
      // Otherwise some other concurrent invocation might think there are no tasks left and finishes the workflow run!
      _runStorage.finishTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(), workerCounter,
        currentTask.getProperties());
      // check if workflow run is finished
      if (checkAndHandleWorkflowRunCompleted(jobName, jobRunId, workflowRunId)) {
        // check if job run is finished
        checkAndHandleJobRunCompleted(jobName, jobRunId);
      }
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling successful task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
    return followUpTasks;
  }

  /** invoke finishTask method of task generator that belongs to the given worker. */
  private void notifyTaskGenerator(final WorkerDefinition worker, final Task task, final TaskCompletionStatus status)
    throws TaskGeneratorException {
    final TaskGenerator taskGenerator = getTaskGenerator(worker);
    taskGenerator.finishTask(task, status);
  }

  /**
   * finish obsolete task - don't create follow up tasks.
   */
  protected void handleObsoleteTask(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask) throws JobManagerException {
    try {
      final JobRun jobRun = ensureJobRun(jobName, jobRunId);
      final WorkerDefinition worker = jobRun.getWorkerDefinition(workerName);

      if (!checkJobStateForTaskCreation(jobName, jobRunId, false)) {
        throw new IllegalJobStateException("Could not finish obsolete task '" + currentTask.getTaskId()
          + "'. Job with name'" + jobName + "' is not running or is already completed.");
      }

      notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.OBSOLETE);
      _runStorage.obsoleteTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(),
        currentTask.getProperties());

      // check if workflow run is finished
      if (checkAndHandleWorkflowRunCompleted(jobName, jobRunId, workflowRunId)) {
        // check if job run is finished
        checkAndHandleJobRunCompleted(jobName, jobRunId);
      }
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling obsolete task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
  }

  /**
   * remove object from store, do not throw exception if that fails.
   */
  private void removeStoreObjectQuietly(final String storeName, final String objectId) {
    try {
      _objectStore.removeObject(storeName, objectId);
    } catch (final Exception ex) {
      _log.warn("Error removing obsolete input object '" + objectId + "'", ex);
    }
  }

  /**
   * @return follow up tasks. This could be the current task in case of a retry. If the number of retries exceeded no
   *         follow up tasks are generated. In case of mode AUTOCOMMIT task is handled as succeeded.
   */
  protected List<Task> handleRecoverableTaskError(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final ResultDescription resultDescription) throws JobManagerException {

    final List<Task> followUpTasks = new ArrayList<Task>();
    try {
      final WorkerDefinition worker;
      try {
        worker = getWorker(workerName);
      } catch (final ConfigNotFoundException e) {
        throw new JobManagerException("Worker '" + workerName + "' not found while handling recoverable task '"
          + currentTask.getTaskId() + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
      }
      if (worker.getModes().contains(Mode.AUTOCOMMIT)) {
        // autoCommit=true -> commit task's output and handle task as "successful" processed
        handleRecoverableTaskWithAutocommit(jobName, jobRunId, workflowRunId, workerName, currentTask,
          resultDescription, followUpTasks);
      } else {
        // autoCommit=false -> repeat current task if it's recoverable
        if (currentTask.getProperties().containsKey(Task.PROPERTY_RECOVERABLE)
          && !Boolean.valueOf(currentTask.getProperties().get(Task.PROPERTY_RECOVERABLE))) {
          _log.warn("Could not retry failed task '" + currentTask.getTaskId() + "' cause it's not recoverable.");
          handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, false);
        } else {
          final int numberOfRetries =
            _runStorage.getTaskRetries(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId()) + 1;
          // if we can retry it again, do so. If not, task is failed.
          if (numberOfRetries <= _clusterConfigService.getMaxRetries()) {
            _runStorage.retriedTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(),
              !TaskManager.TASKERROR_TIME_TO_LIVE.equals(resultDescription.getErrorCode()),
              currentTask.getProperties());
            final Task retryTask = currentTask.createRetryTask(IdGenerator.createTaskId());
            _taskManager.addTask(retryTask);
            _runStorage.startTask(jobName, jobRunId, workflowRunId, retryTask.getWorkerName(),
              retryTask.getTaskId(), numberOfRetries);
            followUpTasks.add(retryTask);
          } else {
            _log.error("Could not retry failed recoverable task '" + currentTask.getTaskId()
              + "' , because maximum number of retries was reached.");
            handleFatalError(jobName, jobRunId, workflowRunId, workerName, currentTask, true);
          }
        }
      }
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling recoverable task '" + currentTask.getTaskId()
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
    return followUpTasks;
  }

  /**
   * recoverable error on an AUTOCOMMIT worker's task -> task is handled as succeeded.
   */
  protected void handleRecoverableTaskWithAutocommit(final String jobName, final String jobRunId,
    final String workflowRunId, final String workerName, final Task currentTask,
    final ResultDescription resultDescription, final List<Task> followUpTasks) throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("Task '" + currentTask.getTaskId() + "' of autocommit worker '" + workerName + "' in job '"
        + jobName + "' failed with an recoverable error: finishing it as successful.");
    }
    for (final Entry<String, List<BulkInfo>> entry : currentTask.getOutputBulks().entrySet()) {
      for (final BulkInfo bulkInfo : entry.getValue()) {
        final String storeName = bulkInfo.getStoreName();
        final String objectId = bulkInfo.getObjectName();
        // finish already created objects so the failing worker can no longer append data to it, if it's still alive.
        // and the data is closed cleanly.
        try {
          if (ObjectStoreRetryUtil.retryExistsObject(getObjectStore(), storeName, objectId)) {
            getObjectStore().finishObject(storeName, objectId);
          }
        } catch (final ServiceUnavailableException ex) {
          _log.warn("Error finishing object '" + objectId + "' in store '" + storeName + "', retrying.", ex);
          throw new JobManagerException("Could not finish object '" + objectId + "' in store '" + storeName + "'.",
            ex);
        } catch (final Exception ex) {
          _log.warn("Error finishing object '" + objectId + "' in store '" + storeName
            + "', ignoring and continuing.", ex);
        }
      }
    }
    followUpTasks.addAll(handleSuccessfulTask(jobName, jobRunId, workflowRunId, workerName, currentTask,
      resultDescription.getCounters()));
  }

  /** 
   */
  protected void handleFatalError(final String jobName, final String jobRunId, final String workflowRunId,
    final String workerName, final Task currentTask, final boolean failedAfterRetry) throws JobManagerException {
    try {
      try {
        final WorkerDefinition worker = getWorker(workerName);
        notifyTaskGenerator(worker, currentTask, TaskCompletionStatus.FATAL_ERROR);
      } finally {
        // delete current task first for statistical reasons
        _runStorage.failedTask(jobName, jobRunId, workflowRunId, workerName, currentTask.getTaskId(),
          failedAfterRetry, currentTask.getProperties());

        // delete whole workflow run
        if (_runStorage.failedWorkflowRun(jobName, jobRunId, workflowRunId)) {
          deleteTransientBulks(jobName, jobRunId, workflowRunId);
          _runStorage.deleteWorkflowRun(jobName, jobRunId, workflowRunId);
          checkAndHandleJobRunCompleted(jobName, jobRunId);
        }
      }
      // TODO we have to do some smarter handling here:
      // - what to do with keep-alives from tasks of given workflow run?
      // - what to do with successful finished tasks of given workflow run?
    } catch (final Exception e) {
      throw new JobManagerException("Error while handling failed task in workflow run '" + workflowRunId
        + "' in job run '" + jobRunId + "' of job '" + jobName + "'.", e);
    }
  }

  /**
   * Stores transient bulks in RunStorage.
   * 
   * @param jobRun
   *          job run
   * @param workflowRunId
   *          the id of the workflow run.
   * @param transientBulks
   *          the bulks to store in the runStorage
   * @throws RunStorageException
   *           exception while storing the information in RunStorage
   */
  private void storeTransientBulks(final JobRun jobRun, final String workflowRunId,
    final List<BulkInfo> transientBulks) throws RunStorageException {
    final String jobName = jobRun.getJobName();
    final String jobRunId = jobRun.getJobRunId();
    for (final BulkInfo bulkInfo : transientBulks) {
      _runStorage.addTransientBulk(jobName, jobRunId, workflowRunId,
        bulkInfo.getStoreName() + "/" + bulkInfo.getObjectName());
    }
  }

  /**
   * Checks if a workflow run is finished and handles finishing the workflow run. I.e. deleting transient buckets,
   * merging statistics and finishing job if all runs are finished...
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param workflowRunId
   *          The id of the workflow run.
   * @return true if workflow run completed.
   * @throws JobManagerException
   *           error
   */
  private boolean checkAndHandleWorkflowRunCompleted(final String jobName, final String jobRunId,
    final String workflowRunId) throws JobManagerException {
    final boolean completed = isWorkflowRunCompleted(jobName, jobRunId, workflowRunId);
    if (completed) {
      deleteTransientBulks(jobName, jobRunId, workflowRunId);
      // RunStorage aggregates workflow run statistics to job run
      if (_runStorage.finishWorkflowRun(jobName, jobRunId, workflowRunId)) {
        // delete workflow run from RunStorage
        _runStorage.deleteWorkflowRun(jobName, jobRunId, workflowRunId);
      }
    }
    return completed;
  }

  /**
   * Checks if a job run is completed.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @throws JobManagerException
   *           error
   */
  private void checkAndHandleJobRunCompleted(final String jobName, final String jobRunId)
    throws JobManagerException {
    if (isJobRunCompleted(jobName, jobRunId)) {
      final boolean success = _runStorage.setJobState(jobName, jobRunId, JobState.FINISHING, JobState.COMPLETING);
      if (success) {
        completeJobRun(jobName, jobRunId, JobState.SUCCEEDED);
      } else {
        _log.warn("Couldn't change job state from " + JobState.FINISHING + " to " + JobState.COMPLETING
          + " for job run '" + jobRunId + "' of job '" + jobName + "'");
      }
    }
  }

  /**
   * set final state, read, persist and delete job run data.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param finalState
   *          final state: {@link JobState#SUCCEEDED} or {@link JobState#FAILED}.
   * @throws JobManagerException
   *           error completing the job
   */
  private void completeJobRun(final String jobName, final String jobRunId, JobState finalState)
    throws JobManagerException {
    if (_log.isInfoEnabled()) {
      _log.info("Completing job run '" + jobRunId + "' for job '" + jobName + "' with final state " + finalState);
    }
    final AnyMap jobRunData = DataFactory.DEFAULT.createAnyMap();
    // we have to make sure that no other job run was started in parallel
    final String currentJobRunId = _runStorage.getJobRunId(jobName);
    if (jobRunId.equals(currentJobRunId)) {
      // get job run data (with details) from run storage
      jobRunData.putAll(_runStorage.getJobRunData(jobName, true));
      if (finalState == JobState.SUCCEEDED) {
        // if workflow runs where started, but none was successful, the job actually failed.
        final AnyMap workflowRunData = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
        final int startedWorkflowRuns =
          Integer
            .parseInt(workflowRunData.getStringValue(JobManagerConstants.DATA_JOB_NO_OF_STARTED_WORKFLOW_RUNS));
        final int successfulWorkflowRuns =
          Integer.parseInt(workflowRunData
            .getStringValue(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS));
        if (startedWorkflowRuns > 0 && successfulWorkflowRuns == 0) {
          finalState = JobState.FAILED;
        }
      }
    }
    // store job run data in persistent storage
    jobRunData.put(JobManagerConstants.DATA_JOB_STATE, finalState.name());
    jobRunData.put(JobManagerConstants.DATA_JOB_RUN_END_TIME, _runStorage.getCurrentTimestamp());
    _defStorage.storeJobRun(jobName, jobRunId, sortedJobRunData(jobRunData));

    // delete job run from run storage if no other job run was started
    if (jobRunId.equals(currentJobRunId)) {
      _runStorage.deleteJobRun(jobName, jobRunId);
      _jobRuns.remove(jobName);
    }
  }

  /** re-order job run data: simple values sorted alphabetically first, complex value as ordered in input last. */
  private AnyMap sortedJobRunData(final AnyMap jobRunData) {
    final Map<String, Any> simpleValues = new TreeMap<String, Any>();
    final AnyMap complexValues = jobRunData.getFactory().createAnyMap();
    for (final Entry<String, Any> entry : jobRunData.entrySet()) {
      final String key = entry.getKey();
      final Any value = entry.getValue();
      if (value.isValue()) {
        simpleValues.put(key, value);
      } else {
        complexValues.put(key, value);
      }
    }
    final AnyMap sortedData = jobRunData.getFactory().createAnyMap();
    sortedData.putAll(simpleValues);
    sortedData.putAll(complexValues);
    return sortedData;
  }

  /**
   * Checks if non-optional output bulks have been created. If a non-optional output bulk has not been created an
   * exception is thrown. If an optional output bulk has not been created, it will be removed from the map.
   * 
   * A bulk has not been created, if it is empty or does not even exist. An empty optional bulk will be deleted.
   * 
   * @param jobRun
   *          The JobRun instance for the current job run.
   * @param task
   *          The Task that should be checked.
   * @return output bulks without the not created optional buckets. (key: bucket name, value: BulkInfo)
   * @throws JobManagerException
   *           An error occurred while checking the objects or a non-optional bulk has not been created.
   */
  private MultiValueMap<String, BulkInfo> checkOutputBulksForCreation(final JobRun jobRun, final Task task)
    throws JobManagerException {
    final WorkerDefinition workerDef = jobRun.getWorkerDefinition(getOriginalWorkerName(task));
    final MultiValueMap<String, BulkInfo> outputBulkMap = new MultiValueMap<String, BulkInfo>();
    final Iterator<Entry<String, List<BulkInfo>>> entryIter = task.getOutputBulks().entrySet().iterator();
    while (entryIter.hasNext()) {
      final Entry<String, List<BulkInfo>> currentEntry = entryIter.next();
      final String currentSlotName = currentEntry.getKey();
      final boolean isOptional = workerDef.getOutput(currentSlotName).getModes().contains(OutputMode.OPTIONAL);
      for (final BulkInfo bulkInfo : currentEntry.getValue()) {
        final String storeName = bulkInfo.getStoreName();
        final String objectName = bulkInfo.getObjectName();
        try {
          if (!ObjectStoreRetryUtil.retryExistsObject(_objectStore, storeName, objectName)) {
            if (!isOptional) {
              throw new JobManagerException("Output bulk '" + objectName + "' in bucket '"
                + bulkInfo.getBucketName() + "' of worker '" + workerDef.getName() + "' (job '"
                + jobRun.getJobName() + "') has not been created but is not optional.");
            }
          } else {
            // all's ok, add it to the output map.
            outputBulkMap.add(bulkInfo.getBucketName(), bulkInfo);
          }
        } catch (final Exception e) {
          throw new JobManagerException(e.getClass().getName() + " occurred while checking output buckets.", e);
        }
      }
    }
    return outputBulkMap;
  }

  /**
   * Returns a list of jobs (other than triggeringJob) where a startAction exists that is triggered by the persistent
   * bucket referenced in bucketStorageInfo.
   * 
   * @param triggeringJobRun
   *          the triggering job run
   * @param bucketName
   *          The BucketStorageInfo.
   * @return A list of job runs (other than triggeringJob) where a startAction exists that is triggered by the
   *         persistent bucket referenced in bucketStorageInfo.
   * @throws JobManagerException
   *           Error occurred while accessing JobDefinitions instances.
   */
  private Collection<JobRun> getJobsTriggeredByBucket(final JobRun triggeringJobRun, final String bucketName)
    throws JobManagerException {
    final Bucket triggeringBucket = triggeringJobRun.getBucket(bucketName);
    final Collection<String> jobNames = _runStorage.getTriggeredJobs(triggeringBucket.getBucketId());
    // ignore triggering job
    jobNames.remove(triggeringJobRun.getJobName());
    final Collection<JobRun> triggeredJobs = new ArrayList<JobRun>();
    for (final String jobName : jobNames) {
      try {
        final JobRunInfo runInfo = _runStorage.getJobRunInfo(jobName);
        if (runInfo != null && runInfo.getState() == JobState.RUNNING) {
          final String jobRunId = runInfo.getId();
          final JobRun jobRun = ensureJobRun(jobName, jobRunId);
          if (jobRun.isTriggeredBy(triggeringBucket)) {
            triggeredJobs.add(jobRun);
          }
        }
      } catch (final Exception e) {
        _log.warn("Tried to check potentially triggered jobs, but got an error. Job '" + jobName
          + "' will not be triggered.", e);
      }
    }
    return triggeredJobs;
  }

  /**
   * Checks if the workflow run is finished, i.e. no tasks for this workflow run are open.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @param workflowRunId
   *          The id of the workflow run.
   * @return 'true' if workflow run is finished
   * @throws RunStorageException
   *           exception while accessing RunStorage.
   */
  private boolean isWorkflowRunCompleted(final String jobName, final String jobRunId, final String workflowRunId)
    throws RunStorageException {
    return !_runStorage.hasTasks(jobName, jobRunId, workflowRunId);
  }

  /**
   * Checks if the job run is finished, i.e. no workflow runs for this job are open.
   * 
   * @param jobName
   *          The name of the job.
   * @param jobRunId
   *          The id of the job run.
   * @return 'true' if job run is finished
   * @throws RunStorageException
   *           exception while accessing RunStorage.
   */
  private boolean isJobRunCompleted(final String jobName, final String jobRunId) throws RunStorageException {
    return _runStorage.getJobState(jobName) == JobState.FINISHING
      && !_runStorage.hasWorkflowRuns(jobName, jobRunId);
  }

  /**
   * Deletes transient instances of buckets from the collection of bucket instance ids. If the bucket instance is not
   * found in this instance of jobmanager it is lazily loaded before deleted.
   * 
   * @param jobName
   *          The job name
   * @param jobRunId
   *          The job run id
   * @param workflowRunId
   *          The workflow run id
   */
  private void deleteTransientBulks(final String jobName, final String jobRunId, final String workflowRunId) {
    try {
      final Collection<String> transientBulks = _runStorage.getTransientBulks(jobName, jobRunId, workflowRunId);
      for (final String transientBulkId : transientBulks) {
        // transientBulk id is of format: storeName/objectId
        final int indexOfSlash = transientBulkId.indexOf('/');
        if (indexOfSlash <= 0 || transientBulkId.length() <= indexOfSlash) {
          _log.warn("Stored transient bulk id '" + transientBulkId + "' is invalid, skipping");
        } else {
          final String storeName = transientBulkId.substring(0, indexOfSlash);
          final String objectId = transientBulkId.substring(indexOfSlash + 1);
          try {
            removeStoreObjectQuietly(storeName, objectId);
            if (_log.isDebugEnabled()) {
              _log.debug("Deleted transient bulk '" + objectId + "' from store'" + storeName + "'.");
            }
          } catch (final Exception e) {
            _log.warn("Error while deleting transient bulk object '" + objectId + "' in store '" + storeName
              + "'. It could not be deleted.", e);
          }
        }
      }
    } catch (final JobManagerException ex) {
      _log.warn("Failed to retrieve stored transient bulk Ids, obsolete data remains in stores possibly.", ex);
    }
  }

  // ----- OSGI stuff -----

  /**
   * @param defStorage
   *          DefinitionStorage reference.
   */
  public void setDefinitionStorage(final DefinitionStorage defStorage) {
    _defStorage = defStorage;
  }

  /**
   * @param defStorage
   *          DefinitionStorage reference.
   */
  public void unsetDefinitionStorage(final DefinitionStorage defStorage) {
    if (_defStorage == defStorage) {
      _defStorage = null;
    }
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void setRunStorage(final RunStorage runStorage) {
    _runStorage = runStorage;
  }

  /**
   * @param runStorage
   *          RunStorage reference.
   */
  public void unsetRunStorage(final RunStorage runStorage) {
    if (_runStorage == runStorage) {
      _runStorage = null;
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStore = objectStore;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStore == objectStore) {
      _objectStore = null;
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param taskGeneratorProvider
   *          TaskGeneratorProvider reference.
   */
  public void setTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    _taskGeneratorProvider = taskGeneratorProvider;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param taskGeneratorProvider
   *          TaskGeneratorProvider reference.
   */
  public void unsetTaskGeneratorProvider(final TaskGeneratorProvider taskGeneratorProvider) {
    if (_taskGeneratorProvider == taskGeneratorProvider) {
      _taskGeneratorProvider = null;
    }
  }

  /**
   * method for DS to set a service reference.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * method for DS to unset a service reference.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

  /**
   * method for DS to set a service reference.
   */
  public void setClusterConfigService(final ClusterConfigService clusterConfigService) {
    _clusterConfigService = clusterConfigService;
  }

  /**
   * method for DS to unset a service reference.
   */
  public void unsetClusterConfigService(final ClusterConfigService clusterConfigService) {
    if (_clusterConfigService == clusterConfigService) {
      _clusterConfigService = null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TaskGenerator getTaskGenerator(final WorkerDefinition worker) {
    TaskGenerator taskGenerator;
    if (worker.getTaskGenerator() != null && !worker.getTaskGenerator().equals("")) {
      taskGenerator = _taskGeneratorProvider.getTaskGenerator(worker.getTaskGenerator());
    } else {
      taskGenerator = _taskGeneratorProvider.getDefaultTaskGenerator();
    }
    return taskGenerator;
  }

  /**
   * Add a RequestHandler..
   * 
   * @param listener
   *          the new RequestHandler
   */
  public void addJobListener(final JobListener listener) {
    _jobListeners.add(listener);
  }

  /**
   * Remove the given RequestHandler.
   * 
   * @param listener
   *          the RequestHandler
   */
  public void removeJobListener(final JobListener listener) {
    if (_jobListeners.contains(listener)) {
      _jobListeners.remove(listener);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DefinitionPersistence getDefinitionPersistence() {
    return _defPersistence;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> getJobRunsUsingStore(String storeName) {
    final Map<String, String> jobRuns = new HashMap<String, String>();
    try {
      final Collection<String> currentJobs = _runStorage.getCurrentJobs();
      final Iterator<String> currentJobsIterator = currentJobs.iterator();
      while (currentJobsIterator.hasNext()) {
        final String jobName = currentJobsIterator.next();
        final String jobRunId = _runStorage.getJobRunId(jobName);
        JobRun jobRun = null;
        if (jobRunId != null) {
          try {
            jobRun = ensureJobRun(jobName, jobRunId);
          } catch (JobManagerException e) {
            ;// do nothing, try next job name.
          }
          if (jobRun != null) {
            final Collection<Bucket> buckets = jobRun.getBuckets();
            final Iterator<Bucket> bucketIterator = buckets.iterator();
            while (bucketIterator.hasNext()) {
              final Bucket bucket = (Bucket) bucketIterator.next();
              if (bucket.getStoreName().equals(storeName)) {
                jobRuns.put(jobName, jobRunId);
              }
            }
          }
        }
      }
    } catch (RunStorageException e) {
      ;// nothing to do, empty Map will be removed.
    }
    return jobRuns;
  }

  @Override
  // "synchronized" to synchronize it with ensureJobRun()
  public synchronized void notifiyAboutJobRunCompletion(final String jobName) {
    final JobRun jobRun = _jobRuns.get(jobName);
    if (jobRun != null) {
      final String jobRunId = jobRun.getJobRunId();
      try {
        // check if job run is completed
        if (_defStorage.containsJobRun(jobName, jobRunId)) {
          _jobRuns.remove(jobName);
        }
      } catch (final PersistenceException e) {
        // ignore errors
        _log.warn("Error while checking if job run is finished.", e);
      }
    }
  }

}
