/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.taskgenerator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.Bucket;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;

/**
 * TaskGenerator for two input buckets that generates tasks for each combination of:
 * 
 * [changed input bulk of first bucket X (all) object(s) from second input bucket]
 * 
 * If the second input bucket is empty, no tasks will be generated.
 */
public class CombineInputWithAllTaskGenerator extends TaskGeneratorBase {

  /** Prefix for partner parameter. */
  public static final String PARAMETER_PARTNER_PREFIX = "_partner.";

  /** parameter for triggered by. */
  public static final String PARAMETER_TRIGGERED_BY = "_triggeredBy";

  /**
   * Generate task by combining changed bulk with each bulk from the other bucket. For each changed input bulk:
   * <ul>
   * <li>first input bucket is bucket of changed bulk
   * <li>second input bucket is other bucket from allInput
   * <li>read all objects from objectstore for second input bucket
   * <li>create a task for each combination of [changed bulk X object from second store]
   * <li>but no duplicates
   * </ul>
   * 
   * {@inheritDoc}
   */
  @Override
  public List<Task> createTasks(final Map<String, List<BulkInfo>> changedInput,
    final Map<String, Bucket> inputBuckets, final Map<String, Bucket> outputBuckets,
    final AnyMap parameters, final String workerName) throws TaskGeneratorException {

    checkBucketCounts(inputBuckets, 2, outputBuckets, -1);
    final List<Task> tasks = new ArrayList<Task>();
    for (final Entry<String, List<BulkInfo>> changedSlot : changedInput.entrySet()) {
      final String changedSlotName = changedSlot.getKey();
      for (final Entry<String, Bucket> otherSlot : inputBuckets.entrySet()) {
        final String otherSlotName = otherSlot.getKey();
        if (!changedSlotName.equals(otherSlotName)) {
          final String partnerBulkName = findPartnerBulk(changedInput, otherSlotName);
          final Bucket partnerBucket = otherSlot.getValue();
          final Collection<StoreObject> objects = getObjectsFromBucket(partnerBucket);
          final List<BulkInfo> changedBulks = changedSlot.getValue();
          for (final BulkInfo changedBulk : changedBulks) {
            final List<Task> newTasks =
              createTasksFromObjects(objects, otherSlotName, partnerBucket, changedSlotName, changedBulk,
                partnerBulkName, workerName);
            addUniqueTasks(tasks, newTasks, parameters, outputBuckets);
          }
        }
      }
    }

    return tasks;
  }

  /** @return object ID of first object from given slot, if changedInput contains any. Null else. */
  private String findPartnerBulk(final Map<String, List<BulkInfo>> changedInput, final String slotName) {
    final List<BulkInfo> bulks = changedInput.get(slotName);
    if (bulks != null && !bulks.isEmpty()) {
      return bulks.get(0).getObjectName();
    }
    return null;
  }

  /** @return all objects from the bucket with a size greater than 0. */
  private Collection<StoreObject> getObjectsFromBucket(final Bucket bucket) throws TaskGeneratorException {
    try {
      return ObjectStoreRetryUtil.retryGetStoreObjectInfos(_objectStore, bucket.getStoreName(),
        bucket.getDataObjectNamePrefix());
    } catch (final Exception e) {
      throw new TaskGeneratorException(e.getClass().getSimpleName() + " during task creation: " + e.getMessage(), e);
    }
  }

  /** @return list of tasks for the given objects and the changed bulk. */
  private List<Task> createTasksFromObjects(final Collection<StoreObject> objects, final String partnerSlotName,
    final Bucket partnerBucket, final String changedSlotName, final BulkInfo changedBulk,
    final String partnerBulkName, final String workerName) {

    final List<Task> tasks = new ArrayList<Task>(objects.size());

    final String partnerBucketName = partnerBucket.getBucketDefinition().getName();
    final String partnerStoreName = partnerBucket.getStoreName();
    final String changedBulkName = changedBulk.getObjectName();

    for (final StoreObject objectInfo : objects) {
      final String currentObjectName = objectInfo.getId();
      final BulkInfo currentObjectBulk = new BulkInfo(partnerBucketName, partnerStoreName, currentObjectName);
      final AnyMap parameters =
        getTriggerInfoParameters(changedSlotName, changedBulkName, partnerSlotName, partnerBulkName,
          currentObjectName);
      final Task task = createTask(workerName, parameters);
      task.getInputBulks().put(changedSlotName, Arrays.asList(changedBulk));
      task.getInputBulks().put(partnerSlotName, Arrays.asList(currentObjectBulk));
      tasks.add(task);
    }
    return tasks;
  }

  /** @return task parameters to describe what triggered this task. */
  private AnyMap getTriggerInfoParameters(final String changedBucketName, final String changedBulkName,
    final String partnerBucketName, final String partnerBulkName, final String currentObjectName) {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    if (currentObjectName.equals(partnerBulkName)) {
      parameters.put(PARAMETER_PARTNER_PREFIX + changedBucketName, changedBulkName);
      parameters.put(PARAMETER_PARTNER_PREFIX + partnerBucketName, currentObjectName);
    } else {
      parameters.put(PARAMETER_TRIGGERED_BY, changedBucketName);
      if (partnerBulkName != null) {
        parameters.put(PARAMETER_PARTNER_PREFIX + partnerBucketName, partnerBulkName);
      }
    }
    return parameters;
  }

  /**
   * add each task with a new combination of input bulks to the list and add all parameters and necessary output bulks.
   */
  private void addUniqueTasks(final List<Task> tasks, final List<Task> newTasks, final AnyMap parameters,
    final Map<String, Bucket> outputBuckets) {
    for (final Task newTask : newTasks) {
      if (!isTaskDuplicate(tasks, newTask)) {
        newTask.getParameters().putAll(parameters);
        addOutputBulks(newTask, outputBuckets, parameters);
        tasks.add(newTask);
      }
    }
  }

  /** @return true if list of tasks contains a task with the same combination of input bulks. */
  private boolean isTaskDuplicate(final List<Task> tasks, final Task newTask) {
    for (final Task task : tasks) {
      if (newTask.getInputBulks().equals(task.getInputBulks())) {
        return true;
      }
    }
    return false;
  }

}
